<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet version="2.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
   xmlns:fo="http://www.w3.org/1999/XSL/Format"
   xpath-default-namespace="http://www.ieee.org/11073/nomenclature"
   xmlns:xs="http://www.w3.org/2001/XMLSchema">
	<!--
	tableOutput.8g.xsl   2012-09-26
	Display/print IEEE 11073 nomenclature partition of base terms with discriminator sets as html tables [8f].
	Includes Enum_Value with group expansion [8f].
	Version '8g' adds minor enhancements relative to '8f' to prepare for publication as IEEE Std 11073-10102.
	  -	Include additional "bi-uni", "pos" and "neg-ref" when outputing ECG lead discriminator table [8g]
	  - If REFID = MDC_ECG_LEAD, print as MDC_ECG_LEAD_CONFIG as the "unspecified lead identifier" [8g]
	  - Now supports automatic generation of "bi-uni", "pos" and "neg-ref" columns [8g]  2012-09-26
	Paul Schluter, GE Healthcare
	-->
	<!-- -->
	<xsl:output method="html"/>
	<!-- Filename for discriminators -->
	<xsl:variable name="discrimFileName" select="'ECG_ANN_discrims.8g.xml'"/>
	<!-- root template -->
	<xsl:template match="/">
		<html>
			<head>
				<title>
					<xsl:value-of select="//partitionDescription/title"/>
				</title>
			</head>
			<body>
				<!-- Output table of discriminators, if any are used -->
				<xsl:if test="boolean(//discriminatorRef)">
					<table border="1" cellpadding="2" cellspacing="0">
						<caption align="top">
							<b><xsl:value-of select="'Discriminators (most significant listed first)'"/></b>
						</caption>
						<tbody>
							<!-- Output sub-table(s) for each discriminator -->
							<xsl:apply-templates select="//discriminatorRef"/>
						</tbody>
					</table>
					<!-- Output blank line -->
					<xsl:text>		
					</xsl:text>
				</xsl:if>
				<!-- -->
				<!-- Scan the partition for columns that are included only if they are present -->
				<!-- ENABLE with select="boolean(string-join(distinct-values(//SysName),'|'))" DISABLE with select="boolean(false)" -->
				<xsl:variable name="SysNameFlag" select="boolean(string-join(distinct-values(//SysName),'|'))"/>	
				<xsl:variable name="MnemonicFlag" select="boolean(string-join(distinct-values(//Mnemonic),'|'))"/>
				<xsl:variable name="Enum_ValuesFlag" select="boolean(string-join(distinct-values(//Enum_Values),'|'))"/>
				<!-- -->
				<!-- Output table for all the partition terms -->
				<table border="1" cellpadding="2" cellspacing="0">
					<caption align="top">
						<b><xsl:value-of select="/partition/partitionDescription/title"/></b>
					</caption>
					<tbody>
						<tr align="left" bgcolor="D3D3D3">
							<xsl:if test="$SysNameFlag"><th>SysName</th></xsl:if>
							<th>REFID</th>
							<xsl:if test="$MnemonicFlag"><th>Mnemonic</th></xsl:if>
							<th>Description</th>
							<th align="right">PART</th>
							<th align="right">CODE10</th>
							<xsl:if test="$Enum_ValuesFlag"><th>Enum_REFID</th></xsl:if>
							<xsl:if test="$Enum_ValuesFlag"><th>Enum_TOKEN</th></xsl:if>
						</tr>
						<!-- Find each Term find a match -->
						<xsl:for-each select="/partition/Terms/term">
							<xsl:variable name="refid" select="REF_ID"/>
							<tr>
								<xsl:if test="$SysNameFlag">
									<td align="left"><xsl:value-of select="SysName"/></td>
								</xsl:if>
								<!-- If REFID = MDC_ECG_LEAD, print as MDC_ECG_LEAD_CONFIG as the "unspecified lead identifier" 2012-06-14 -->
								<td align="left"><xsl:value-of select="if (REFID='MDC_ECG_LEAD') then 'MDC_ECG_LEAD_CONFIG' else REFID"/></td>
								<xsl:if test="$MnemonicFlag">								
									<td align="left">
										<xsl:value-of select="Mnemonic"/>
										<!-- THIS IS NOT DRAWING A BORDER AROUND NULL ENTRIES  PSS 2009-05-02 -->
										<xsl:if test="empty(Mnemonic)"><xsl:value-of select="'&#xa0;'"/></xsl:if>
									</td>
								</xsl:if>	
								<td align="left"><xsl:apply-templates select="Description"/></td>
								<td align="right"><xsl:value-of select="(PART, /partition/partitionDescription/basePart, '?')[1]"/></td>
								<td align="right">
									<xsl:value-of select="CODE10"/>
									<xsl:if test="empty(CODE10)"><xsl:value-of select="'*'"/></xsl:if>
								</td>
								<xsl:if test="$Enum_ValuesFlag">								
									<!-- Output Enumerations from main Rosetta table (with _ENUM_GROUP expansion) -->
									<td align="left" valign="top">
										<xsl:variable name="enum1" select="distinct-values(tokenize(string-join(Enum_Values,' '), '\s+'))"/>
										<xsl:variable name="enum2" select="for $x in $enum1 return if ($x='') then () else $x"/>   <!-- remove null string due to empty element node(s) -->
										<xsl:variable name="enum3" select="distinct-values(for $x in $enum2 return if (starts-with($x,'_')) then (tokenize(string-join(/partition/Enums/term[_ENUM_GROUPS=$x]/(REFID),' '), '\s+')) else $x)"/>
										<xsl:variable name="enum4" select="for $x in $enum3 return if ($x='') then () else $x"/>
										<pre>
											<xsl:value-of select="$enum4" separator="&#x000D;"/>	<!-- use $enum2" for no _enum group expansion and $enum4 for _enum group expansion -->
											<xsl:if test="empty($enum4)"><xsl:value-of select="'&#xa0;'"/></xsl:if>   <!-- consider italicized period for greater HTML viewer compatibility -->
										</pre>
									</td>
									<td align="left" valign="top">
										<xsl:variable name="enum1" select="distinct-values(tokenize(string-join(Enum_Values,' '), '\s+'))"/>
										<xsl:variable name="enum2" select="for $x in $enum1 return if ($x='') then () else $x"/>   <!-- remove null string due to empty element node(s) -->
										<xsl:variable name="enum3" select="distinct-values(for $x in $enum2 return if (starts-with($x,'_')) then (tokenize(string-join(/partition/Enums/term[_ENUM_GROUPS=$x]/(TOKEN),' '), '\s+')) else $x)"/>
										<xsl:variable name="enum4" select="for $x in $enum3 return if ($x='') then () else $x"/>
										<pre>
											<xsl:value-of select="$enum4" separator="&#x000D;"/>	<!-- use $enum2" for no _enum group expansion and $enum4 for _enum group expansion -->
											<xsl:if test="empty($enum4)"><xsl:value-of select="'&#xa0;'"/></xsl:if>   <!-- consider italicized period for greater HTML viewer compatibility -->
										</pre>
									</td>
								</xsl:if>	
							</tr>
						</xsl:for-each>
					</tbody>
				</table>
				<!-- List footnote content at end of table -->
				<table border="0">
					<xsl:apply-templates select="//footnote" mode="footnote"/>
				</table>			
			</body>
		</html>
	</xsl:template>
	<!-- -->
	<!-- Process footnotes for selected Term child elements and output any remaining text -->
	<xsl:template match="Description">
		<xsl:value-of select="text()"/>
		<xsl:apply-templates select="footnote"/>
	</xsl:template>

	<!-- Process footnote element (is xsl:number the right number?) -->
	<xsl:template match="footnote">
		<xsl:variable name="inc"><xsl:number level="any" count="footnote"></xsl:number></xsl:variable>
		<xsl:text> </xsl:text><a href="#{$inc}" name="f{$inc}">[<xsl:value-of select="$inc"/>]</a>
	</xsl:template>
	<!-- List footnotes at end of table -->
	<xsl:template match="footnote" mode="footnote">
		<xsl:variable name="incr"><xsl:number level="any" count="footnote"/></xsl:variable>
		<tr>
			<td class="right" valign="top"><a name="{$incr}"><sup><xsl:value-of select="$incr"/></sup></a></td>
			<td class="left"><xsl:value-of select="text()"/><xsl:text> </xsl:text><a href="#f{$incr}">back</a></td>
		</tr>
	</xsl:template>

	<!-- Output sub-table for each <discriminatorRef> discriminator set -->
	<!-- If <discriminatorRef idref="MDC_ECG_discrim_LEADS">, output three additional columns for "bi-uni", "pos" and "neg-ref" -->	
	<xsl:template match="discriminatorRef">
		<xsl:variable name="ecgLeadExtras" select="if (@idref = 'MDC_ECG_discrim_LEADS') then 3 else 0"/>
		<!-- List the discriminator sub-table title -->
		<tr align="left" bgcolor="D3D3D3">
			<th colspan="3 + $ecgLeadExtras">
				<xsl:value-of select="document($discrimFileName)//discriminator[@id=current()/@idref]/dDescription[1]"/>
			</th>
		</tr>
		<!-- Column headings -->
		<tr align="left" bgcolor="E3E3E3">
			<th>dOffset</th>
			<th>dSuffix</th>
			<th>dDescription</th>
			<xsl:if test="$ecgLeadExtras ne 0">
				<th>bi-uni</th>
				<th>pos</th>
				<th>neg-ref</th>
			</xsl:if>
		</tr>
		<!-- List each discriminator term in set -->
		<xsl:for-each select="document($discrimFileName)//discriminator[@id=current()/@idref]/discrim">
			<tr align="left">
				<td><xsl:value-of select="dOffset"/></td>
				<td><xsl:value-of select="dSuffix"/>&#xa0;</td>
				<xsl:if test="$ecgLeadExtras eq 0">
					<td><xsl:value-of select="dDescription/text()"/>&#xa0;</td>
				</xsl:if>
				<!-- output all information for "original" ECG leads -->
				<xsl:if test="($ecgLeadExtras ne 0) and not (starts-with(dSuffix,'_d'))">
					<td><xsl:value-of select="dDescription/text()"/>&#xa0;</td>
					<td><xsl:value-of select="bi-uni"/>&#xa0;</td>
					<td><xsl:value-of select="pos"/>&#xa0;</td>
					<td><xsl:value-of select="neg-ref"/>&#xa0;</td>
				</xsl:if>
				<!-- output empty shaded cells for "derived" ECG leads -->
				<xsl:if test="($ecgLeadExtras ne 0) and (starts-with(dSuffix,'_d'))">
					<td bgcolor="D3D3D3">&#xa0;</td>
					<td bgcolor="D3D3D3">&#xa0;</td>			
					<td bgcolor="D3D3D3">&#xa0;</td>			
					<td bgcolor="D3D3D3">&#xa0;</td>							
				</xsl:if>
			</tr>
		</xsl:for-each>
	</xsl:template>
	<!-- Fin -->
</xsl:stylesheet>
