% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function analyse_subjective_patches()
%ANALYSE_SUBJECTIVE_PATCHES - Run analysis of subjective images
%   ANALYSE_SUBJECTIVE_PATCHES()
%
% References:
% 1. ISO 15739 WD#3 Oct 2010 Photography - Electronic still picture imaging - Noise measurements
% 2. Keelan, B.W., Jin, E.W. and Prokushkin, S.,
%   "Development of a perceptually calibrated object metric of noise",
%    Proc SPIE 7867, Image Quality and System Performance VIII. (2011)
% 3. Baxter, D.J., Murray, A.,
%   "Calibration and adaptation of ISO visual noise for I3A's Camera Phone Image Quality initiative",
%    Proc. SPIE 8293, Image Quality and System Performance IX, (2012)
% 4. Baxter, D., Phillips, J., Denman, H.,
%    "The Subjective Importance of the noise spectral content",
%     Proc. SPIE 9016, Image Quality and System Performance XI, 901603 (January 7, 2014); 

    image_root = '../images/cpiq_subjective/ppm';
    op_root    = '../images/cpiq_subjective';
    
    % Visual Noise Stats
    vn_types = {'iso', 'cpiq', 'aptina'};
    
    for vi=1:length(vn_types)
        vn_type = vn_types{vi};
        [actual, col_labels, row_labels] = process_subjective_images(image_root, op_root, vn_type, 50.0);
        
        op_file = fullfile(op_root, vn_type, sprintf('cpiq_subjective_%s_vn_interp.csv', vn_type));
        fprintf(1, '%s: Writing "%s"\n', vn_type, op_file);
        table_writer(op_file, ...
                     actual,  col_labels,  row_labels, ...
                     'delimiter', ',', ...
                     'precision', 8, ...
                     'col_width', 0);
    end

end

function [op_data, col_labels, row_labels] = process_subjective_images(image_root, op_root, vn_type, lightness_point)

    patch_file_bases = {'macbeth_lpf_all_0128e_02cpd', ...
                        'macbeth_lpf_all_0128e_04cpd', ...
                        'macbeth_lpf_all_0128e_08cpd', ...
                        'macbeth_lpf_all_0128e_16cpd', ...
                        'macbeth_lpf_all_0256e_02cpd', ...
                        'macbeth_lpf_all_0256e_04cpd', ...
                        'macbeth_lpf_all_0256e_08cpd', ...
                        'macbeth_lpf_all_0256e_16cpd', ...
                        'macbeth_lpf_luma_0128e_02cpd', ...
                        'macbeth_lpf_luma_0128e_04cpd', ...
                        'macbeth_lpf_luma_0128e_08cpd', ...
                        'macbeth_lpf_luma_0128e_16cpd', ...
                        'macbeth_lpf_luma_0256e_02cpd', ...
                        'macbeth_lpf_luma_0256e_04cpd', ...
                        'macbeth_lpf_luma_0256e_08cpd', ...
                        'macbeth_lpf_luma_0256e_16cpd', ...
                        'macbeth_ls_0128e', ...
                        'macbeth_ls_0512e', ...
                        'macbeth_ls_2048e', ...
                        'macbeth_ls_8196e'};
        
     patch_count = length(patch_file_bases);
     
     op_data    = [];
     row_labels = {};
     
     for pi=1:patch_count
         
         patch_base = patch_file_bases{pi};
         
         switch(lower(vn_type))
             case {'aptina'}
                % Aptina Visual Noise Stats (From Keelan, Jin paper)
                [vn_stats, col_labels, vn_row_labels] = aptina_visual_noise_tool(fullfile(image_root, [patch_base,'*.ppm']), ...
                                                                                 'image_type',         'rgb', ...
                                                                                 'cycles_per_degree',   30.1593, ...
                                                                                 'radial_mode',         'new', ...
                                                                                 'lightness_point',     lightness_point, ...
                                                                                 'stats_return_type',   'output');
             case {'cpiq'}
                % IEEE-SA P1858 CPIQ Visual Noise Stats
                [vn_stats, col_labels, vn_row_labels]  = cpiq_visual_noise_tool(fullfile(image_root, [patch_base,'*.ppm']), ...
                                                                                'image_type',         'rgb', ...
                                                                                'cycles_per_degree',   30.1593, ...
                                                                                'radial_mode',         'new', ...
                                                                                'high_pass_filter',    true, ...
                                                                                'lightness_point',     lightness_point, ...
                                                                                'stats_return_type',   'output');
             otherwise
                 % ISO 15739 Visual Noise Stats
                 [vn_stats, col_labels, vn_row_labels]  = iso_visual_noise_tool(fullfile(image_root, [patch_base,'*.ppm']), ...
                                                                                'image_type',        'rgb', ...
                                                                                'cycles_per_degree', 30.1593, ...
                                                                                'radial_mode',       'new', ...
                                                                                'lightness_point',   lightness_point, ...
                                                                                'stats_return_type', 'output');
         end
         
         op_file = fullfile(op_root, vn_type, sprintf('%s_%s_vn_%s.csv', patch_base, vn_type, 'output'));
         fprintf(1, 'Patch %02d: Writing "%s"\n', pi, op_file);
         
         table_writer(op_file, vn_stats, ...
                      col_labels, vn_row_labels, ...
                      'delimiter', ',', ...
                      'precision', 8, ...
                      'col_width', 0);

         interp_vn = interpolate_visual_noise(vn_stats, lightness_point);
         
         op_data    = cat(1, op_data,   interp_vn);
         row_labels = cat(1, row_labels, patch_base);
         
     end

end

function [op_array] = interpolate_visual_noise(ip_array, lightness_point)
%INTERPOLATE_VISUAL_NOISE - interpolates the VN @ input lightness (L*) value
%   [op_array] = INTERPOLATE_VISUAL_NOISE(ip_array, lightness_point)

    op_array = ones(1, size(ip_array, 2)) * lightness_point;
    for di=2:size(ip_array, 2)
        op_array(1,di) = interp1(ip_array(:,1), ip_array(:,di), lightness_point, 'linear');
    end
 
end


