% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [op_array] = aptina_icclab_to_float(ip_array)
%APTINA_ICCLAB_TO_FLOAT APTINA_ICCLAB_TO_FLOAT
%    [op_array] = APTINA_ICCLAB_TO_FLOAT(ip_array)
%
% Description:
% Aptina ICCLAB 16-bit encoding is a customised version of the
% Matlab lab2double function. The L* is the same but the a* and b*
% have been customised to span a wider code range.
%   
% image channel 1 = (L*/100) x 65280;
% image channel 2 = (1+a*/100) x 65280/2;
% image channel 3 = (1+b*/100) x 65280/2;
%
% Reference(s):
%  1. http://www.aptina.com/ImArch/readme.doc
%  2. http://www.mathworks.co.uk/help/toolbox/images/ref/lab2double.html
%  3. Specification ICC.1:2001-12 File Format for Color Profiles (Version 4.0.0)
%     Annex A.2 & A.3
%     http://www.color.org/newiccspec.pdf

    if nargin < 1
       error('%s(): 1 arguments required, %d supplied\nUsage: op_array = %s(ip_array);', mfilename(), nargin, mfilename()); 
    end
    
    % Declare output image
    op_array = double(ip_array);

    % Perform the Aptina 16-bit to float conversion
    op_array(:,:,1) = 100.0*(op_array(:,:,1)/65280.0);
    op_array(:,:,2) = 100.0*(op_array(:,:,2)/(65280.0/2) - 1);
    op_array(:,:,3) = 100.0*(op_array(:,:,3)/(65280.0/2) - 1);

end

