% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [op_data, op_labels] =  aptina_visual_noise_patch(ip_patch, varargin) 
%APTINA_VISUAL_NOISE_PATCH Calculates the CPIQ visual noise for a single patch
%  [op_data, op_labels] = APTINA_VISUAL_NOISE_PATCH(ip_patch)
%  [op_data, op_labels] = APTINA_VISUAL_NOISE_PATCH(ip_patch, ParmName, ParmValue, ...)
%
% Description:
% This function implements the Aptin Visual Noise (VN) metric as reported
% in the Keelan paper [2]. With support for both sRGB and  Aptina's 16-bit
% ICCLAB images.
%
% The content of the returned output data and labels depends on the
% stats_return_type value.
%
% The visual noise procotol consists of the following steps:
%
% 1. sRGB flow:
%    a. sRGB linearisation with viewer observed black point correction
%    b. RGB linear -> XYZ(D65)
%    b  XYZ(D65) -> CIELAB(D65)
%    
% 2. Aptina 16-bit ICCLAB image
%    a. ICC 16-bit LAB encoding -> float32 CIELAB(D50) image
%       Note this is really the Aptina Modified ICC 16-bit encoding...
%       The encoding is defined in the Readme.doc [1] supplied with the images
% 3. Noise Power spectrum including cross spectra
% 4. Average horizontal and vertical NPS cross sections
% 5. Apply Circular Aperture MTF [3]
% 6. Apply Apple Cinema Monitor MTF
% 7. Polar Integration
%
% The objective metric value is a weight sum of the spatially filtered
% CIELAB Variances and Covariance's
%
% objective  = log10(1 + 100*var(L*) + 5*var(a*) + 12*covar(L*a*))
% subjective = ihif(objective)
%
% where ihif =  Integrated Hyperboloic Increment Function
%
% Aptina 16-bit ICCLAB Encoding:
% 
% Aptina ICCLAB 16-bit encoding is a customised version of the
% Matlab lab2double function. The L* is the same but the a* and b*
% have been customised to span a wider code range.
%
% image channel 1 = (L*/100) x 65280
% image channel 2 = (1+a*/100) x 65280/2
% image channel 3 = (1+b*/100) x 65280/2
%
% Parameters:
% big_endian
%   Used when reading reading binary .ppm/.pgm files. Set to True
%   for the .ppm reference image patches
%   Default: true
%
% image_type
%   Select 'rgb' for sRGB images like bmp, JPEG. Use 'icclab' when
%   using the Aptina 16-bit ICCLAB images.
%   Default: 'rgb'
%
% nominal_white
%   Max valid code in input image. Used for sRGB linearisation step
%   Not used for ICCLAB image inputs
%   Default: 255.0
%
% block_size
%   Block size for NPS Averaging e.g 32, 64, 128, 256.
%   Must be less than or equal to the patch size
%   Default = 64
%
% cycles_per_degree
%   Viewing distances in terms of cycles per degree.
%   Set to 30.1593cpd for the Aptina 16-bit ICCLAB images.
%   (30.1593 cpd = 34inch viewing condition for Apple 30in Cinema Monitor)
%   Default: 30.0
%
% radial_mode
%   'old' reproduces the behaviur of the 1st generation of visual noise
%   MATLAB funstions
%   See FFT_RADIAL_FREQUENCY for more details
%   Default: 'new'
%   Options: 'new', 'old'
%
% stats_return_type
%   Type of image statistics to return
%   'debug' = debug means & std dev for each step in the processing pipe
%   'covar' = the means, variances and covariances for the final SLAB image
%   'output' = lightness, objective metric and subjective metric values
%   Any type other than 'debug', 'covar' are treated as 'output'
%   Default: 'output'
%   Options: 'output', 'covar', 'stats'
%
% label
%   Patch Name/Label
%   Default: 'Patch XX'
%
% References:
% 1. Aptina noise study archived data (December 2010), Readme.doc, http://www.aptina.com/ImArch/
% 2. Keelan, B.W., Jin, E.W. and Prokushkin, S.,
%   "Development of a perceptually calibrated object metric of noise",
%    Proc SPIE 7867, Image Quality and System Performance VIII. (2011)
% 3. Bartleson, C.J.,
%    "Predicting Graininess from Granularity",
%    J. Photographic Science 33, 117-126 (1985)
%
% See also APTINA_VISUAL_NOISE_TOOL, SRGB_LINEARISATION, XYZ_TO_CIELAB,
% MATRIX_TRANSFORM,NOISE_POWER_SPECTRUM,  FFT_RADIAL_FREQUENCY,
% FFT_CROSS_SECTION, POLAR_INTEGRATION

    if nargin < 1
       error('%s(): min of 1 argument required, %d supplied\nUsage: [op_data, op_labels] = %s(ip_array, ParmName, ParmValue, ...);', mfilename(), nargin, mfilename()); 
    end

    % Define default values for the option parameters
    ip_parms = struct('big_endian',             true,...
                      'image_type',             'rgb', ...
                      'nominal_white',          255.0, ...
                      'block_size',              64.0, ...
                      'cycles_per_degree',       30.0, ...
                      'radial_mode',            'new', ...
                      'stats_return_type',    'output', ...
                      'label',                'Patch XX');
                  
    % Check for input parms name /value pairs              
    for i =1:2:length(varargin)-1
        parm_name = varargin{i};
        if isfield(ip_parms, parm_name);
            ip_parms.(parm_name) = varargin{i+1};
        else
            error('%s(): unknown parameter name "%s");', mfilename(), parm_name); 
        end
    end            

    % Check if the input is a file names. If so read in the data
    if ischar(ip_patch) && exist(ip_patch, 'file')
        [ip_root, ip_base, ip_ext] = fileparts(ip_patch);
        fprintf('Reading "%s"\n', ip_patch);
        switch(lower(ip_ext))
            case {'.pgm','.ppm'}
                [ip_patch, ip_parms.nominal_white] = pnm_reader(ip_patch, 'big_endian', ip_parms.big_endian);
            otherwise
                ip_patch = imread(ip_patch);
        end
    end
    
    % Catch icclab case for calibration images
    switch(lower(ip_parms.image_type))
        
        case {'icclab'}
            % This is option is required to process the Aptina LAB images
            % fprintf('%s(): Detected image type "%s", perfroming LAB->XYZ(D50)->XYZ(E) Conversion\n', mfilename, ip_parms.image_type);
            % Convert from 16-bit ICCLAB encoding to float
            images.LAB = aptina_icclab_to_float(double(ip_patch));
            ip_parms.nominal_white = 100.0;
            
        otherwise
            % Save input sRGB image
            images.RGB    = double(ip_patch);
            
            % Linearise inpyt RGB data with viewer observed black point
            % correction
            images.linRGB = srgb_linearisation( images.RGB, ...
                                                'nominal_white', ip_parms.nominal_white, ...
                                                'viewer_observed_black_point_correction', true );
            % Transform into XYZ(D65)
            images.XYZ_D65  = matrix_transform(   images.linRGB, get_matrix('RGBtoXYZ_D65'));
            
            % XYZ(D65) to CIELAB
            white_point = [0.9505, 1.0000, 1.0891]; % D65
            images.LAB = xyz_to_cielab(images.XYZ_D65, white_point);
            
    end
    
    % Generate NPS include cross spectra
    data.nps = noise_power_spectrum(images.LAB, ...
                                    'block_size',    ip_parms.block_size, ...
                                    'mean_type',     'local', ...
                                    'cross_spectra' , true);

    % Convert viewing conditions into pixels_per_degree
    pixels_per_degree = ip_parms.cycles_per_degree * 2.0;
    
    % Generate Viewing and Display Frequency filters
    uniform_image = ones(ip_parms.block_size, ip_parms.block_size, 6);
    
    data.view    = circular_aperture_mtf(uniform_image, ...
                                         'diameter_degrees',    0.090382, ...
                                         'pixels_per_degree',   pixels_per_degree, ...
                                         'mtf_squared',         true, ...
                                         'radial_mode',         ip_parms.radial_mode);

    data.display = display_print_mtf(uniform_image, ...
                                     'kprint',               5.65, ...
                                     'kdisplay',            0.947, ...
                                     'freq_scale_factor',     1.0, ...
                                     'mtf_squared',          true, ...
                                     'radial_mode',         ip_parms.radial_mode);
    
    % Get average cross sections
    xsections.nps     = fft_cross_section(data.nps,     'xsection_type', 'avg', 'radial_mode', ip_parms.radial_mode);
    xsections.view    = fft_cross_section(data.view,    'xsection_type', 'avg', 'radial_mode', ip_parms.radial_mode);
    xsections.display = fft_cross_section(data.display, 'xsection_type', 'avg', 'radial_mode', ip_parms.radial_mode);

    % Multiply together the responses
    % NPS  * view * display
    xsections.nps(:,2:end) =  xsections.nps(:,2:end) .*  xsections.view(:,2:end);
    xsections.nps(:,2:end) =  xsections.nps(:,2:end) .*  xsections.display(:,2:end);
    
    % Perform  polar integration to calculate variance and co-variance
    cv_data   = polar_integration(xsections.nps);
    cv_labels = {'SLAB.Var.L', 'SLAB.Var.a', 'SLAB.Var.b', 'SLAB.Covar.La', 'SLAB.Covar.Lb', 'SLAB.Covar.ab'};

    % Calculate objective noise metric
    % log10(1 + 100*var(L*) + 5*var(a*) + 12*covar(L*a*))
    objective_metric = log10( 1.0 + 100.0*cv_data(1) + 5.0*cv_data(2) + 12.0*cv_data(4));
    
    % Calculate quality loss in jnds 
    subjective_metric = ihif(objective_metric);
    
    % Calculate Image statistics
    if strcmpi(ip_parms.image_type, 'icclab')
        image_labels    = {'LAB'};
    else
        image_labels    = {'RGB', 'linRGB', 'XYZ_D65', 'LAB'};
    end
    
    image_count  = length(image_labels);
    db_labels    = {}; db_data = [];
    lightness    = 0.0;
    
    for i=1:image_count
        
        % Generate stats
        image_label = image_labels{i};
        [tmp_data, tmp_labels] = image_stats(images.(image_label), image_label);
        
        % Accumulate internal debug stats
        db_data   = cat(2, db_data,   tmp_data);
        db_labels = cat(2, db_labels, tmp_labels);
                
        % If LAB image then remember the lightness value
        if strcmpi(image_label, 'LAB')
            lightness = tmp_data(1);
        end
    end

    % Append results to VN stats
    vn_data   = [lightness, objective_metric, subjective_metric];
    vn_labels = {'Aptina.VN.lightness', 'Aptina.VN.objective', 'Aptina.VN.subjective'};

    % Select which set of data to return
    switch(lower(ip_parms.stats_return_type))
        case {'debug'}
            op_data   = db_data;
            op_labels = db_labels;
        case {'covar'}
            op_data   = cv_data;
            op_labels = cv_labels;
        otherwise
            op_data   = vn_data;
            op_labels = vn_labels;
    end
    
end
