% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [op_data, col_labels, row_labels] =  aptina_visual_noise_tool(ip_file_pattern, varargin) 
%APTINA_VISUAL_NOISE_TOOL Calculates the Aptina visual noise for each OECF patch
%  [op_data, col_labels, row_labels] = APTINA_VISUAL_NOISE_TOOL(ip_patch)
%  [op_data, col_labels, row_labels] = APTINA_VISUAL_NOISE_TOOL(ip_patch, ParmName, ParmValue, ...)
%
% Description:
% This a simple example wrapper function which generates ISO Visual Noise
% statistics over muliple images for the unit tests
%
% Parameters:
% big_endian
%   Used when reading reading binary .ppm/.pgm files. Set to True
%   for the .ppm reference image patches
%   Default: true
%
% image_type
%   Select 'rgb' for sRGB images like bmp, JPEG. Use 'icclab' when
%   using the Aptina 16-bit ICCLAB images.
%   Default: 'rgb'
%   Options: 'rgb', 'icclab'
%
% nominal_white
%   Max valid code in input image. Used for sRGB linearisation step
%   Not used for ICCLAB image inputs
%   Default: 255.0
%
% block_size
%   Block size for NPS Averaging e.g 32, 64, 128, 256.
%   Must be less than or equal to the patch size
%   Default = 64
%
% cycles_per_degree
%   Viewing distances in terms of cycles per degree.
%   Set to 30.1593 cpd for the Aptina 16-bit ICCLAB images
%   (30.1593 cpd = 34inch viewing condition for Apple 30in Cinema Monitor)
%   Default: 30.0
%
% radial_mode
%   'old' reproduces the behaviur of the 1st generation of visual noise
%   MATLAB funstions
%   See FFT_RADIAL_FREQUENCY for more details
%   Default: 'new'
%   Options: 'new', 'old'
%
% lightness_point
%   Lightness (L*) for interpolating the visual noise value
%   Default: 50.0
%
% stats_return_type
%   Type of image statistics to return
%   'debug' = debug means & std dev for each step in the processing pipe
%   'covar' = the means, variances and covariances for the final SLAB image
%   'output' = lightness, objective metric and subjective metric values
%   'interp' = inteprolated lightness, objective metric and subjective metric values
%   Default: 'output'
%   Options: 'output', 'covar', 'stats'
%
% References:
% 1. Aptina noise study archived data (December 2010), Readme.doc, http://www.aptina.com/ImArch/
% 2. Keelan, B.W., Jin, E.W. and Prokushkin, S.,
%   "Development of a perceptually calibrated object metric of noise",
%    Proc SPIE 7867, Image Quality and System Performance VIII. (2011)
% 3. Bartleson, C.J.,
%    "Predicting Graininess from Granularity",
%    J. Photographic Science 33, 117-126 (1985)
%
% See also APTINA_VISUAL_NOISE_PATCH
             
    if nargin < 1
       error('%s(): min of 1 arguments required, %d supplied\nUsage: [op_data, col_labels, row_labels] = %s(ip_file_pattern, ParmName, ParmValue, ...);', mfilename(), nargin, mfilename()); 
    end

    % Define default values for the option parameters
    ip_parms = struct('big_endian',             true,...
                      'image_type',             'rgb', ...
                      'nominal_white',          255.0, ...
                      'block_size',              64.0, ...
                      'cycles_per_degree',       30.0, ...
                      'radial_mode',            'new', ...
                      'lightness_point',         50.0, ...
                      'stats_return_type',     'output');
                  
    % Check for input parms name /value pairs              
    for i =1:2:length(varargin)-1
        parm_name = varargin{i};
        if isfield(ip_parms, parm_name);
            ip_parms.(parm_name) = varargin{i+1};
        else
            error('%s(): unknown parameter name "%s");', mfilename(), parm_name); 
        end
    end            

    % build file list
    file_list = get_file_list(ip_file_pattern);
    
    % Get number of Images's and ROI's
    file_count = size(file_list, 1);
    
    op_data = [];
    row_labels = {};
    
    % Process each image
    for fi = 1:file_count
        
        % Get input file name and split into root, base and ext
        ip_file = file_list{fi};
        [ip_root, ip_base, ip_ext] = fileparts(ip_file);
    
        % Get visual noise value for input image
        [patch_data, col_labels] = aptina_visual_noise_patch( ip_file, ...
                                                           'big_endian',             ip_parms.big_endian,...
                                                           'image_type',             ip_parms.image_type, ...
                                                           'block_size',             ip_parms.block_size, ...
                                                           'cycles_per_degree',      ip_parms.cycles_per_degree, ...
                                                           'radial_mode',            ip_parms.radial_mode, ...                                                             
                                                           'nominal_white',          ip_parms.nominal_white, ...
                                                           'stats_return_type',      ip_parms.stats_return_type, ...
                                                           'label',                  ip_base);

        % Accumulate patch data
        op_data    = cat(1, op_data, patch_data);
        row_labels = cat(1, row_labels, ip_base);
    end
    
    if strcmpi('interp', ip_parms.stats_return_type)
        op_data = interpolate_visual_noise(op_data, ip_parms.lightness_point);
        row_labels = {'interp'};
    end
    
end

function [op_array] = interpolate_visual_noise(ip_array, lightness_point)
%INTERPOLATE_VISUAL_NOISE - interpolates the VN @ input lightness (L*) value
%   [op_array] = INTERPOLATE_VISUAL_NOISE(ip_array, lightness_point)

    op_array = ones(1, size(ip_array, 2)) * lightness_point;
    for di=2:size(ip_array, 2)
        op_array(1,di) = interp1(ip_array(:,1), ip_array(:,di), lightness_point, 'linear');
    end
 
end
