% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [op_array] = cielab_to_xyz(ip_array, white_point)
%CIELAB_TO_XYZ CIELAB to XYZ color space transform
%    [op_array] = CIELAB_TO_XYZ(ip_array, white_point)
%
% Description:
% This is the reverse XYZ to CIELAB transform
%  
%  X = Xn * f((1/116)*(L* +16) + (1/500)a*)
%  Y = Yn * f((1/116)*(L* +16))
%  Z = Zn * f((1/116)*(L* +16) - (1/200)b*)
%   
%  where
%  
%  f(t) = t^3                    if t > (6/29)^3
%         3*((6/29)^2)*(t-4/29)  otherwise
%          
%   Xn, Yn, Zn is the reference white point
%   
% Reference(s):
%  1. http://en.wikipedia.org/wiki/Lab_color_space#Reverse_transformation
%  2. http://www.brucelindbloom.com/index.html?Eqn_Lab_to_XYZ.html
%
% See also XYZ_TO_CIELAB, GET_WHITE_POINT

    if nargin < 2
       error('%s(): 2 arguments required, %d supplied\nUsage: op_array = %s(ip_array, white_point);', mfilename(), nargin, mfilename()); 
    end
    
    % Declare output image
    tmp_image = zeros(size(ip_array));

    % Build intermediate temporary image
    tmp_image(:,:,1) =  ((1/116)*(ip_array(:,:,1)+16) + (1/500).*ip_array(:,:,2));    
    tmp_image(:,:,2) =  ((1/116)*(ip_array(:,:,1)+16));
    tmp_image(:,:,3) =  ((1/116)*(ip_array(:,:,1)+16) - (1/200).*ip_array(:,:,3));

    % Cube the input
    op_array = tmp_image.^3;
    op_array(tmp_image<=(6/29)) = 3*((6/29)^2)*(tmp_image(tmp_image<=(6/29))-(4/29));

    % Multiply by the white ponit
    op_array(:,:,1) =  white_point(1)*op_array(:,:,1);    
    op_array(:,:,2) =  white_point(2)*op_array(:,:,2);
    op_array(:,:,3) =  white_point(3)*op_array(:,:,3);

end

