% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [op_array] = circular_aperture_mtf(ip_array, varargin)
%CIRCULAR_APERTURE_MTF  Applies the Circular Aperture MTF the input FFT data
% [op_array] = CIRCULAR_APERTURE_MTF(ip_array)
% [op_array] = CIRCULAR_APERTURE_MTF(ip_array, ParmName, ParmValue, ...)
%
% Description:
% Applies the Circular Aperture MTF to input FFT data. 
%
% Bartleson's statement:
% 0.56mm diameter circular aperture was appropriate for predicting
% preceived grainness @ 355 mm viewing distance (naked eye)
%
% arctan(0.56 / 355) = 0.090382 degrees @ the retina (0.090 quoted by Keelan) 
%
%  f(v)  = 2 * Besselj(pi * D * v) / (pi * D * v)
%
% where v is cycles/degree
%
% Parameters:
% diameter_degress
%   Aperture diameter in degrees
%   Default: 0.090382
%
% pixels_per_degree:
%   2 * viewing angle in cycles per visual e.g 30.0 cpd -> 60.0 ppd
%   Default = 60.0
%
% mtf_squared:
%   Apply MTF Squared if true
%   Default: false
%
% radial_mode:
%   Workaround to allow the old unit test data to pass - mode 'old'
%   Default: 'new'
%
% Reference(s):
% 1. Keelan, B.W., Jin, E.W. and Prokushkin, S.,
%   "Development of a perceptually calibrated object metric of noise",
%    Proc SPIE 7867, Image Quality and System Performance VIII. (2011)
% 2. Bartleson, C.J.,
%    "Predicting Graininess from Granularity",
%    J. Photographic Science 33, 117-126 (1985)
%
% See also FFT_RADIAL_FREQUENCY, FFT_CROSS_SECTION
    
    if nargin < 1
       error('%s(): min of 1 arguments required, %d supplied\nUsage: [op_array] = %s(ip_array, ParmName, ParmValue, ...);', mfilename(), nargin, mfilename()); 
    end

    % Define default values for the option parameters
    ip_parms = struct( 'diameter_degrees',    0.090382, ...
                       'pixels_per_degree',   60.0, ...
                       'mtf_squared',         false, ...
                       'radial_mode',         'new');
                  
    % Check for input parms name /value pairs              
    for i =1:2:length(varargin)-1
        parm_name = varargin{i};
        if isfield(ip_parms, parm_name);
            ip_parms.(parm_name) = varargin{i+1};
        else
            error('%s(): unknown parameter name "%s");', mfilename(), parm_name); 
        end
    end

    % Build 2-D frequency map
    freq_range  = fft_radial_frequency(size(ip_array, 2),    size(ip_array, 1), ...
                                       'freq_scale_factor',  ip_parms.pixels_per_degree, ...
                                       'radial_mode',        ip_parms.radial_mode);
                                      
    % Build frequency domain filter
    op_filter = build_filter(freq_range, ip_parms);
    
    % Apply frequency domainfilter
    if ndims(ip_array) == 2 
        op_array = ip_array .* op_filter;    
    else
        op_array = ones(size(ip_array));
        for ci= 1: size(ip_array, 3)
            op_array(:, :, ci) = ip_array(:, : , ci) .* op_filter;
        end
    end

end

 
function [op_filter] = build_filter(freq_range, filter_parms)             
%BUILD_FILTER returns the CSF frequency domain filter
%  [op_filter] = BUILD_FILTER(freq_range, filter_parms)

    % Build Filter
    ip_data   = pi() .* filter_parms.diameter_degrees .* freq_range;
    op_filter = 2.0 .* besselj(1, ip_data) ./ ip_data; 
    op_filter(freq_range == 0.0) = 1.0;
    
    % Square the MTF if selected
    if filter_parms.mtf_squared
        op_filter = op_filter .* op_filter;
    end

end

