% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [op_array, electrons_per_code] = codes_to_electrons(ip_array, varargin)
%CODES_TO_ELECTRONS Summary of this function goes here
% [op_array, electrons_per_code] = CODES_TO_ELECTRONS(ip_array)
% [op_array, electrons_per_code] = CODES_TO_ELECTRONS(ip_array, ParmName, ParmValue, ...)
%
% Description:
% Converts the input Mono or RGB image from codes into electrons.
% 
% This step is necessary before photon shot noise can be determined.
%
% Parameters:
%
% ip_nominal_white
%   The maximum valid input code.
%   e.g. 255 for a 8-bit and 1023 for a 10-bit image
%   Default: 255.0
%
% max_signal_e
%   The maximum signal in terms of electrons. Varies as a function of
%   the image sensors analogue gain e.g. gain 1x = 5000e, gain 2x = 2500e
%   gain 4x = 1250e etc
%   Default: 5000.0e
%
% See also ELECTRONS_TO_CODES, SENSOR_NOISE_MODEL

    if nargin < 1
       error('%s(): min of 1 arguments required, %d supplied\nUsage: [op_array, electrons_per_code] = %s(ip_array, ParmName, ParmValue, ...);', mfilename(), nargin, mfilename()); 
    end

    % Define default values for the optional parameters
    ip_parms = struct('ip_nominal_white', 255.0, ...
                      'max_signal_e',     5000.0);
      
    % Check for input parms name /value pairs              
    for i =1:2:length(varargin)-1
        parm_name = varargin{i};
        if isfield(ip_parms, parm_name);
            ip_parms.(parm_name) = varargin{i+1};
        else
            error('%s(): unknown parameter name "%s");', mfilename(), parm_name); 
        end
    end
    
    electrons_per_code = ip_parms.max_signal_e / ip_parms.ip_nominal_white;
    op_array           = double(ip_array) * electrons_per_code;
   
end

