% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [op_data, op_labels] = covariance_stats(ip_array, image_type)
%COVARIANCE_STATS  Calculate channel variance and covariance values
%  [op_data, op_labels] = COVARIANCE_STATS(ip_array, image_type)
%
% Required inputs:
% * ip_array 
%     3 channel planar input image  e.g. RGB, XYZ, CIELAB
% * image_type
%     Image type string e.g. RGB, XYZ, LAB, LUV
%
% Returned values:
% op_data   = list of 3 var vlaues followed by 3 covariance values
% op_labels = list of value labels
%
% See also IMAGE_STATS

    if nargin < 2
       error('%s(): 2 arguments required, %d supplied\nUsage: op_array = %s(ip_array, ip_labels);', mfilename(), nargin, mfilename()); 
    end

    % Manage image labels
    channel_labels = get_channels(image_type);
    channel_count = length(channel_labels);
    stats_types = {'Mean', 'Var'};
    stats_count = length(stats_types);
    op_labels   = cell((stats_count+1)*channel_count,1);
    
    for si=1:stats_count
        for ci=1:channel_count
            op_labels{ci+(si-1)*channel_count} = sprintf('%s.%s.%s', image_type, stats_types{si}, channel_labels{ci});
        end
    end
    op_labels{7} = sprintf('%s.%s.%s%s', image_type, 'Covar', channel_labels{1}, channel_labels{2});
    op_labels{8} = sprintf('%s.%s.%s%s', image_type, 'Covar', channel_labels{1}, channel_labels{3});
    op_labels{9} = sprintf('%s.%s.%s%s', image_type, 'Covar', channel_labels{2}, channel_labels{3});
    
    % Reshape into a 1-D array
    [ip_height, ip_width, ip_depth] = size(ip_array);
    ip_array = reshape(ip_array, ip_height*ip_width, ip_depth);

    % Calculate variances and covariances
    cov_matrix = cov(ip_array);
    
    % Return variances and covariances    
    op_data = cat(2, mean(ip_array), diag(cov_matrix)', cov_matrix(1,2), cov_matrix(1,3), cov_matrix(2,3));

end