% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [op_array] = cpiq_spatial_filtering(ip_array, varargin)
%CPIQ_SPATIAL_FILTERING -  Filter
%  [op_array] = CPIQ_SPATIAL_FILTERING(ip_array)
%  [op_array] = CPIQ_SPATIAL_FILTERING(ip_array, ParmName, ParmValue, ...)
%   
% Description:
% Performs the following processing steps:
%
%  1. subtract patch means and 2-D FFT 
%  2. CSF - Either ISO Visual Noise or Johnson & Fairchild options
%  3. Apple Display MTF (Not part of ISO Visual Noise)
%  4. High Pass Filter (Not part of ISO Visual Noise)
%  5. Inverse 2-D FFT and restore patch means
%
% Parameters:
% csf_type
%   Contrast Senstivity Function (CSF) option
%     'iso' - ISO 15739 CSF
%     'jf'  - Johnson & Fairchild S-CIELAB CSF
%   Default: 'jf'
%   Options: 'iso', 'jf'
%
% pixel_per_degree
%   2 * viewing angle in cycles per visual e.g 30.0 cpd -> 60.0 ppd
%   Default: 60.0
%
% subtract_means
%   For CSF's with zero response at DC ('jf') the patch mean must
%   be subtracted before the FFT and restored after the inverse FFT.
%   Default: true
%
% radial_mode
%   'old' reproduces the behaviur of the 1st generation of visual noise
%   MATLAB funstions. See FFT_RADIAL_FREQUENCY for more details
%   Default: 'new'
%   Options: 'new', 'old'
%
% high_pass_filter
%   if true applies the high pass filter
%   Default: true
%
% See also FFT_RADIAL_FREQUENCY, ISO_VISUAL_NOISE_CSF, DISPLAY_PRINT_MTF,
% HIGH_PASS_FILTER, ISO_SPATIAL_FILTERING

    if nargin < 1
       error('%s(): min of 1 argument required, %d supplied\nUsage: op_array = %s(ip_array);', mfilename(), nargin, mfilename()); 
    end
    
    % Define default values for the option parameters
    ip_parms = struct('csf_type',                  'jf', ...
                      'pixels_per_degree',         60.0, ...
                      'subtract_means',            true, ...
                      'radial_mode',              'new', ...
                      'high_pass_filter',          true);
                  
    % Check for input parms name /value pairs              
    for i =1:2:length(varargin)-1
        parm_name = varargin{i};
        if isfield(ip_parms, parm_name);
            ip_parms.(parm_name) = varargin{i+1};
        else
            error('%s(): unknown parameter name "%s");', mfilename(), parm_name); 
        end
    end
    
    % Determine the input patch size
    [ip_height, ip_width, ip_depth] = size(ip_array);

    % Apply 2-D FFT to the image data and shift data
    % so that it is in the centre of the image
    freq_data   = zeros(ip_height, ip_width, ip_depth );
    patch_means = zeros( 1, ip_depth );
    for index=1:ip_depth
        patch_data = ip_array(:,:,index);
        if ip_parms.subtract_means
            patch_means(index) = mean(mean(ip_array(:,:,index)));
            patch_data         = patch_data - patch_means(index);
        end
        freq_data(:,:,index) = fftshift(fft2(patch_data));
    end

    % Apply csf frequency weighting function
    if ip_parms.pixels_per_degree > 0
        %fprintf('%s(): CSF: %s Selected\n', mfilename, ip_parms.csf_type);
        freq_data = iso_visual_noise_csf(freq_data, ...
                                         'csf_type',            ip_parms.csf_type, ...
                                         'pixels_per_degree',   ip_parms.pixels_per_degree, ...
                                         'radial_mode',         ip_parms.radial_mode);
    end

    % Display/Printer MTF
    if ip_parms.pixels_per_degree > 0
         %fprintf('%s(): Display/Printer MTF: Apple Cinema Monitor Selected\n', mfilename);
         freq_data = display_print_mtf(freq_data, ...
                                       'kprint',               5.65, ...
                                       'kdisplay',            0.947, ...
                                       'freq_scale_factor',     1.0, ...
                                       'mtf_squared',         false, ...
                                       'radial_mode',         ip_parms.radial_mode);
    end
     
     % High Pass Filter
     if (ip_parms.pixels_per_degree > 0) && ip_parms.high_pass_filter
         %fprintf('%s(): High Pass Filter Selected\n', mfilename);
         freq_data = high_pass_filter(freq_data, ...
                                      'pixels_per_degree',   ip_parms.pixels_per_degree, ...
                                      'radial_mode',         ip_parms.radial_mode);
     end
           
    % Unshift the filtered/weighted fft data and
    % perfrom the inverse FFT
    op_array = zeros(ip_height, ip_width, ip_depth);
    for index=1:ip_depth
        patch_data = real(ifft2(ifftshift(freq_data(:,:,index))));
        if ip_parms.subtract_means
            patch_data = patch_data + patch_means(index);
        end
        op_array(:,:,index) = patch_data;
    end

end

