% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [op_data, op_labels] = cpiq_visual_stats(lab_covar, varargin)
%CPIQ_VISUAL_STATS  Calculate ISO Visual noise value from the input LAB covar
%  [op_values, op_labels] = CPIQ_VISUAL_STATS(lab_covar)
%  [op_values, op_labels] = CPIQ_VISUAL_STATS(lab_covar, ParmName, ParmValue, ...)
%
% Description:
% Calculates the final CPIQ VN objective and subjectives metric value from
% the input CIELAB stats and covar data structures e.g.
%
% lab_covar  = [l_mean, a_mean, b_mean, l_var, a_var, b_var, la_covar, lb_covar, ab_covar]
%
% Returns array of objective stats values and cell array of column labels e.g.
%
% op_labels = {'CPIQ.VN.lightness', 'CPIQ.VN.objective', 'CPIQ.VN.subjective'}
% op_data   = [lightness, cpiq_objective_vn, cpiq_subjective_jnd_loss]
%
% Parameters:
% l_var_coeff
%   L* variance weight, Default: 23.00
%
% a_var_coeff
%   a* variance weight, Default: 4.24
%
% b_var_coeff
%   b* variance weight, Default: -5.47
%
% la_covar_coeff
%   l*a* covariance weight, Default: 4.77
%
% lb_covar_coeff
%   l*b* covariance weight, Default: 0.0
%
% ab_covar_coeff
%   a*b* covariance weight, Default: 0.0
%
% References:
% 1. ISO 15739 Working Draft 2010-03-10
% 2. Baxter, D.J., and Murray, A.,
%    Calibration and adaptation of ISO visual noise for I3A's Camera Phone Image Quality initiative,
%    Proc SPIE 8293, (2012)
%
% See also ISO_VISUAL_STATS, IHIF

    if nargin < 1
       error('%s(): 1 arguments required, %d supplied\nUsage: op_array = %s(lab_covar, ParmName, ParmValue, ...);', mfilename(), nargin, mfilename()); 
    end
 
     % Define default values for the option parameters
    ip_parms = struct( 'l_var_coeff',    23.00, ...
                       'a_var_coeff',     4.24, ...
                       'b_var_coeff',    -5.47, ...
                       'la_covar_coeff',  4.77, ...
                       'lb_covar_coeff',  0.00, ...
                       'ab_covar_coeff',  0.00, ...
                       'var_offset',      3);
                  
    % Check for input parms name /value pairs              
    for i =1:2:length(varargin)-1
        parm_name = varargin{i};
        if isfield(ip_parms, parm_name);
            ip_parms.(parm_name) = varargin{i+1};
        else
            error('%s(): unknown parameter name "%s");', mfilename(), parm_name); 
        end
    end 
    
    % Calculate CPIQ Visual Noise Objective Value
    noise    =  1.0 + ...
                ip_parms.l_var_coeff    * lab_covar(ip_parms.var_offset+1) + ...
                ip_parms.a_var_coeff    * lab_covar(ip_parms.var_offset+2) + ...
                ip_parms.b_var_coeff    * lab_covar(ip_parms.var_offset+3) + ...
                ip_parms.la_covar_coeff * lab_covar(ip_parms.var_offset+4) + ...
                ip_parms.lb_covar_coeff * lab_covar(ip_parms.var_offset+5) + ...
                ip_parms.ab_covar_coeff * lab_covar(ip_parms.var_offset+6);
    
    if noise > 0
        noise  = log10(noise);
    else
        noise = 0.0;
    end
    
    % Calculate CPIQ Visual Noise Subjective Value        
    jnd_loss = ihif(noise);

    op_data   = [lab_covar(1), noise, jnd_loss];
    op_labels = {'Aptina.VN.lightness', 'Aptina.VN.objective', 'Aptina.VN.subjective'};

end