% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [op_array] = display_print_mtf(ip_array, varargin)
%DISPLAY_PRINT_MTF -  Applies the Display/Print MTF the input FFT data
% [op_array] = DISPLAY_PRINT_MTF(ip_array)
% [op_array] = DISPLAY_PRINT_MTF(ip_array, ParmName, ParmValue, ...)
%
% Description:
% Applies the Display/Print MTF to the input FFT data
% Default is Apple 30inch Cinema MTF
%
% MTF(v) = exp(-v/kprint) |sinc((kdisplay*v)|
%
% where v is cycles per pixel (Nyquist = 0.5 units)
%
% Viewing conditions:
%   display pixel pitch = 0.250 mm
%   viewing distance    =   864 mm
%
% Parameters:
% kprint
%   Exponent function parameter
%   Default:  5.65
%
% kdisplay
%   Sinc function parameter
%   Default:  0.947
%
% freq_scale_factor
%   Nominally the frequency range generate is 0 - 0.5 cycles per pixel,
%   with 0(DC) in the centre of the array. The freq_scale_factor enables
%   this range to be scaled as required.
%   Default: 1.0
%
% mtf_squared:
%   Apply MTF Squared if true
%   Default: false
%
% radial_mode
%   Workaround to allow the old unit test data to pass - mode 'old'
%   Default: 'new'
%
% Reference(s):
% 1. Aptina Noise Study archived data (Dec 2010) readme.doc at Aptina Image Archives, http://www.aptina.com/ImArch/
%
% See also FFT_RADIAL_FREQUENCY

    if nargin < 1
       error('%s(): min of 1 arguments required, %d supplied\nUsage: [op_array] = %s(ip_array, ParmName, ParmValue, ...);', mfilename(), nargin, mfilename()); 
    end

    % Define default values for the option parameters
    ip_parms = struct( 'kprint',               5.65, ...
                       'kdisplay',            0.947, ...
                       'freq_scale_factor',     1.0, ...
                       'mtf_squared',         false, ...
                       'radial_mode',         'new');
                  
    % Check for input parms name /value pairs              
    for i =1:2:length(varargin)-1
        parm_name = varargin{i};
        if isfield(ip_parms, parm_name);
            ip_parms.(parm_name) = varargin{i+1};
        else
            error('%s(): unknown parameter name "%s");', mfilename(), parm_name); 
        end
    end
    
    % Build 2-D frequency map
    freq_range  = fft_radial_frequency(size(ip_array, 2), size(ip_array, 1), ...
                                       'freq_scale_factor',   ip_parms.freq_scale_factor, ...
                                       'radial_mode',         ip_parms.radial_mode);

    % Build frequency domain filter
    op_filter = build_filter(freq_range, ip_parms);
    
    % Apply frequency domainfilter
    if ndims(ip_array) == 2 
        op_array = ip_array .* op_filter;    
    else
        op_array = ones(size(ip_array));
        for ci= 1: size(ip_array, 3)
            op_array(:, :, ci) = ip_array(:, : , ci) .* op_filter;
        end
    end

end
        
function [op_filter] = build_filter(freq_range, filter_parms)             
% BUILD_FILTER returns the CSF frequency domain filter
%  [op_filter] = BUILD_FILTER(freq_range, filter_parms)

    % Build Filter
    op_filter = exp(-freq_range ./ filter_parms.kprint) .* abs(fn_sinc( filter_parms.kdisplay .* freq_range));

    % Square the MTF if selected
    if filter_parms.mtf_squared
        op_filter = op_filter .* op_filter;
    end

end
