% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [op_array] = fft_cross_section(ip_array, varargin)
%FFT_CROSS_SECTION Returns a horizontal or vertical cross section of the input FFT data
% op_array = FFT_CROSS_SECTION(ip_array)
% op_array = FFT_CROSS_SECTION(ip_array, ParmName, ParmValue, ...)
%
% Description:
% Extracts the requested cross section from a frequency representation
% of an image, e.g. either the 2-D FFT or noise power spectrum output.
% It is assumed that fftshift() function has alread been applied
% to FFT or NPS data i.e. the DC value is in the centre of the supplied
% input frequency data.
%
% Parameters:
% xsection_type
%   Cross section type - 'vert', 'hori' or 'avg'
%   where 'avg' is the average of the vertical and horizontal cross sections
%   Default = 'hori'
%
% freq_scale_factor
%   Nominally the frequency range generate is 0 - 0.5 cycles per pixel,
%   with 0(DC) in the centre of the array. The freq_scale_factor enables
%   this range to be scaled as required.
%   Default: 1.0
%
% radial_mode
%   Workaround to allow the old unit test data to pass - mode 'old'
%   Default: 'new'
%
% See also FFT_RADIAL_FREQUENCY

    if nargin < 1
       error('%s(): min of 1 arguments required, %d supplied\nUsage: [op_array] = %s(ip_array, ParmName, ParmValue, ...);', mfilename(), nargin, mfilename()); 
    end

    % Define default values for the option parameters
    ip_parms = struct('xsection_type',       'hori', ...
                      'freq_scale_factor',   1.0', ...
                      'radial_mode',         'new');
                  
    % Check for input parms name /value pairs              
    for i =1:2:length(varargin)-1
        parm_name = varargin{i};
        if isfield(ip_parms, parm_name);
            ip_parms.(parm_name) = varargin{i+1};
        else
            error('%s(): unknown parameter name "%s");', mfilename(), parm_name); 
        end
    end

    % Extract size and centre for FFT (DC location)
    height = size(ip_array, 1);
    width  = size(ip_array, 2);
    depth  = size(ip_array, 3);
    x_dc_index = floor(width/2)+1;
    y_dc_index = floor(height/2)+1;
    
    % Build 2-D frequency map
    freq_range = fft_radial_frequency(width, height, ...
                                     'freq_scale_factor',  ip_parms.freq_scale_factor, ...
                                     'radial_mode',        ip_parms.radial_mode);

    % Extract the appropriate cross section
    hori_xsection = squeeze(ip_array(y_dc_index, x_dc_index:end, :));
    vert_xsection = squeeze(ip_array(y_dc_index:end, x_dc_index, :));
    
    switch(lower(ip_parms.xsection_type))
        case {'vert'}    
            op_array = zeros(height - y_dc_index + 1, depth+1);
            op_array(:, 1)     = freq_range(y_dc_index:end, x_dc_index);
            op_array(:, 2:end) = vert_xsection;
        case {'hori'}
            op_array = zeros(width - x_dc_index + 1, depth+1);
            op_array(:, 1)     = freq_range(y_dc_index, x_dc_index:end);
            op_array(:, 2:end) = hori_xsection;
        case {'avg'}
            op_array = zeros(width - x_dc_index + 1, depth+1);
            op_array(:, 1)     = freq_range(y_dc_index, x_dc_index:end);
            op_array(:, 2:end) = (hori_xsection + vert_xsection) / 2.0;
        otherwise
            error('%s(): unsupported X-Section type "%s"!  - valid "hori", "vert", "avg"', mfilename(), ip_parms.xsection_type);
    end
    
end
