% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [op_array] = fft_radial_frequency(width, height, varargin)
%FFT_RADIAL_FREQUENCY - returns a 2-D radial frequency map
% op_array = FFT_RADIAL_FREQUENCY(width, height);
% op_array = FFT_RADIAL_FREQUENCY(width, height, ParmName, ParmValue, ...);
%
% Description:
% Builds a 2D radial frequency map of the requested width and height. 
% Intended for use with 2-D FFT data that has been fftshifted or to
% generate frequency domain filters to apply to 2-D FFT data
%
% The 'old' radial mode is only for passing old unit tests where
% the my 1st MATLAB implementation of the radial frequency generation
% had an issue, e.g.
%
% radial_mode = 'old':
% 1.414214    1.118034    1.000000    1.118034
% 1.118034    0.707107    0.500000    0.707107
% 1.000000    0.500000    0.000000    0.500000
% 1.118034    0.707107    0.500000    0.707107
% 
% radial_mode = 'new' (correct):
% 0.707107    0.559017    0.500000    0.559017
% 0.559017    0.353553    0.250000    0.353553
% 0.500000    0.250000    0.000000    0.250000
% 0.559017    0.353553    0.250000    0.353553
%
% Parameters:
%
% freq_scale_factor:
%   Nominally the frequency range generate is 0 - 0.5 cycles per pixel,
%   with 0(DC) in the centre of the array. The freq_scale_factor enables
%   this range to be scaled as required.
%   Default: 1.0
%
% radial_mode:
%   Workaround to allow the old unit test data to pass - mode 'old'
%   Default: 'new'
%
% See also FFT_CROSS_SECTION

    if nargin < 2
       error('%s(): min of 2 arguments required, %d supplied\nUsage: [op_array] = %s(width, height, ParmName, ParmValue, ...);', mfilename(), nargin, mfilename()); 
    end

    % Define default values for the option parameters
    ip_parms = struct( 'freq_scale_factor',   1.0, ...
                       'radial_mode',         'new');
                  
    % Check for input parms name /value pairs              
    for i =1:2:length(varargin)-1
        parm_name = varargin{i};
        if isfield(ip_parms, parm_name);
            ip_parms.(parm_name) = varargin{i+1};
        else
            error('%s(): unknown parameter name "%s");', mfilename(), parm_name); 
        end
    end  
    
    % For fft the 0.5 cycles/pixel (nyquist) point = width/2
    % 0.5 cycles/pixel = x_nyquist = width/2
    % 0.5 cycles/pixel = y_nyquist = height/2
    x_dc_index = floor(width/2)+1;
    y_dc_index = floor(height/2)+1;
    
    x_nyquist_point = floor(width/2);
    y_nyquist_point = floor(height/2);
    
    if strcmpi(ip_parms.radial_mode, 'old')
        if mod(width,2) == 0
            x_nyquist_point = x_nyquist_point - 1;
        end
        if mod(height,2) == 0
            y_nyquist_point = y_nyquist_point - 1;
        end
    end

    % Build cycles per pixel array, converting the radius so that
    % at the nyquist radius the cycles per pixel is 0.5 cycles/pixel
    [xi, yi] = meshgrid(1:width, 1:height);
     xi      = (xi - x_dc_index );
     xi      = ((0.5 .* xi) ./ x_nyquist_point) .* ip_parms.freq_scale_factor;
     yi      = (yi - y_dc_index);
     yi      = ((0.5 .* yi) ./ y_nyquist_point) .* ip_parms.freq_scale_factor;
    op_array  = sqrt(xi.^2 + yi.^2);
  
end
