% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [op_array] = get_channels(image_type)
%GET_CHANNELS Returns a cell array of the channel labels for the input image type
%  op_array = GET_CHANNELS()
%  op_array = GET_CHANNELS(image_type)
%
% Description:
% Returns a list of the channel labels for the input image type.
% If no input is specified then 'RGB' is assumed
%

    if nargin < 1, image_type = 'RGB'; end
    
    channels_lut = struct('RGB',      {'R', 'G',  'B' }, ...
                          'XYZ',      {'X', 'Y',  'Z' }, ...
                          'AC1C2',    {'A', 'C1', 'C2'}, ...
                          'LAB',      {'L', 'a',  'b' }, ...
                          'LUV',      {'L', 'u',  'v' }, ...
                          'linRGB',   {'R', 'G',  'B' }, ...
                          'XYZ_D65',  {'X', 'Y',  'Z' }, ... 
                          'XYZ_D50',  {'X', 'Y',  'Z' }, ... 
                          'XYZ_E',    {'X', 'Y',  'Z' }, ...
                          'SAC1C2',   {'A', 'C1', 'C2'}, ...
                          'SXYZ_E',   {'X', 'Y',  'Z' }, ...
                          'SXYZ_D65', {'X', 'Y',  'Z' }, ...
                          'SLAB',     {'L', 'a',  'b' }, ...
                          'SLUV',     {'L', 'u',  'v' }, ...
                          'linSRGB',  {'R', 'G',  'B' }, ...
                          'SRGB',     {'R', 'G',  'B' } );

    op_array = get_channels_as_cell_array(channels_lut, image_type);
 
end

function [op_array] = get_channels_as_cell_array(channels_lut, image_type)
% GET_CHANNELS_AS_CELL_ARRAY - convenience function for fetching channel labels
%  [op_array] = GET_CHANNELS_AS_CELL_ARRAY(channels_lut, image_type)
    if isfield(channels_lut, image_type)
        op_array = {channels_lut(1).(image_type), channels_lut(2).(image_type), channels_lut(3).(image_type)};
    else
        valid_types = fieldnames(channels_lut);
        error('Unknown Image Type "%s"!\nValid Image Types:\n%s', image_type, sprintf('"%s"\n', valid_types{:}));
    end
end
