% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [op_array] = get_white_point(illuminant, data_source)
%GET_WHITE_POINT Returns the XYZ white point for specified illuminant
%  op_array = GET_WHITE_POINT()
%  op_array = GET_WHITE_POINT(illuminant)
%
% Description:
% Returns the XYZ white point for the input illuminant name
% If no illuminant is specified, 'D65' white point is returned
%
% For reasons which are not clear the D65 white point is slight different
% to the MATLAB values
%
% Please note the Octave does not seem to support the MATLAB whitepoint
% function
%
% Matlab XYZ White point definitions:
% 
% A   = [1.0985, 1.0000, 0.3558]
% C   = [0.9807, 1.0000, 1.1823]
% D50 = [0.9642, 1.0000, 0.8251]
% D55 = [0.9568, 1.0000, 0.9214]
% D65 = [0.9504, 1.0000, 1.0889] (default)
% ICC = [0.9642, 1.0000, 0.8249]
%
% ISO Visual Noise White point definition:
%
% D65 = [0.9505, 1.0000, 1.0891] (default)
%
% References:
% 1. http://www.mathworks.ch/ch/help/images/ref/whitepoint.html
% 2. http://www.brucelindbloom.com/
%
% See also GET_MATRIX

    if nargin < 1, illuminant  = 'D65'; end
    if nargin < 2, data_source = 'MATLAB'; end
    
    if strcmpi(data_source, 'matlab')
        % Use MATLAB white point defintions
        white_point_lut = struct('A',   [1.0985,  1.0000,  0.3558], ...
                                 'C',   [0.9807,  1.0000,  1.1823], ...
                                 'D50', [0.9642,  1.0000,  0.8251], ...
                                 'D55', [0.9568,  1.0000,  0.9214], ...
                                 'D65', [0.9504,  1.0000,  1.0889], ...
                                 'ICC', [0.9642,  1.0000,  0.8249], ...
                                 'E',   [1.0000,  1.0000,  1.0000]);
    elseif strcmpi(data_source, 'lindbloom')
        % Use Bruce Lindbloom (ASTM E308-01) white point defintions
        white_point_lut = struct('A',   [1.09850, 1.00000, 0.35585], ...
                                 'B',   [0.99072, 1.00000, 0.85223], ...
                                 'C',   [0.98074, 1.00000, 1.18232], ...
                                 'D50', [0.96422, 1.00000, 0.82521], ...
                                 'D55', [0.95682, 1.00000, 0.92149], ...
                                 'D65', [0.95047, 1.00000, 1.08883], ...
                                 'D75', [0.94972, 1.00000, 1.22638], ...
                                 'E',   [1.00000, 1.00000, 1.00000], ...
                                 'F2',  [0.99186, 1.00000, 0.67393], ...
                                 'F7',  [0.95041, 1.00000, 1.08747], ...
                                 'F11', [1.00962, 1.00000, 0.64350]);
    else
        % ISO Visual Noise White Point Defintion
        % fprintf('%s(): Using ISO Visual Noise White point definition.\n', mfilename);
        white_point_lut = struct('D65', [0.9505,  1.0000,  1.0891], ...
                                 'E',   [1.0000,  1.0000,  1.0000]);
    end
                             
    if isfield(white_point_lut, illuminant)
    	op_array = white_point_lut.(illuminant);
    else
        valid_illuminants = fieldnames(white_point_lut);
        error('Unknown XYZ Illuminant "%s"!\nValid illuminants:\n%s', illuminant,sprintf('"%s"\n', valid_illuminants{:}));
    end
 
end

