% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [op_array] = high_pass_filter(ip_array, varargin)
%HIGH_PASS_FILTER  Applies a high pass filter to the input FFT data
% [op_array] = HIGH_PASS_FILTER(ip_array)
% [op_array] = HIGH_PASS_FILTER(ip_array, ParmName, ParmValue, ...)
%
% Description:
% Applies the High Pass Filter to input FFT data. Assume to be AC1C2
% opponent colour space.
%
%  f(v)  = (2.0 / ( 1 + exp(-cut_off * v))) - 1
%
% where v is cycles/degree
%
% Parameters:
% cut_off
%   Sets the cut off frequency
%   Default: 4.0
%
% pixels_per_degree
%   2 * viewing angle in cycles per visual e.g 30.0 cpd -> 60.0 ppd
%   Default: 60.0
%
% radial_mode:
%   Workaround to allow the old unit test data to pass - mode 'old'
%   Default: 'new'
%
% Reference:
%
%
% See also FFT_RADIAL_FREQUENCY, FFT_CROSS_SECTION
    
    if nargin < 1
       error('%s(): min of 1 arguments required, %d supplied\nUsage: [op_array] = %s(ip_array, ParmName, ParmValue, ...);', mfilename(), nargin, mfilename()); 
    end

    % Define default values for the option parameters
    ip_parms = struct( 'cut_off',              4.0, ...
                       'pixels_per_degree',   60.0, ...
                       'radial_mode',         'new');
                  
    % Check for input parms name /value pairs              
    for i =1:2:length(varargin)-1
        parm_name = varargin{i};
        if isfield(ip_parms, parm_name);
            ip_parms.(parm_name) = varargin{i+1};
        else
            error('%s(): unknown parameter name "%s");', mfilename(), parm_name); 
        end
    end;

    % Build 2-D frequency map
    freq_range  = fft_radial_frequency(size(ip_array, 2), size(ip_array, 1), ...
                                       'freq_scale_factor',   ip_parms.pixels_per_degree, ...
                                       'radial_mode',         ip_parms.radial_mode);
                                      
    % Build frequency domain filter
    op_filter = build_filter(freq_range, ip_parms);
    
    % Apply frequency domainfilter
    op_array = ip_array .* op_filter;

end

 
function [op_filter] = build_filter(freq_range, filter_parms)             
%BUILD_FILTER returns the CSF frequency domain filter
%  [op_filter] = BUILD_FILTER(freq_range, filter_parms)

    % Initialise filter
    op_filter = zeros(size(freq_range,1), size(freq_range, 2), 3);

    % Build Filter
    op_filter(:,:,1) = (2.0 ./ ( 1 + exp(-filter_parms.cut_off .* freq_range))) - 1.0; 
    op_filter(:,:,2) = (2.0 ./ ( 1 + exp(-filter_parms.cut_off .* freq_range))) - 1.0; 
    op_filter(:,:,3) = (2.0 ./ ( 1 + exp(-filter_parms.cut_off .* freq_range))) - 1.0;           

end

