% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function jnd_loss = ihif(ip_array, varargin)
% IHIF - Integrated Hyperbolic Increment Function
%  jnd_loss = ihif(ip_array)
%  jnd_loss = ihif(ip_array, ParmName, ParmValue, ...)
%
% Description:
% The Integrated Hyperbolic Increment Function (IHIF) maps the Visual
% Noise objective metric to quality loss JND values
%
% Parameters:
% jnd_max
% Default: 30.93
%
% ohm_r
% Default: 0.4627
%
% r_r
% Default: 0.1418
%
% delta_ohm_inf
% Default: 0.02312
%
% References:
% 1. Keelan, B.W., Jin, E.W and Prokushkin, S.,
%    Development of a perceptually calibrated object metric of noise,
%    Proc SPIE 7867, p786708 (2011).
% 2. Baxter, D.J., and Murray, A.,
%    Calibration and adaptation of ISO visual noise for I3A's Camera Phone Image Quality initiative,
%    Proc SPIE 8293, (2012)
%
% See also CPIQ_VISUAL_STATS

    if nargin < 1
       error('%s(): min of 1 arguments required, %d supplied\nUsage: [op_array] = %s(ip_array, ParmName, ParmValue, ...);', mfilename(), nargin, mfilename()); 
    end

    % Define default values for the option parameters
    ip_parms = struct( 'jnd_max',       30.93,  ...
                       'ohm_r',         0.4627, ...
                       'r_r',           0.1418, ...
                       'delta_ohm_inf', 0.02312);
                  
    % Check for input parms name /value pairs              
    for i =1:2:length(varargin)-1
        parm_name = varargin{i};
        if isfield(ip_parms, parm_name);
            ip_parms.(parm_name) = varargin{i+1};
        else
            error('%s(): unknown parameter name "%s");', mfilename(), parm_name); 
        end
    end  

    % Calculate Integrated Hyperbolic Increment Function
    jnd_loss = (ip_array - ip_parms.ohm_r) / ip_parms.delta_ohm_inf;
    jnd_loss = jnd_loss - ((ip_parms.r_r/(ip_parms.delta_ohm_inf^2)).*log( 1 + ((ip_parms.delta_ohm_inf.*(ip_array - ip_parms.ohm_r))./ip_parms.r_r)));
    
    % Clip
    jnd_loss(ip_array <= ip_parms.ohm_r) = 0;

end

