% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [op_array] = iso_visual_noise_csf(ip_array, varargin)
%ISO_VISUAL_NOISE_CSF  Applies the CSF to the input frequency domain image data 
% [op_array] = ISO_VISUAL_NOISE_CSF(ip_array)
% [op_array] = ISO_VISUAL_NOISE_CSF(ip_array, ParmName, ParmValue, ...)
%
% Description:
% Applies the ISO or Johnson & Fairchild Contrast Sensitivity Functions (CSF)
% to the input 3 channel frequency domain image data. It is assumed that the
% input data is already in either the AC1C2 or YC1C2 opponent color space
%
% The channels are as follows:
% 
% A  = Luminance CSF
% C1 = Red-Green Chrominance CSF
% C2 = Blue-Yellow Chrominance CSF 
% 
% The ISO CSF is a variant of the Johnson & Fairchild frequency
% domain S-CIELAB filter but with peak response of the Luminance
% channel CSF increased to 3.0. This can result is clipping later
% in the visual noise protocol if the noise level in the image is
% high e.g. the level of noise observered in low light conditions
% for cell phone cameras
%
% Johnson & Fairchild SCIELAB CSF Parameters:
%
% Parameter    Luminance    Red-Green    Blue-Yellow    Comments
%    a         1.00                                    
%    b         0.20            
%    c         0.80                                    
%    K         1.00
%    S         0.00
%
%    a1                    109.1413       7.0328    
%    b1                      0.0004       0.0000       Sign wrong in [2]
%    c1                      3.4244       4.2582    
%    a2                     93.5971      40.6910    
%    b2                      0.0037       0.1039    
%    c2                      2.1677       1.6487       Sign wrong in [2]
%    K                     202.7384      40.6910       From ISO 15739 - Not mentioned in [3] but required to normalise the response
%    S                       0.0000       7.0328       From ISO 15739 - Not mentioned in [3] but required to ensure that the response tends to zero
%
% ISO 15739 Visual Noise CSF Parameters:
% As above but with
%
% Parameter    Luminance    Red-Green    Blue-Yellow    Comments
%    a        75.00                                    Different lluminance equation vs S-CIELAB
%    b         0.20            
%    c         0.90                                     0.8 in[3]
%    K        46.00                                    Not mentioned in [3] paper but required to normalise the response (determines peak value)
%    S        46.00
%
%
% Parameters:
% csf_type:
%   'iso' for ISO 15739 CSF
%   'jf'  for the Johnson & Fairchild SCIELAB CSF
%   Default: 'iso'
%
% pixels_per_degree:
%   2 * viewing angle in cycles per visual e.g 30.0 cpd -> 60.0 ppd
%   Default = 60.0
%
% radial_mode
%   Workaround to allow the old unit test data to pass - mode 'old'
%   Default: 'new'
%
% Reference(s):
% 1. ISO 15739 WD#3 Oct 2010 Photography - Electronic still picture imaging - Noise measurements
% 2. Aoyama, K., Enomoto H., and. Hung, P.C, "An evaluation of scanner noise based upon a human
%    visual model", IS&T's 49th Annual Conference, pp. 322-324, (1996)
% 3. Johnson, G.M., and Fairchild, M., A top down description of S-CIELAB and CIEDE2000,
%    Color Res. Appl. 28, 435-35 (2003)
% 4. Color Appearance Models, second edition, John Wiley & son, Mark Fairchild, p28
%
% See also FFT_RADIAL_FREQUENCY, FFT_CROSS_SECTION

    if nargin < 1
       error('%s(): min of 1 arguments required, %d supplied\nUsage: [op_array] = %s(ip_array, ParmName, ParmValue, ...);', mfilename(), nargin, mfilename()); 
    end

    % Define default values for the option parameters
    ip_parms = struct('csf_type',           'iso', ...
                      'pixels_per_degree',   60.0', ...
                      'radial_mode',        'new');
                  
    % Check for input parms name /value pairs              
    for i =1:2:length(varargin)-1
        parm_name = varargin{i};
        if isfield(ip_parms, parm_name);
            ip_parms.(parm_name) = varargin{i+1};
        else
            error('%s(): unknown parameter name "%s");', mfilename(), parm_name); 
        end
    end
    
    % Get CSF parms
    filter_parms = get_filter_parms(ip_parms.csf_type);

    % Build 2-D frequency map
    freq_range  = fft_radial_frequency(size(ip_array, 2),   size(ip_array, 1), ...
                                       'freq_scale_factor', ip_parms.pixels_per_degree, ...
                                       'radial_mode',       ip_parms.radial_mode);
    
    % Build frequency domain filter
    op_filter = build_filter(freq_range,  filter_parms);
    
    % Apply frequency domainfilter
    op_array = ip_array .* op_filter;

end

function [op_filter] = build_filter(freq_range, filter_parms)             
% BUILD_FILTER returns the CSF frequency domain filter
%  [op_filter] = BUILD_FILTER(freq_range, filter_parms)

    % Initialise filter
    op_filter = zeros(size(freq_range,1), size(freq_range, 2), 3);

    % Luminance (A) Frequency Filter
    op_filter(:,:,1) = ((filter_parms.a.S + filter_parms.a.a1*(freq_range.^filter_parms.a.c1)).*exp(-filter_parms.a.b1.* freq_range))./filter_parms.a.K;

    % Red-Green Chrominance (C1) Frequency Filter
    op_filter(:,:,2) = ( filter_parms.c1.a1.*exp(-filter_parms.c1.b1.*(freq_range.^filter_parms.c1.c1)) + ...
                         filter_parms.c1.a2.*exp(-filter_parms.c1.b2.*(freq_range.^filter_parms.c1.c2)) - filter_parms.c1.S )./filter_parms.c1.K;

    % Blue-Yellow Chrominance (C2) Frequency Filter
    op_filter(:,:,3) = ( filter_parms.c2.a1.*exp(-filter_parms.c2.b1.*(freq_range.^filter_parms.c2.c1)) + ...
                         filter_parms.c2.a2.*exp(-filter_parms.c2.b2.*(freq_range.^filter_parms.c2.c2)) - filter_parms.c2.S )./ filter_parms.c2.K;           

end
        
  
function [op_parms] = get_filter_parms(csf_type)
%GET_FILTER_PARMS  Returns CSF parms for ISO Visual Noise CSF function
% [op_parms] = get_filter_parms(csf_variant)
     
    % Initialise ISO Constrast senstivity function 
    op_parms  = struct( 'a', struct(), 'c1', struct(), 'c2', struct() );
    
    % Luminance (A) CSF
    if strcmpi(csf_type, 'iso')
        % ISO adaptation of Johnson & Fairchild parameters
        op_parms.a.a1  =  75;
        op_parms.a.b1  =   0.2;
        op_parms.a.c1  =   0.9;
        op_parms.a.K   =  46.0;
        op_parms.a.S   =  46.0;
    else
        % Original Johnson & Fairchild SCIELAB parameters
        op_parms.a.a1  =  1;
        op_parms.a.b1  =   0.2;
        op_parms.a.c1  =   0.8;
        op_parms.a.K   =  1;
        op_parms.a.S   =  0;
    end
    
    % Red-Green Chrominance (C1) CSF
    op_parms.c1.a1 = 109.1413;
    op_parms.c1.b1 =   0.0004;
    op_parms.c1.c1 =   3.4244;
    op_parms.c1.a2 =  93.5971;
    op_parms.c1.b2 =   0.0037;
    op_parms.c1.c2 =   2.1677;
    op_parms.c1.K  = 202.7384;
    op_parms.c1.S  =   0.0000;
    
    % Blue-Yellow Chrominance (C2) CSF
    op_parms.c2.a1 =   7.0328;
    op_parms.c2.b1 =   0.0000;
    op_parms.c2.c1 =   4.2582;
    op_parms.c2.a2 =  40.6910;
    op_parms.c2.b2 =   0.1039;
    op_parms.c2.c2 =   1.6487;
    op_parms.c2.K  =  40.6910;
    op_parms.c2.S  =   7.0328;
    
end

