% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [op_data, op_labels] = iso_visual_noise_patch(ip_patch, varargin) 
%ISO_VISUAL_NOISE_PATCH Calculates the ISO visual noise for a single patch
%  [op_data, op_labels] = ISO_VISUAL_NOISE_PATCH(ip_patch)
%  [op_data, op_labels] = ISO_VISUAL_NOISE_PATCH(ip_patch, ParmName, ParmValue, ...)
%
% Description:
% This function implements the ISO 15739 Visual Noise (VN) metric as per
% the latest in progress working draft. 
%  
% Support for Aptina's 16-bit ICCLAB calibrated noise images [3.4] has been
% added.
%
% The content of the returned output data and labels depends on the
% stats_return_type value.
%
% The visual noise procotol consists of the following steps:
%   
% 1. sRGB flow:
%    a. sRGB linearisation with viewer observed black point correction
%    b. RGB linear -> XYZ(E)
%    
% 2. Aptina 16-bit ICCLAB image
%    a. ICC 16-bit LAB encoding -> float32 CIELAB(D50) image
%       Note this is really the Aptina Modified ICC 16-bit encoding...
%       The encoding is defined in the Readme.doc [3] supplied with the images
%    b  CIELAB(D50) -> XYZ(D50)
%    c. XYZ(D50) -> XYZ(E) Bradford Transform
%
% 3. XYZ(E) -> AC1C2 opponent colour space
% 4. Frequency based visual aligned spatial filtering
%    a. Contrast Sensitivity Function
% 5. AC1C2 -> XYZ(E)
% 6. XYZ(E) -> XYZ(D65)
% 7. XYZ(D65) -> CIELUV
%    Note that the white point is defined as [0.9505, 1.0000, 1.0891]
%    This is subtly different to other definitions of the D65 white point
%
%  The final visual noise value is a weight sum of the CIELUV channel
%  standard deviation values
%   
%  visual_noise = 1.000*noise(L*) + 0.852*noise(u*) + 0.323*noise(v*)
%
% Parameters:
% big_endian
%   Used when reading reading binary .ppm/.pgm files. Set to True
%   for the .ppm reference image patches
%   Default: true
%
% image_type
%   Select 'rgb' for sRGB images like bmp, JPEG. Use 'icclab' when
%   using the Aptina 16-bit ICCLAB images.
%   Default: 'rgb'
%   Options: 'rgb', 'icclab'
%
% nominal_white
%   Max valid code in input image. Used for sRGB linearisation step
%   Not used for ICCLAB image inputs
%   Default: 255.0
%
% csf_type
%   Contrast Senstivity Function (CSF) option
%     'iso' - ISO 15739 CSF
%     'jf'  - Johnson & Fairchild S-CIELAB CSF
%   Default: 'iso'
%   Options: 'iso', 'jf'
%
% cycles_per_degree
%   Viewing distances in terms of cycles per degree.
%   Set to 30.1593 cpd for the Aptina 16-bit ICCLAB images
%   (30.1593 cpd = 34inch viewing condition for Apple 30in Cinema Monitor)
%   Default: 30.0
%
% subtract_means
%   For CSF's with zero response at DC ('jf') the patch mean must
%   be subtracted before the FFT and restored after the inverse FFT.
%   Default: false
%
% radial_mode
%   'old' reproduces the behaviur of the 1st generation of visual noise
%   MATLAB funstions
%   See FFT_RADIAL_FREQUENCY for more details
%   Default: 'new'
%   Options: 'new', 'old'
%
% high_pass_filter
%   If true applies the high pass filter
%   Default: true
%
% stats_return_type
%   Type of image statistics to return
%   'debug' = debug means & std dev for each step in the processing pipe
%   'covar' = the means, variances and covariances for the final SLAB image
%   'output' = lightness, objective metric and subjective metric values
%   Any type other than 'debug', 'covar' are treated as 'output'
%   Default: 'output'
%   Options: 'output', 'covar', 'stats'
%
% label
%   Patch Name/Label
%   Default: 'Patch XX'
%
% References:
%  1. ISO 15739 WD#3 Oct 2010 Photography - Electronic still picture imaging - Noise measurements
%  2. Aoyama, K., Enomoto H., and. Hung, P.C,
%     "An evaluation of scanner noise based upon a human visual model",
%     IS&T's 49th Annual Conference, pp. 322-324,(1996)
%  3. Aptina noise study archived data (December 2010), Readme.doc, http://www.aptina.com/ImArch/
%  4. "Digital Video and HDTV Algorithms and Interfaces", Charles Poynton
%
% See also ISO_VISUAL_NOISE_TOOL, SRGB_LINEARISATION, CIELAB_TO_XYZ,
% MATRIX_TRANSFORM, FFT_RADIAL_FREQUENCY, ISO_SPATIAL_FILTERING, XYZ_TO_CIELUV

             
    if nargin < 1
       error('%s(): min of 1 argument required, %d supplied\nUsage: [vn_stats, vn_labels, db_stats, db_labels] = %s(ip_array, ParmName, ParmValue, ...);', mfilename(), nargin, mfilename()); 
    end

    % Define default values for the option parameters
    ip_parms = struct('big_endian',             true,...
                      'image_type',             'rgb', ...
                      'nominal_white',          255.0, ...
                      'csf_type',               'iso', ...
                      'cycles_per_degree',       30.0, ...
                      'subtract_means',         false, ...
                      'radial_mode',            'new', ...
                      'stats_return_type',    'output', ...
                      'label',                'Patch XX');
                  
    % Check for input parms name /value pairs              
    for i =1:2:length(varargin)-1
        parm_name = varargin{i};
        if isfield(ip_parms, parm_name);
            ip_parms.(parm_name) = varargin{i+1};
        else
            error('%s(): unknown parameter name "%s");', mfilename(), parm_name); 
        end
    end            

    % Check if the input is a file names. If so read in the data
    if ischar(ip_patch) && exist(ip_patch, 'file')
        [ip_root, ip_base, ip_ext] = fileparts(ip_patch);
        fprintf('Reading "%s"\n', ip_patch);
        switch(lower(ip_ext))
            case {'.pgm','.ppm'}
                [ip_patch, ip_parms.nominal_white] = pnm_reader(ip_patch, 'big_endian', ip_parms.big_endian);
            otherwise
                ip_patch = imread(ip_patch);
        end
    end
    
    % Catch icclab case for calibration images
    switch(lower(ip_parms.image_type))
        case {'icclab'}
            % This is option is required to process the Aptina LAB images
            % fprintf('%s(): Detected image type "%s", perfroming LAB->XYZ(D50)->XYZ(E) Conversion\n', mfilename, ip_parms.image_type);
            % Convert from 16-bit ICCLAB encoding to float (Aptina specific encoding)
            images.LAB = aptina_icclab_to_float(double(ip_patch));
            % Transform in XYZ using D50 white point and then adapt
            % to Illuminant E
            images.XYZ_D50 = cielab_to_xyz(    images.LAB,     get_white_point('ICC'));
            images.XYZ_E   = matrix_transform( images.XYZ_D50, get_matrix('XYZ_D50toXYZ_E'));
            % Calculate stats
            ip_parms.nominal_white = 255.0;
        otherwise
            % Save input sRGB image
            %figure('Name', ip_parms.label), imshow(uint8(ip_patch));
            images.RGB    = double(ip_patch);
            % Linearise inpyt RGB data with viewer observed black point
            % correction
            images.linRGB = srgb_linearisation( images.RGB, ...
                                                'nominal_white', ip_parms.nominal_white, ...
                                                'viewer_observed_black_point_correction', true );
            % Transform into ACC oponent colour space
            images.XYZ_E  = matrix_transform(   images.linRGB, get_matrix('RGBtoXYZ_E'));
    end

    % Convert to AC1C2 oponent color space
    images.AC1C2    = matrix_transform( images.XYZ_E,  get_matrix('XYZ_EtoAC1C2'));

    % Convert viewing conditions into pixels_per_degree
    pixels_per_degree = ip_parms.cycles_per_degree * 2.0;
    % fprintf('%s - Pixels/degree: %6.2f (Cycles per degree = %6.3f)\n',...
    %                               ip_parms.label, ...
    %                               pixels_per_degree, ...
    %                               ip_parms.cycles_per_degree)  

    % Apply Frequency Based Spatial filtering
    images.SAC1C2 = iso_spatial_filtering(images.AC1C2, ...
                                          'csf_type',          ip_parms.csf_type, ...
                                          'subtract_means',    ip_parms.subtract_means, ...
                                          'pixels_per_degree', pixels_per_degree, ...
                                          'radial_mode',       ip_parms.radial_mode);
    
    % Transform into sRGB colour space
    images.SXYZ_E   = matrix_transform(images.SAC1C2,   get_matrix('AC1C2toXYZ_E'));   
    images.SXYZ_D65 = matrix_transform(images.SXYZ_E,   get_matrix('XYZ_EtoXYZ_D65'));

    % Clip XYZ to ensure no negative XYZ values
    clip_mask = images.SXYZ_D65 < 0;
    % clipped_pixel_count = sum(sum(sum(clip_mask)));
    % fprintf('%s(): %s - SXYZ(D65) - %4d clipped pixels\n', mfilename, ip_parms.label, clipped_pixel_count);
    images.SXYZ_D65( clip_mask ) = 0;

    % Linear RGB to RGB
    %images.linSRGB  = matrix_transform( images.SXYZ_D65, get_matrix('XYZ_D65toRGB'));
    %images.SRGB = srgb_gamma(images.linSRGB);
    %images.SRGB = images.SRGB * double(ip_parms.nominal_white);

    % XYZ(D65) to CIELAB and CIELUV
    images.SLUV = xyz_to_cieluv(images.SXYZ_D65, get_white_point('D65', 'ISO_VN'));

    % Calculate Image statistics
    if strcmpi(ip_parms.image_type, 'icclab')
        image_labels    = {'LAB', 'XYZ_D50', 'XYZ_E', 'AC1C2', 'SAC1C2', 'SXYZ_E', 'SXYZ_D65', 'SLUV'};
    else
        image_labels    = {'RGB', 'linRGB',  'XYZ_E', 'AC1C2', 'SAC1C2', 'SXYZ_E', 'SXYZ_D65', 'SLUV'};
    end
    
    image_count  = length(image_labels);
    db_labels    = {}; db_data = [];
    cv_labels    = {}; cv_data = [];
    vn_labels    = {}; vn_data = [];
    
    for i=1:image_count
        
        % Generate stats
        image_label = image_labels{i};
        [tmp_data, tmp_labels] = image_stats(images.(image_label), image_label);
        
        % Accumulate internal debug stats
        db_data   = cat(2, db_data,   tmp_data);
        db_labels = cat(2, db_labels, tmp_labels);
        
        % If SLUV stats then calculate covariances and objective visual noise value
        if strcmpi(image_label, 'SLUV')
            % Calculate covariances
            [cv_data, cv_labels] = covariance_stats(images.(image_label), image_label);
            % Generate ISO Visual Noise
            [vn_data, vn_labels] = iso_visual_stats(tmp_data);
        end
    end
    
    % Select which set of data to return
    switch(lower(ip_parms.stats_return_type))
        case {'debug'}
            op_data   = db_data;
            op_labels = db_labels;
        case {'covar'}
            op_data   = cv_data;
            op_labels = cv_labels;
        otherwise
            op_data   = vn_data;
            op_labels = vn_labels;
    end

end
