% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [op_data, op_labels] = iso_visual_stats(luv_data, varargin)
%ISO_VISUAL_STATS  Calculate ISO Visual noise value from the input LUV stats
%  [op_data, op_labels] = ISO_VISUAL_STATS(luv_stats)
%  [op_data, op_labels] = ISO_VISUAL_STATS(luv_stats, ParmName, ParmValue, ...)
%
% Description:
% Calculates the final ISO VN objective metric value from the input
% CIELUV stats data structure e.g.
%
% luv_stats.data  = [l_mean, u_mean, v_mean, l_std, u_std, v_std]
%
% Returns array of objective stats values and cell array of column labels e.g.
%
% op_labels = {'ISO.VN.lightness', 'ISO.VN.objective'}
% op_data   = [lightness, iso_vn_objective]
%
% Parameters:
% l_noise_coeff
%   L* noise weight, Default: 1.000
%
% u_noise_coeff
%   u* noise weight, Default: 0.852
%
% v_noise_coeff
%   v* noise weight, Default: 0.323
%
% References:
% 1. ISO 15739 WD#3 Oct 2010 Photography - Electronic still picture imaging - Noise measurements
%
% See also CPIQ_VISUAL_STATS, IHIF

    if nargin < 1
       error('%s(): 1 arguments required, %d supplied\nUsage: op_array = %s(luv_stats);', mfilename(), nargin, mfilename()); 
    end
 
     % Define default values for the option parameters
    ip_parms = struct( 'l_noise_coeff', 1.000,  ...
                       'u_noise_coeff', 0.852, ...
                       'v_noise_coeff', 0.323);
                  
    % Check for input parms name /value pairs              
    for i =1:2:length(varargin)-1
        parm_name = varargin{i};
        if isfield(ip_parms, parm_name);
            ip_parms.(parm_name) = varargin{i+1};
        else
            error('%s(): unknown parameter name "%s");', mfilename(), parm_name); 
        end
    end 

    % Calculate ISO Visual Noise Objective Value
    % visual_noise = noise(L*) + 0.852*noise(u*) + 0.323*noise(v*)
    noise    = ip_parms.l_noise_coeff * luv_data(4) +  ...
               ip_parms.u_noise_coeff * luv_data(5) +  ...
               ip_parms.v_noise_coeff * luv_data(6);
    
    op_labels = {'ISO.VN.lightness', 'ISO.VN.objective'};
    op_data   = [luv_data(1), noise];

end

