% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [op_array]=matrix_transform(ip_array, transform_matrix, varargin)
% MATRIX_TRANSFORM Applies the 3x3 the color transform matrix to supplied
% input image
%  [op_array] = COLOR_TRANSFORM(ip_array, conversion_matrix)
%  [op_array] = COLOR_TRANSFORM(ip_array, conversion_matrix, ParmName, ParmValue ...)
%
% Required inputs:
% * ip_array
%     3-D arrays with 3-channels e.g. RGB, XYX, YC1C2
% * transform_matrix
%    input 3x3 transform matrix
%
% Returned values:
% * op_array
%     returned transformed output
%
% Parameters:
% clip
%   If true clips the output array to the specified
%   [nominal_black, nominal_white] range
%   Default:  false
%
% nominal_black
%   Value representing black
%   Default:  0.0
%
% nominal_white
%   Value representing white
%   Default: 1.0
%
% Assumed but not checked that the number of input planes (e.g. 3)
% matches the size of the color transform matrix (e.g. 3x3)
%
% See also GET_TRANSFORM_MATRIX

    if nargin < 2
       error('%s(): 2 arguments required, %d supplied\nUsage: op_array = %s(ip_array, transform_matrix, ParmName, ParmValue ...);', mfilename(), nargin, mfilename()); 
    end

    % Define default values for the option parameters
    ip_parms = struct( 'clip',           false, ...
                       'nominal_black',  0.0, ...
                       'nominal_white',  1.0);
                  
    % Check for input parms name /value pairs              
    for i =1:2:length(varargin)-1
        parm_name = varargin{i};
        if isfield(ip_parms, parm_name);
            ip_parms.(parm_name) = varargin{i+1};
        else
            error('%s(): unknown parameter name "%s");', mfilename(), parm_name); 
        end
    end
    
    % Extract the size of the input image
    [ip_height, ip_width, ip_depth] = size(ip_array);
    
    % Reshape 3-D array into a 2-D array to allow
    % matrix multiplication
    ip_array  = reshape(ip_array, ip_height*ip_width, ip_depth);
    
    % Apply the conversion matrix - note the matrix is transposed
    op_array = double(ip_array) * double(transform_matrix)'; 
    
    % Reshape 2-D array back into a 3-D array
    op_array = reshape(op_array, ip_height, ip_width, ip_depth);
    
    % Optional clip the data
    if ip_parms.clip
        op_array = max(op_array, ip_parms.nominal_black);
        op_array = min(op_array, ip_parms.nominal_white);        
    end
    
end

