% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [op_array] = noise_power_spectrum(ip_array, varargin)
%NOISE_POWER_SPECTRUM - Returns 2D Noise Power Spectrum for the input image
% op_array = NOISE_POWER_SPECTRUM(ip_array)
% op_array = NOISE_POWER_SPECTRUM(ip_array, ParmName, ParmValue, ...)
%
% Description:
% Generates a 2-D Noise Power Spectrum for the input image with optional
% cross spectra data.
%
% Parameters:
% block_size
%   Block size for NPS Averaging e.g 32, 64, 128, 256.
%   Must be less than or equal to the patch size
%   Default = 64
%
% mean_type
%   Selects between local (per block), global (whole image) mean substraction
%   Set to 'none' to disable.
%   Default: 'local'
%   Options: 'local', 'global', 'none'
%
% cross_spectra
%   If True then generate the cross spectra
%   Default: false
    
    if nargin < 1
       error('%s(): min of 1 argument required, %d supplied\nUsage: [op_array] = %s(ip_array, ParmName, ParmValue, ...);', mfilename(), nargin, mfilename()); 
    end

    % Define default values for the option parameters
    ip_parms = struct('block_size',     64, ...
                      'mean_type',     'local', ...
                      'cross_spectra',  false);
                  
    % Check for input parms name /value pairs              
    for i =1:2:length(varargin)-1
        parm_name = varargin{i};
        if isfield(ip_parms, parm_name);
            ip_parms.(parm_name) = varargin{i+1};
        else
            error('%s(): unknown parameter name "%s");', mfilename(), parm_name); 
        end
    end        
                              
    % Get size of input image
    [height, width, planes] = size(ip_array);

    % Determine block grid size
    if ip_parms.block_size <= 0
       block_size_int = min(height,width );
    else
       block_size_int = ip_parms.block_size;               
    end

    block_x_count = floor(height / block_size_int);
    block_y_count = floor(width  / block_size_int);
    block_count   = block_x_count * block_y_count;
    
    % Initialise output array
    if ip_parms.cross_spectra > 0
        op_array = zeros(block_size_int, block_size_int, planes+3);
    else
        op_array = zeros(block_size_int, block_size_int, planes);
    end

    % Calculate Global means
    global_means = zeros(planes,1);
    for pi = 1:planes
        global_means(pi) = mean(mean(ip_array(:,:,pi)));
    end

    % Loop through each block
    fft2d = zeros(block_size_int, block_size_int, planes);
    
    for y_index=0:(block_y_count-1)
        for x_index=0:(block_x_count-1)
            
            % Calculate offset for block
            x = (x_index * block_size_int)+1;
            y = (y_index * block_size_int)+1;
         
            % Loop through each plane
            for index=1:planes      

                % Extract block
                block_data = double(ip_array(y:y + block_size_int - 1, x:x + block_size_int - 1, index));

                % Subtract either patch, block or none
                switch(lower(ip_parms.mean_type))
                    case 'local'
                        block_data = block_data - mean(mean(block_data));
                    case 'global'
                        block_data = block_data - global_means(index);
                    case {'none'}
                    otherwise
                        error( [mfilename,':UnknownMeanType'],...
                               'Unknown Mean Type: "%s", Valid types: "local", "global", "none"', ip_parms.mean_type );
                end

                % Compute 2D fft and shift results so that the
                % DC point is in the centre
                fft2d(:,:,index) = fftshift(fft2(block_data));

                % Accumulate noise power spectrum result
                op_array(:,:,index) = op_array(:,:,index) + abs(fft2d(:,:,index).^2)/(block_size_int^2);
            end

            if ip_parms.cross_spectra > 0
                % Generate cross spectral data
                index = planes+1;
                for channel1=1:planes-1
                    for channel2=channel1+1:planes;
                        op_array(:,:,index) = op_array(:,:,index) + (real(fft2d(:,:,channel1)).*real(fft2d(:,:,channel2)))/(block_size_int^2);
                        index = index+1;
                    end
                end
            end
            
        end
    end
    
    % Average over the number of blocks
    op_array = op_array / block_count;
  
end

