% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [img_data, max_code] = pnm_reader(ip_file, varargin)
%PNM_READER Readers .pgm or .ppm files
% [op_array, max_code] = PNM_READER(ip_file)
% [op_array, max_code] = PNM_READER(ip_file, ParmName, ParmValue, ...)
%
% Description:
% Reader function for pgm/ppm/pfm files with support for
% 8 and 16-bit pgm/ppm image data 
% 
% Table of File descriptor types:
% 
% FILE DESCRIPTOR TYPE                        ENCODING  EXTENSION
%      P1         Portable bitmap             ASCII     .pbm
%      P2         Portable graymap  (Mono)    ASCII     .pgm
%      P3         Portable pixmap   (Color)   ASCII     .ppm
%      P4         Portable bitmap             Binary    .pbm
%      P5         Portable graymap  (Mono)    Binary    .pgm
%      P6         Portable pixmap   (Color)   Binary    .ppm
%      Pf         Portable Floatmap (Color)   Binary    .pfm
%      PF         Portable Floatmap (Mono)    Binary    .pfm
% 
% TYPE CODE      VALUE TYPE
%     0          Integer
%     1         Float/Double
%     2          Boolean
%     3          String
%     4         RAW Data
% 
% Parameters:
%
% big_endian
%   Default: true
%
% References:
% 
% http://en.wikipedia.org/wiki/Netpbm_format
% http://www.fileformat.info/format/pbm/egff.htm
% http://netpbm.sourceforge.net/doc/pfm.html
% http://gl.ict.usc.edu/HDRShop/PFM/PFM_Image_File_Format.html
% 
% Limitations:
% 
% Portable pixel maps are NOT supported
% Portable float maps are NOT verified
%
% See also PNM_WRITER

    if nargin < 1
       error('%s(): min of 1 argument required, %d supplied\nUsage: op_array = %s(ip_array);', mfilename(), nargin, mfilename()); 
    end
    
    % Define default values for the option parameters
    ip_parms = struct('big_endian',  true);
                  
    % Check for input parms name /value pairs              
    for i =1:2:length(varargin)-1
        parm_name = varargin{i};
        if isfield(ip_parms, parm_name);
            ip_parms.(parm_name) = varargin{i+1};
        else
            error('%s(): unknown parameter name "%s");', mfilename(), parm_name); 
        end
    end
            
    % Attempt to open PNM File
    fid = fopen(ip_file, 'rb' );

    if fid > -1
                           
        % Read file header
        [file_type, height, width, max_code] = read_pnm_metadata(fid);

        % Set number of planes
        [data_type, separator, image_planes] =  get_file_info(file_type, max_code);

        % Endianess
        file_byteorder = 'ieee-le'; % Little Endian
        if ip_parms.big_endian
            file_byteorder = 'ieee-be'; % Big Endian
        end

        % For PF, Pf files the sign of the maximum code indicates the endianess
        % of the data
        % max_code <= 0 = little endian
        % max_code >  0 = big endian
        if strcmpi(file_type, 'PF')
            if max_code > 0
                file_byteorder = 'ieee-be';
            else
                file_byteorder = 'ieee-le';
            end
        end                

        % Read in data values
        if isempty(separator)
            img_data = fread(fid, data_type, file_byteorder);
        else
            img_data = fscanf( fid, '%d', inf );
        end

        % Close file
        fclose(fid);

        % Force cast back to correct data type
        switch data_type
            case 'uint8'
                img_data = uint8(img_data);
            case 'uint16'
                img_data = uint16(img_data);
            case 'uint32'
                img_data = uint32(img_data);
            case {'single', 'float32'}
                img_data = single(img_data);
        end

        % Re-scale image data using the max code in float data
        if strcmpi(file_type, 'PF')
            img_data = img_data * abs(max_code);
        end

        % Transform image
        img_data = reshape(img_data, width*image_planes, height)';

        % Split into 3 colour planes if required
        if image_planes == 3
            img_data = cat( 3, img_data(:,1:3:end), img_data(:,2:3:end), img_data(:,3:3:end) );
            % note the short below does not work in matlab
            % obj.data = reshape(obj.data, obj.image_height, obj.image_width, obj.image_planes );
        end

    else
        error( [mfilename,':Load:FailedToOpenFile'],...
               'Failed to open File "%s"', ip_file_name );
        img_data = [];
        max_code = -1;
    end 
            
end

function [file_type, height, width, max_code] = read_pnm_metadata(fid)
%READ_PNM_METADATA - Reads in pgm file information header
%  [file_type, height, width, max_code] = READ_PNM_METADATA(fid)
%
% Example PNM Header
%
% P5
% #parsing details,3,loaded from 'nsp_impulse_test_24x24x8.raw' as byte-aligned binary data (24 x 24, Bayer data, 8 bits/value, 1 bytes/value)
% 24 24
% 255 

    % Initialise local variables
    prev_data_flag = false;
    comment_flag   = false;
    header_text    = '';
    index          = 0;

    % First read in the 2-byte file type value as a string
    file_type = char( reshape( fread( fid, 2, 'uint8' ), 1, 2 ) );

    % How read the rest of the file header one character at a time until
    % 3 uncommented integer values have been read splitting the header data
    % into header text and commented text (may contain meta data)
    while( index < 3 )

        % Read in a character 
        byte = fread( fid, 1, 'uint8' );

        % If '#' Comment Character then set appropriate flags
        % A comment remains valid until the end of the current line
        % (10=NL or 13=CR)
        if byte == uint8('#')  % # Comment Character
            comment_flag  = true;
        elseif byte == 10 || byte == 13
            comment_flag = false;
        end

        % If an uncommented decimal number mark as valid
        % Otherwise data false i.e. white space out with a comment
        if ~comment_flag && ((byte >= uint8('0') && byte <= uint8('9')) || byte == uint8('.') || byte == uint8('-') )
            data_flag = true;
        else
            data_flag = false;
        end

        % If previous character was a number and current character is
        % not then end of on one of the integer's in the pgm header
        % Also ensures that the white space after the final integer
        % is read (important)
       if prev_data_flag && ~data_flag
           index = index + 1;
       end

       % Accumulate comment and header text strings
       if ~comment_flag
           header_text  = [header_text,char(byte)];
       end    

       prev_data_flag = data_flag;               
    end

    % Extract width, height and max code from header text
    values = regexpi( header_text, '([0-9\.\-]+)', 'match' );
    if ~isempty(values)
        width     = int32(str2double(values(1)));
        height    = int32(str2double(values(2)));
        max_code  =       str2double(values(3)); 
    end

end

function [data_type, separator, planes] = get_file_info(file_type, max_code)
%GET_FILE_INFO - return file information for PNM file type
% [data_type, separator, plane] = GET_FILE_INFO(file_type, max_code)
%
% Initialise input file type mappings
% Grey:  planes    = 1,   Color:  planes    = 3
% ASCII: separator = ' ', Binary: separator = '' (empty)
    
    file_info_lut = struct( 'P2',  struct(  'data_type', 'uint8',...
                                            'separator', ' ',...
                                            'planes',    1),...
                            'P3',  struct(  'data_type', 'uint8',...                                   
                                            'separator', ' ',...
                                            'planes',    3),...
                            'P5',  struct(  'data_type', 'uint8', ...                                   
                                            'separator', '',...
                                            'planes',    1),...
                            'P6',  struct(  'data_type', 'uint8', ...                                   
                                            'separator', '',...
                                            'planes',    3),...
                            'Pf',  struct(  'data_type', 'float32',...                                    
                                            'separator', '',...
                                            'planes',    1),...
                            'PF',  struct(  'data_type', 'float32',...                                    
                                            'separator', '',...
                                            'planes',    3));
                                            
    if isfield(file_info_lut, file_type)
        
        % Get data tyoes
    	data_type = file_info_lut.(file_type).data_type;
    	separator = file_info_lut.(file_type).separator;
    	planes    = file_info_lut.(file_type).planes;
        
         % Select integer type based on the maximum code
        if ~strcmpi(file_type, 'PF' )
            if max_code > 65535
                data_type = 'uint32';
            elseif max_code > 255
                data_type = 'uint16';
            else
                data_type = 'uint8';
            end
        end
        
    else
        valid_transforms = fieldnames(file_info_lut);
        error('Unknown PNM File Type "%s"!\nValid PNM File Types:\n%s', file_type, sprintf('"%s"\n', valid_transforms{:}));
    end
end


