% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [status] = pnm_writer(op_file, ip_array, max_code, varargin)
%PNM_WRITER Writes .pgm or .ppm files
% [status] = PNM_WRITER(op_file, ip_array, max_code)
% [status] = PNM_WRITER(op_file, ip_array, max_code, ParmName, ParmValue, ...)
%
% Description:
% Writer function for pgm/ppm/pfm files with support for
% 8 and 16-bit pgm/ppm image data 
% 
% Table of File descriptor types:
% 
% FILE DESCRIPTOR TYPE                        ENCODING  EXTENSION
%      P1         Portable bitmap             ASCII     .pbm
%      P2         Portable graymap  (Mono)    ASCII     .pgm
%      P3         Portable pixmap   (Color)   ASCII     .ppm
%      P4         Portable bitmap             Binary    .pbm
%      P5         Portable graymap  (Mono)    Binary    .pgm
%      P6         Portable pixmap   (Color)   Binary    .ppm
%      Pf         Portable Floatmap (Color)   Binary    .pfm
%      PF         Portable Floatmap (Mono)    Binary    .pfm
% 
% TYPE CODE      VALUE TYPE
%     0          Integer
%     1          Float/Double
%     2          Boolean
%     3          String
%     4          RAW Data
% 
% Parameters:
%
% big_endian
%   If true then binary PNM data is saved as big_endian
%   Has no effect if an ASCII PNM file output is selected
%   Default: true
%
% binary
%   If true data saved as binary PNM file
%   Default: true
%
% References:
% 
% http://en.wikipedia.org/wiki/Netpbm_format
% http://www.fileformat.info/format/pbm/egff.htm
% http://netpbm.sourceforge.net/doc/pfm.html
% http://gl.ict.usc.edu/HDRShop/PFM/PFM_Image_File_Format.html
% 
% Limitations:
% 
% Portable pixel maps are NOT supported
% Portable float maps are NOT verified
%
% See also PNM_READER

    if nargin < 3
       error('%s(): min of 3 arguments required, %d supplied\nUsage: status = %s(op_file, ip_array, max_code);', mfilename(), nargin, mfilename()); 
    end
    
    % Define default values for the option parameters
    ip_parms = struct('big_endian',  true, ...
                      'binary',      true);
                  
    % Check for input parms name /value pairs              
    for i =1:2:length(varargin)-1
        parm_name = varargin{i};
        if isfield(ip_parms, parm_name);
            ip_parms.(parm_name) = varargin{i+1};
        else
            error('%s(): unknown parameter name "%s");', mfilename(), parm_name); 
        end
    end
            
    % Determine Image width and height
    [height, width, no_of_planes] = size(ip_array);
            
    % Attempt to open Binary Pnm Image File
    fid = fopen(op_file, 'wb');

    if fid > -1

        % Look up file type
        data_type = class(ip_array);
        file_type = get_file_info(data_type, no_of_planes, ip_parms.binary);

        % Foc color planes data merge planes Transform image
        if no_of_planes == 3
            % Merge Planes
            op_array = zeros(height, width*no_of_planes, data_type);
            op_array(:,1:3:end) = ip_array(:,:,1);
            op_array(:,2:3:end) = ip_array(:,:,2);
            op_array(:,3:3:end) = ip_array(:,:,3);
        else
            op_array = ip_array;
        end               

        % If float then sign of max code denotes endian
        % max_code <= 0 = little endian
        % max_code >  0 = big endian           
        if strcmpi(file_type, 'PF')
            if ip_parms.big_endian
                max_code =  1.0;
            else
                max_code = -1.0;
            end
        end

        % Print Output PNM File header
        fprintf(fid, '%s\n',    file_type );
        fprintf(fid, '%d %d\n', width, height);
        if strcmpi(file_type, 'PF')
            fprintf(fid, '%.6f\n', max_code);
        else
            fprintf(fid, '%.0f\n', max_code);
        end

        % Endianess
        if ip_parms.big_endian
            file_byteorder = 'ieee-be'; % Big Endian
        else
            file_byteorder = 'ieee-le'; % Little Endian
        end

        % Transpose output data
        op_array = op_array';

        % Write in binary data
        if ip_parms.binary
            fwrite(fid, op_array, data_type, file_byteorder);
        else
            fprintf(fid, '%d\n', output_data);
        end

        % Close output file
        fclose(fid);

    else
        error( [mfilename,':Save:FailedToOpenFile'],...
               'Failed to open File "%s"', ip_file_name );
        status = -1;
    end 
            
end


function [file_type] = get_file_info(data_type, no_of_planes, binary)
%GET_FILE_INFO - return file information for PNM file type
%  [file_type] = GET_FILE_INFO(data_type, no_of_planes, binary)
    
    file_info_lut = struct('uint8',  struct( 'planes1', struct( 'binary0', 'P2',...
                                                                'binary1', 'P5' ),...
                                             'planes3', struct( 'binary0', 'P3',...
                                                                'binary1', 'P6' ) ),...
                           'uint16', struct( 'planes1', struct( 'binary0', 'P2',...
                                                                'binary1', 'P5' ),...
                                             'planes3', struct( 'binary0', 'P3',...
                                                                'binary1', 'P6' ) ),...
                           'uint32', struct( 'planes1', struct( 'binary0', 'P2',...
                                                                'binary1', 'P5' ),...
                                             'planes3', struct( 'binary0', 'P3',...
                                                                'binary1', 'P6' ) ),...
                           'single', struct( 'planes1', struct( 'binary0', 'Pf',...
                                                                'binary1', 'Pf' ),...
                                             'planes3', struct( 'binary0', 'PF',...
                                                                'binary1', 'PF' ) ),...
                           'double', struct( 'planes1', struct( 'binary0', 'Pf',...
                                                                'binary1', 'Pf' ),...
                                             'planes3', struct( 'binary0', 'PF',...
                                                                'binary1', 'PF' ) ) );
                                            
    if isfield(file_info_lut, data_type)
        % Get file type
    	file_type = file_info_lut.(data_type).(['planes',num2str(no_of_planes)]).(['binary',num2str(binary)]);  
    else
        valid_transforms = fieldnames(file_info_lut);
        error('Unsupported PNM Data Type "%s"!\nValid PNM Data Types:\n%s', data_type, sprintf('"%s"\n', valid_transforms{:}));
    end
end


