% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [op_array] = polar_integration(ip_array)
%POLAR_INTEGRATION Returns the volume under the input NPS cross sections
% [op_array] = POLAR_INTEGRATION(ip_array)
%
% Description:
% Performs a polar integration on the input 1-D NPS cross section data
%
% The input data must be s 2-D array where the 1st column contains the
% NPS spatial frequencies in cycles per pixel (e.g max - 0.5 cyc/pix) and
% the remaining columns contain 1 or more cross sections.
%
% See also APTINA_VISUAL_NOISE_METRIC

    data_count = size(ip_array, 2)-1;
    op_array   = zeros( 1, data_count);

    % Calculate the are circles the x values as radius
    circle_area = pi() .* (ip_array(:,1).^2);
    
    % Calculate the difference between the circle areas to 
    % give the radial area for each X increment
    radial_area = abs(diff(circle_area));
        
    for index=1:data_count

        % Calculate the mean y_value for each on the above
        % x deltas (assuming linear interpolation)
        y_means  = (ip_array(1:end-1,index+1) + ip_array(2:end,index+1) )/2.0;

        % Multiples the x deltas by the y means to get the area
        % for each part of the data and then sum to get the total
        % area
        op_array(index) = sum(radial_area .* y_means);
        
    end
            
end