% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [op_array] = srgb_gamma(ip_array, varargin)
%SRGB_GAMMA Applies sRGB or Simple gamma to input array
% op_array = SRGB_LINEARISATION(ip_array)
% op_array = SRGB_LINEARISATION(ip_array, ParmName, ParmValue, ...)
%
% Required input(s):
% * ip_array
%     input array containing planar image data
%
% Returned value(s):
% * op_array
%     output data with Gamma curve applied. In the [0.0 - 1.0] data range.
%
% Optional input parameters:
%
% * 'true_srgb'
%     If true SRGB is applied, if false simple gamma is applied
%     Default = true
%
% * 'simple_gamma'
%     Value for simple gamma
%     Default = 2.2
%
% Desciption:
% Convert input linear sRGB image into sRGB image by applying
% the sRGB transform function
%
% Assumes the input sRGB data is in the 0-1 range
%
% Please note in the references below there is disagreement in the
% threshold at which the linear part of the sRGB response starts
%
% References:
%  1. IEC 61966-2-1
%  2. http://en.wikipedia.org/wiki/SRGB  => 0.0031308
%  3. http://www.w3.org/Graphics/Color/sRGB.html => 0.00304
%  4. Charles Poynton, "Digital Video and HDTV", pp .267 => 0.0031308
%
% See also SRGB_LINEARISATION

    if nargin < 1
       error('%s(): min of 1 argument required, %d supplied\nUsage: op_coords = %s(ip_array, ParmName, ParmValue, ...);', mfilename(), nargin, mfilename()); 
    end

    % Define default values for the option parameters
    ip_parms = struct('true_srgb',     true, ...
                      'simple_gamma',  2.20);
                  
    % Check for input parms name /value pairs              
    for i =1:2:length(varargin)-1
        parm_name = varargin{i};
        if isfield(ip_parms, parm_name);
            ip_parms.(parm_name) = varargin{i+1};
        else
            error('%s(): unknown parameter name "%s");', mfilename(), parm_name); 
        end
    end
    
    % Apply SRGB or Simple Gamma curves
    if ip_parms.true_srgb == true
        op_array =  (1.055 .* (ip_array.^(1/2.4))) - 0.055;
        op_array( ip_array <= 0.00304 ) = 12.92 .* ip_array( ip_array <= 0.00304 );
    else
        op_array = ip_array .^ (1/ip_params.simple_gamma);
    end

end