% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [op_array] = srgb_linearisation(ip_array, varargin)
%SRGB_LINEARISATION Applies the inverse sRGB Gamma transform. 
% op_array = SRGB_LINEARISATION(ip_array)
% op_array = SRGB_LINEARISATION(ip_array, ParmName, ParmValue, ...)
%
% Required input(s):
% * ip_array
%     input array containing planar image data
%
% Returned value(s):
% * op_array
%     Linearised output data. In the [0.0 - 1.0] data range.
%
% Optional input parameters:
% * 'nominal_white'
%     Nominal white code value e.g. 255.0 used to normalise the input data
%     into [0.0-1.0] data range before applying the linearisation step
%     Default = 255.0
%
% * 'viewer_observed_black_point_correction'
%     Include viewer observed black level correction
%     Default = false
%
% * 'clip'
%     Clip output data to [0.0-1.0] range
%     Default = false
%
% Description:
%
%
% References:
%
% 1. IEC 61966-2-1
% 2. http://en.wikipedia.org/wiki/SRGB => 0.04045
% 3. http://www.w3.org/Graphics/Color/sRGB.html => 0.03928
% 4. Charles Poynton, "Digital Video and HDTV", pp .268 => 0.03928
% 5. ISO 15739 Working Draft  2010-03-10
%
% See also SRGB_GAMMA

    if nargin < 1
       error('%s(): min of 1 argument required, %d supplied\nUsage: op_coords = %s(ip_array, ParmName, ParmValue, ...);', mfilename(), nargin, mfilename()); 
    end

    % Define default values for the option parameters
    ip_parms = struct('nominal_white',                           255.0, ...
                      'viewer_observed_black_point_correction',  false, ...
                      'clip',                                    false);
                  
    % Check for input parms name /value pairs              
    for i =1:2:length(varargin)-1
        parm_name = varargin{i};
        if isfield(ip_parms, parm_name);
            ip_parms.(parm_name) = varargin{i+1};
        else
            error('%s(): unknown parameter name "%s");', mfilename(), parm_name); 
        end
    end

    % Norminalise input to [0.0-1.0] range
    ip_array = double(ip_array) ./ double(ip_parms.nominal_white);

    if ip_parms.viewer_observed_black_point_correction == true
        % Linearise the RGB data according to IEC 61966-2-1
        % with viewer observed black level correction
        op_array =   0.0125 + 0.868423 .* (( 0.055 + ip_array ).^2.4);
        op_array( ip_array <= 0.04045 ) = 0.0125 + 0.0764319 .* ip_array( ip_array <= 0.04045 );
    else
        % Otherwise apply the inverse of the sRGB gamma transform
        % Note that the threshold is different in different references
        % 1. http://en.wikipedia.org/wiki/SRGB = 0.04045
        % 2. http://www.w3.org/Graphics/Color/sRGB.html = 0.03928
        %    This is the same as Poynton quotes
        op_array =  ((0.055 + ip_array) ./ 1.055).^2.4;
        op_array(ip_array <= 0.03928) = ip_array(ip_array <= 0.03928) ./ 12.92;
    end
    
    % Conditional clip the output
    if ip_parms.clip == true
        op_array(op_array < 0.0) = 0.0;
        op_array(op_array > 1.0) = 1.0;        
    end
   
end

