% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [text_string] = table_as_str(ip_array, col_labels, row_labels, varargin)
%TABLE_AS_STR - writes 2-D data including column and row labels to text string
%   [text_string] = TABLE_AS_STR(ip_array, col_labels, row_labels)
%   [text_string] = TABLE_AS_STR(ip_array, col_labels, row_labels, ParmName, ParmValue, ...)
%
% Description:
% Pretty printing for 2-D arrays of data with row and column labels
%
% Parameters:
% delimiter
%   Character to separate array elements
%   Default = ','
%
% precision
%   Number of decimal places to output
%   Default = 4
%
% col_width
%   Column width in characters excluding the delimiter character
%   Default = 8
%
% See also TABLE_WRITER

    if nargin < 3
       error('%s(): min of 3 arguments required, %d supplied\nUsage: text_string] = %s(ip_array, col_labels, row_labels, ParmName, ParmValue, ...);', mfilename(), nargin, mfilename()); 
    end

    % Define default values for the option parameters
    ip_parms = struct('delimiter', ',', ...
                      'precision', 4, ...
                      'col_width', 8);
                  
    % Check for input parms name /value pairs              
    for i =1:2:length(varargin)-1
        parm_name = varargin{i};
        if isfield(ip_parms, parm_name);
            ip_parms.(parm_name) = varargin{i+1};
        else
            error('%s(): unknown parameter name "%s");', mfilename(), parm_name); 
        end
    end 

    % Generate pretty printing formats
    label_format = sprintf('%%%ds',    ip_parms.col_width);
    data_format  = sprintf('%%%d.%df', ip_parms.col_width, ip_parms.precision);
    
    % Build Column Headers
    text_string = sprintf([label_format,'%s\n'], ' ', sprintf([ip_parms.delimiter ,label_format], col_labels{:}));

    % Build table text string
    for ri = 1:size(ip_array, 1)
        text_string = [text_string, sprintf([label_format,'%s\n'], row_labels{ri}, sprintf([ip_parms.delimiter, data_format], ip_array(ri,:)))];
    end    

end

