% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [] = table_writer(op_file, ip_array, col_labels, row_labels, varargin)
%TABLE_WRITER - writes 2-D data including column and row labels to text string
%   TABLE_WRITER(op_file, ip_array, col_labels, row_labels)
%   TABLE_WRITER(op_file, ip_array, col_labels, row_labels, ParmName, ParmValue, ...)
%
% Description:
% Pretty printing for 2-D arrays of data with row and column labels
%
% Parameters:
% delimiter
%   Character to separate array elements
%   Default = ','
%
% precision
%   Number of decimal places to output
%   Default = 4
%
% col_width
%   Column width in characters excluding the delimiter character
%   Default = 8
%
% See also TABLE_AS_STR

    if nargin < 4
       error('%s(): min of 4 arguments required, %d supplied\nUsage: %s(op_file, ip_array, col_labels, row_labels, ParmName, ParmValue, ...);', mfilename(), nargin, mfilename()); 
    end

    % Define default values for the option parameters
    ip_parms = struct('delimiter', ',', ...
                      'precision', 4, ...
                      'col_width', 8);
                  
    % Check for input parms name /value pairs              
    for i =1:2:length(varargin)-1
        parm_name = varargin{i};
        if isfield(ip_parms, parm_name);
            ip_parms.(parm_name) = varargin{i+1};
        else
            error('%s(): unknown parameter name "%s");', mfilename(), parm_name); 
        end
    end 
 
    % Attempt to open file for writing
    fid = fopen(op_file, 'wt');
    
    if fid > -1       
       % Get table as a pretty string
        text_string = table_as_str(ip_array, col_labels, row_labels, ...
                                   'delimiter', ip_parms.delimiter, ...
                                   'precision', ip_parms.precision, ...
                                   'col_width', ip_parms.col_width);
        % Now write string to file
        fprintf(fid, '%s', text_string);
    else
        error([mfilename,'::FailedToOpenFile'], 'Failed to open File "%s"', op_file );
    end
    
    % Close output file
    fclose(fid);

end

