% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [op_array] = xyz_to_cielab(ip_array, white_point)
%XYZ_TO_CIELAB XYX to CIELAB color transform
%    [op_array] = XYZ_TO_CIELAB(ip_array, white_point)
%
% Description:
% Function takes an input XYZ reference set (array is a row array of
% (Xn,Yn,Zn)) and input test set (array is a row array of (X,Y,Z)) and
% calculates the L*a*b* coefficients.
%
% Input reference set is the XYZ of the reference white point.
% Calculation is based on the following:
%
%   L* = 116*(Y/Yn)^(1/3) - 16        for Y/Yn >  (24/116)^3
%      = ((116/12)^3)*(Y/Yn)          for Y/Yn <= (24/116)^3
%   a* = 500.[f(X/Xn)-f(Y/Yn)]
%   b* = 200.[f(Y/Yn)-f(Z/Zn)]
%
% where
%
%   f(t) = t^(1/3) for t > (24/116)^3
%   f(t) = (841/108)t + 16/116, ie for t <= (24/116)^3
%
% Reference(s):
%  1. "Digital Video and HDTV Algorithms and Interfaces"
%     Charles Poynton, Chapter 21
%     - page 225,  Eq 21.3   
%     - page 228,  Eq 21.12
%  2. Errata "Digital Video and HDTV Algorithms and Interfaces"
%
% See also XYZ_TO_CIELUV, GET_WHITE_POINT

    if nargin < 2
       error('%s(): 2 arguments required, %d supplied\nUsage: op_array = %s(ip_array, white_point);', mfilename(), nargin, mfilename()); 
    end
    
    % Extract the size of the input image
    [ip_height, ip_width, ip_depth] = size(ip_array);

    % Reshape 3-D array into a 2-D array to allow
    % matrix multiplication
    ip_array  = reshape(ip_array,  ip_height*ip_width,  ip_depth ); 

    % Build normalisation matrix
    nm = [ 1.0/white_point(1) 0.0                0.0 ;...
           0.0                1.0/white_point(2) 0.0 ;...
           0.0                0.0                1.0/white_point(3) ];

    % Calculate f(X/Xn), f(Y/Yn), f(Z/Zn)
    % where
    % f(t) = t^(1/3) for t > (24/116)^3
    % f(t) = (841/108)t + 16/116, ie for t <= (24/116)^3
    normalised   = double(ip_array) * nm';
    fN           = normalised.^(1.0/3.0);
    fN_threshold = (24.0/116.0)^3;
    fN( normalised <= fN_threshold ) = (841.0/108.0)*(normalised(normalised <= fN_threshold))+(16.0/116.0);

    % Calculate L*
    % L* = 116*(Y/Yn)^(1/3) - 16        for Y/Yn >  (24/116)^3
    %    = ((116/12)^3)*(Y/Yn)          for Y/Yn <= (24/116)^3
    op_array = (116.0 * normalised.^(1.0/3.0)) - 16.0;
    op_array( normalised <= fN_threshold ) = ((116.0/12.0)^3) * normalised( normalised <= fN_threshold );

    % Copy L* to the right plane in the output data
    op_array(:,1) = op_array(:,2);
    % Calculate a*
    % a* = 500.[f(X/Xn)-f(Y/Yn)]
    op_array(:,2) = 500.0 * (fN(:,1) - fN(:,2));
    % Calculate b*
    % b* = 200.[f(Y/Yn)-f(Z/Zn)]
    op_array(:,3) = 200.0 * (fN(:,2) - fN(:,3));

    % Reshape 2-D array back into a 3-D array
    op_array = reshape(op_array, ip_height, ip_width, ip_depth);

end

