% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [op_array] = xyz_to_cieluv(ip_array, white_point)
%XYZ_TO_CIELUV - XYZ to CIE-LUV conversion
%  [op_array] = XYZ_TO_CIELUV(ip_array, white_point)
%   
% Description:
% Function takes an input XYZ reference set (array is a row array of
% (Xn,Yn,Zn)) and input test set (array is a row array of (X,Y,Z)) and
% calculates the L*u*v* coefficients.
%
% Input reference set is the XYZ of the reference white point.
% Calculation is based on the following:
%
%   L* = 116*(Y/Yn)^(1/3) - 16 for Y/Yn >  0.008856
%      = 903.3*(Y/Yn)          for Y/Yn <= 0.008856
%   u* = 13.L.[u-un]
%   v* = 13.L.[v-vn]
%
% Reference(s):
%  1. "Digital Video and HDTV Algorithms and Interfaces"
%      Charles Poynton, Chapter 21
%     - page 225,  Eq 21.3   
%     - page 226,  Eq 21.4
%
% See also XYZ_TO_CIELAB, GET_WHITE_POINT

    if nargin < 2
       error('%s(): 2 arguments required, %d supplied\nUsage: op_array = %s(ip_array, white_point);', mfilename(), nargin, mfilename()); 
    end
    
    % Extract the size of the input image
    [ip_height, ip_width, ip_depth] = size(ip_array);

    % Declare output array
    op_array = zeros(ip_height, ip_width, ip_depth);

    % Calculate L*
    % L* = 116*(Y/Yn)^(1/3) - 16 for Y/Yn >  0.008856
    %    = 903.3*(Y/Yn)          for Y/Yn <= 0.008856
    normalised = ip_array(:,:,2) ./ white_point(2);
    L = (116.0 * normalised.^(1.0/3.0)) - 16.0;
    L(normalised <= 0.008856) = 903.3 * normalised(normalised <= 0.008856);

    % Copy L* to the right plane in the output data
    op_array(:,:,1) = L;

    % Build u and v planes
    % Remember that the . is important to indicate element by
    % element multiplication
    % u = 4X / (X+15Y+3Z)
    op_array(:,:,2) = (4.0.*ip_array(:,:,1) ) ./ ( ip_array(:,:,1) + 15.0.*ip_array(:,:,2) + 3.0.*ip_array(:,:,3));
    % v = 9Y / (X+15Y+3Z)
    op_array(:,:,3) = (9.0.*ip_array(:,:,2) ) ./ ( ip_array(:,:,1) + 15.0.*ip_array(:,:,2) + 3.0.*ip_array(:,:,3));

    % Trap any divide by zero
    op_array(isnan(op_array)) = 0.0;

    % Convert white point
    uN  = (4.0 * white_point(1)) / ( white_point(1) + 15.0*white_point(2) + 3.0*white_point(3) );
    vN  = (9.0 * white_point(2)) / ( white_point(1) + 15.0*white_point(2) + 3.0*white_point(3) );

    % Calculate u* = 13L*(u-un)
    op_array(:,:,2) = 13.0.* op_array(:,:,1) .* (op_array(:,:,2) - uN);
    % Calculate v* = 13L(v-vn)
    op_array(:,:,3) = 13.0.* op_array(:,:,1) .* (op_array(:,:,3) - vN);

end

