% Copyright (c) 2016 STMicroelectronics International N.V.
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software
%    without specific prior written permission.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
% THE POSSIBILITY OF SUCH DAMAGE.

function [status] = test_pnm_reader()
    % Add path to source directory
    addpath('../src');
    % Run tests
    errors = zeros(1,16);
    % Greyscale
    errors(1)  = test_component('check_load_data', 'grey_ascii_8b.pgm',      true, [0, 127, 255; 192, 128, 64],               255.0,  0.000001);
    errors(2)  = test_component('check_load_data', 'grey_ascii_15b.pgm',     true, [0, 16383, 32767; 8192, 16384, 24576],    32767.0, 0.000001);
    errors(3)  = test_component('check_load_data', 'grey_ascii_16b.pgm',     true, [0, 32767, 65535; 16384,  32768,  49152], 65535.0, 0.000001);
    errors(4)  = test_component('check_load_data', 'grey_ascii_31b.pgm',     true, [0, 1073741823, 2147483647; 1610612736, 1073741824, 536870912],  2147483647.0, 0.000001);
    errors(5)  = test_component('check_load_data', 'grey_binary_8b.pgm',     true, [0, 127, 255; 192, 128, 64],               255.0,  0.000001);
    errors(6)  = test_component('check_load_data', 'grey_binary_15b.pgm',    true, [0, 16383, 32767; 8192, 16384, 24576],   32767.0,  0.000001);
    errors(7)  = test_component('check_load_data', 'grey_binary_31b.pgm',    true, [0, 1073741823, 2147483647; 1610612736, 1073741824, 536870912],  2147483647.0, 0.000001);
    errors(8)  = test_component('check_load_data', 'grey_big_endian.pfm',    true, [3.141592; 2.718281; 1.618033],  1.0, 0.000001);
    % Color
    errors(9)  = test_component('check_load_data', 'color_ascii_8b.ppm',     true, cat(3, [0, 192], [127, 128], [255, 64]),               255.0,  0.000001);
    errors(10) = test_component('check_load_data', 'color_ascii_15b.ppm',    true, cat(3, [0, 8192], [16383, 16384], [32767, 24576]),    32767.0, 0.000001);
    errors(11) = test_component('check_load_data', 'color_ascii_16b.ppm',    true, cat(3, [0, 16384], [32767, 32768], [65535, 49152]), 65535.0, 0.000001);
    errors(12) = test_component('check_load_data', 'color_ascii_31b.ppm',    true, cat(3, [0, 1610612736], [1073741823, 1073741824], [2147483647, 536870912]),  2147483647.0, 0.000001);
    errors(13) = test_component('check_load_data', 'color_binary_8b.ppm',    true, cat(3, [0, 192], [127, 128], [255, 64]),               255.0,  0.000001);
    errors(14) = test_component('check_load_data', 'color_binary_15b.ppm',   true, cat(3, [0, 8192], [16383, 16384], [ 32767, 24576]),   32767.0,  0.000001);
    errors(15) = test_component('check_load_data', 'color_binary_31b.ppm',   true, cat(3, [0, 1610612736], [1073741823, 1073741824], [2147483647, 536870912]),  2147483647.0, 0.000001);
    %errors(16) = test_component('check_load_data', 'color_little_endian.pfm', true, cat(3, 3.141592, 2.718281, 1.618033),  1.0, 0.000001);
    % Generate status
    status = sum(errors);
    if status > 0
        fprintf(2, 'ERROR! - Unit Tests Failed! - %d Errors\n', status);
    else
        fprintf(2, 'SUCCESS! - All Unit Tests OK!\n');
    end
end

function [status] = test_component(check_name, ip_file, big_endian, expected, expected_max_code, tolerance)
    % Run component
    full_file_name = fullfile('./data/pnm_reader', ip_file);
    [actual, actual_max_code] = pnm_reader(full_file_name, 'big_endian', big_endian);
    % Run checks
    errors = zeros(1,2);
    errors(1) = check_array([ip_file,':data'],     actual, expected, tolerance, '%.16f. '); 
    errors(2) = check_value([ip_file,':max_code'], actual_max_code,  expected_max_code, tolerance, '%.16f. ');
    status = sum(errors);
end

function [err_cnt] = check_array(parm_name, actual, expected, tolerance, display_format)
    % Check that the 2 inputs are within the required tolerance
    err_cnt = 0;
    err_msg = [parm_name,'\n'];
    % Compare line by line
    if sum(size(actual) == size(expected)) ~= length(size(expected))
        err_msg = sprintf('SIZE MIS-MATCH "%s" Actual: [%s], Expected: [%s]\n', parm_name, sprintf('%d ', size(actual)) , sprintf('%d ', size(expected)));
    else
        rows = size(expected, 1);
        for row = 1:rows
            delta = abs(double(actual(row,:)) - double(expected(row,:)));
            row_err_cnt = sum(delta > tolerance);
            if row_err_cnt > 0
                row_err_msg = sprintf('ERROR! #%02d: Expected: (%-28s), Actual: (%-28s), Delta: (%-28s)\n',...
                                       row,...
                                       sprintf(display_format, expected(row,:)),...
                                       sprintf(display_format, actual(row,:)),...
                                       sprintf(display_format, delta));
                err_msg = [err_msg, row_err_msg];
            end
            err_cnt = err_cnt + row_err_cnt;
        end
        if err_cnt < 1
            err_msg = sprintf('CHECK "%s" OK\n', parm_name);
        end
    end
    fprintf(2, err_msg);

end
        
function [err_cnt] = check_value(parm_name, actual, expected, tolerance, display_format)
    % Check that the 2 inputs are within the required tolerance
    delta = abs(actual - expected);
    err_cnt = int32(delta > tolerance);
    if err_cnt > 0
        err_msg = sprintf('%s\nERROR! #%02d: Expected: (%-28s), Actual: (%-28s), Delta: (%-28s)\n', parm_name, row, sprintf(display_format, expected(row,:)), sprintf(display_format, actual(row,:)), sprintf(display_format, delta));
    else
        err_msg = sprintf('CHECK "%s" OK\n', parm_name);
    end
    fprintf(2, err_msg);

end

