function Config = LcaProcessImage( Config, logFileId )
% Config = LcaProcessImage( Config, logFileId )
%
% Calculates the LCA from a black dot Chart
%
% Config.ImageWidth  = width of raw bayer image in pixels
% Config.ImageHeight = height of raw bayer image in pixels
% Config.MaxCode     = max possible image code
%                     -  8-bit data =  255
%                     - 10-bit data = 1023
% Config.Pedestal    = black level pedestal
%                     -  8-bit data =   16
%                     - 10-bit data =   64
%
% Examples:
% LcaProcessImage( Config, logFileId );
%
%  Author: Donald BAXTER, donald.baxter@st.com
%  Date:   25 September 2008
%
%  Copyright (c) 2008 STMicroelectronics
        
    if nargin < 2, logFileId = 1; end
    
    % Split file name into component parts
    [pathstr, name, ext, versn] = fileparts( Config.imageFileName );
    figureTitleHeader = sprintf( '%s%s', name, ext );
    outputFileRoot    = sprintf( '%s\\%s', pathstr, name );
    roiImageDirectory = sprintf( '%s\\%s_roi_images', pathstr, name );
    
    % If ROI Image Directory does not exist then create it
	% if( isdir( roiImageDirectory ) == 0 )
    %     mkdir( roiImageDirectory );
	% end

	% Attempt to open log file
    Config.logFileName = sprintf( '%s_lca_log.txt', outputFileRoot ); 
    logFileId = fopen( Config.logFileName, 'wt' );
    if logFileId < -1
        logFileId = 1;
        fprintf( logFileId, 'WARNING! - Failed to open log file "%s", reverting to stdout\n',...
                 Config.logFileName );
    end

    % Use the appropriate reader
    switch ext
        case '.pgm'
            % Read ASCII or Binrary PGM file
            [input, Config.ImageWidth, Config.ImageHeight, Config.MaxCode] = ReadPgmFile( Config.imageFileName, Config.Endian, logFileId );
        case '.raw'
            % Read Binary Raw file
            [input, Config.ImageWidth, Config.ImageHeight] = ReadRawFile( Config.imageFileName, Config.ImageWidth, Config.ImageHeight, Config.MaxCode, Config.Endian, logFileId );
        otherwise
            % Let imread try to read in the file
            input = imread( Config.imageFileName );
            fprintf( logFileId, 'Reading Image File: "%s"\n', Config.imageFileName );
            [Config.ImageHeight Config.ImageWidth plane_count] = size( input );
            Config.MaxCode     = 255;
            fprintf( logFileId, 'Image Info: %d x %d x %d, Max: %d\n', Config.ImageWidth, Config.ImageHeight, plane_count, Config.MaxCode );
    end
   
    % Bit shift Data
    if Config.Bitshift ~= 0
        input = bitshift( input, Config.Bitshift );
    end
    
    % Crop input to remove embedded data, black and or dummy cols
    if Config.CropEnable > 0
        input = ImageCrop( input, Config.CropXOff, Config.CropYOff, Config.CropWidth, Config.CropHeight, logFileId );
    end

    % Extract image dimensions
    [Config.CropHeight Config.CropWidth image_planes] = size( input );
    
    % Declare x and y index arrays
    fprintf( logFileId, 'Image Size: %d x %d\n', Config.CropWidth, Config.CropHeight );
    fprintf( logFileId, 'Max Code:   %d\n', Config.MaxCode );
    fprintf( logFileId, 'Image Type: "%s"\n', class(input) );
        
    % Declare x and y index arrays
    [input_xi, input_yi] = meshgrid( 0:Config.CropWidth-1, 0:Config.CropHeight-1 );
    
    % Declare colour list
    c_red          = [1.0,0.2,0.2];
    c_green        = [0.2,1.0,0.2];
    c_blue         = [0.2,0.2,1.0];
    plane_colours  = [c_red; c_green; c_blue];
    
    % Determine number colour planes in image
    % if 3 then RGB data otherwise Raw Bayer Data
    if image_planes > 1

        % RGB input data - 3 colour planes
        rgb = input;

        xi = zeros( Config.CropHeight, Config.CropWidth, 3 );
        xi(:,:,1) = input_xi;
        xi(:,:,2) = input_xi;
        xi(:,:,3) = input_xi;

        yi = zeros( Config.CropHeight, Config.CropWidth, 3 );
        yi(:,:,1) = input_yi;
        yi(:,:,2) = input_yi;
        yi(:,:,3) = input_yi;
              
        % Define the indices for the 3 RGB colour planes
        rr_plane    = 1;
        gr_plane    = 2;
        bb_plane    = 3;
        
    else
      
        % Raw Bayer Input Data
        % Split Raw Bayer Data into the 4 bayer planes
        rgb = BayerSplit( input    );
        xi  = BayerSplit( input_xi );
        yi  = BayerSplit( input_yi );
       
        % Define the indices for the 4 bayer colour planes
        gr_plane    = 1;
        rr_plane    = 2;
        bb_plane    = 3;
        
    end
    
    plane_list     = [rr_plane, gr_plane, bb_plane];
        
    % -------------------------
    % Find Centres
    % -------------------------
    
    % Determine the best thresold for binarisation
    threshold = lca_image_threshold( rgb(:,:,gr_plane), Config.MaxCode, logFileId );

    % Run 1st pass on green channel find ROI for each dot
    % This is used to determine the ROI's for the actuate 2nd Pass
    % circle search
    [roi_list Config] = lca_find_roi( Config,...
                                      rgb(:,:,gr_plane),...
                                      threshold,...
                                      logFileId );
    roi_list = lca_sort_roi( roi_list, Config.CropWidth, Config.CropHeight, Config.MeanDotDiameter, logFileId );
                             
    % Plot ROI data
    Config.figureTitleHeader  = sprintf( '%s - Regions of Interest', figureTitleHeader  );
    Config.plotFileName = sprintf( '%s_dot_centres_roi.png',   outputFileRoot );
    Config = lca_regions_of_interest_plot( Config,...
                                  rgb(:,:,gr_plane), Config.MaxCode,...
                                  roi_list,...
                                  logFileId );
    
    % Find number of roi's (circles) found
    dot_count = length( roi_list );
    fprintf( logFileId, 'Dots Found = %d\n', dot_count );

    % ----------------------------------------------------------
    % Find the centre of mass for each ROI for each colour plane
    % -----------------------------------------------------------
    % 
    % Each row of the centres is a 2-D array contains the red, green
    % and blue centres for a dot, for example
    %
    % Dot 1: [Red_x, Red_y, Green_x, Green_y, Blue_x, Blue_y]
    % Dot 2: [Red_x, Red_y, Green_x, Green_y, Blue_x, Blue_y]
    % Dot 3: [Red_x, Red_y, Green_x, Green_y, Blue_x, Blue_y]
    
    Config.centreMethod = 2;
    centres = lca_find_centres( rgb, xi, yi, Config.MaxCode,...
                                roi_list, Config.centreMethod,...
                                plane_list, plane_colours,...
                                outputFileRoot, logFileId );
                            
    % -----------------------------------------------
    % Write the centres data to an excel spreadsheet
    % -----------------------------------------------
    Config.xlsFileName = sprintf( '%s_dot_centre_data.xls', outputFileRoot );
    fprintf(         1, 'Writing Excel File: "%s"\n', Config.xlsFileName ); 
    fprintf( logFileId, 'Writing Excel File: "%s"\n', Config.xlsFileName ); 
	xlswrite( Config.xlsFileName, centres );
    
    % -----------------------------------------------
    % Process Centre Data
    % -----------------------------------------------
    % Assume centre is in the middle of the input image
    x_image_centre = double( Config.ImageWidth  / 2 );
    y_image_centre = double( Config.ImageHeight / 2 );
        
    % Find the size of the data
    [samples columns] = size( centres );
    plane_count = columns/2;
    
    % Define the indices for the 3 RGB colour planes
    rr_plane    = 1;
    gr_plane    = 2;
    bb_plane    = 3;
    plot_list = [rr_plane bb_plane];

    % Generate Radius data
    radius = sqrt((centres(:,1:2:end)-x_image_centre).^2 + (centres(:,2:2:end)-y_image_centre).^2);
    
    % Generate Radial shifts
    shift  = zeros( size(radius) );
    shift(:,rr_plane) = radius(:,rr_plane) - radius(:,gr_plane);
    shift(:,bb_plane) = radius(:,bb_plane) - radius(:,gr_plane);
    
    % Find the maximum radial distance
    max_radius = floor( max(radius(:,gr_plane)) );

    % Build Best Fit Data
    best_fit   = zeros( max_radius+1, plane_count );
  
    % Fit a 3rd order polynomial to the radial shift data for each plane
    for index=1:length(plot_list)
       plane = plot_list(index);
       polynomial = polyfit( radius(:,gr_plane), shift(:,plane), 3 );
       polyString = 'y = ';
       for order=1:length(polynomial);
           if order > 1
               polyString = sprintf( '%s + ', polyString);
           end
           polyString = sprintf( '%s%10.6g x^%d', polyString, polynomial(order), length(polynomial)-order );
       end
       fprintf( logFileId, 'Plane %d best fit polynominal: %s\n', plane, polyString );
       best_fit(:,plane) = polyval( polynomial,  0:max_radius );
    end
    
    % Lateral Colour Aberration plots
    plane_markers  = cellstr( [ 'r*';'g*';'b+';'k+'] );
    plane_lines    = cellstr( [ '-r';'-g';'-b';'-k'] );
    
    Config.figureTitleHeader = sprintf( '%s - Lateral Colour Aberration Plot', figureTitleHeader  );
    Config.plotFileName = sprintf( '%s_lca_plot.png', outputFileRoot );
    Config = lca_shift_vs_image_height_plot( Config,...
                                             radius, shift, best_fit, max_radius,...
                                             plot_list, gr_plane,...
                                             plane_lines, plane_markers,...
                                             logFileId );
    % Update results
    Config.Results(1,1) = Config.MedianDotDiameter;
    Config.Results(2,1) = Config.DotTolerance;
    Config.Results(3,1) = Config.MinDotDiameter;
    Config.Results(4,1) = Config.MeanDotDiameter;
    Config.Results(5,1) = Config.MaxDotDiameter;
    
	% Close all open file handles 
	fclose( logFileId );

end

    
    
   
