function [centre clipped] = lca_centre_of_mass_method( input, xi, yi, max_code,...
                                                        centreMethod, logFileId )
% centre=lca_centre_of_mass_method(input, xi, yi, x_image_centre, y_image_centre,logFileId )
%
% Calculates the centre of mass of the dot in the ROI
% Prior to this the ROI data is processed to remove
% the chart background from the calculation.
%
% Input Parameters:
% input  = grayscale image data ROI
% xi     = 2D array containing x index data for ROI
%          must have the same dimensions as input image
% yi     = 2D array containing y index data for ROI
%          must have the same dimensions as input image 
%
% Return Value: 4 element array containing the dot centre
%
%  Author: Donald BAXTER, donald.baxter@st.com
%  Date:   24 September 2008
%
%  Copyright (c) 2008 STMicroelectronics

    % If logFileId not specified then set to 1 (stdout)
    if nargin < 6, logFileId = 1; end
    
    % Get ROI size
    [roi_height roi_width] = size( input );
    
    % The first step is to generate 2 masks - one for the background
    % and a second for the dot. 
    % 
    % The Otsu's gray threshold function is used to determine the
    % binarisation threshold
    threshold = graythresh( double(input) / double( max_code) );
    % fprintf( logFileId, 'Threshold = %.3f (%.1f)\n', threshold, threshold*double(max_code) );

    % Generate binary mask using the above threshold
    image_plane = input;
    mask = im2bw( double(input) / double( max_code), threshold );

    % Define a disk filter (radius 3 pixels)
    % this is then used to erode the masks for the
    % background and dot to remove the edge transition
    % data
    se = strel('disk',3);        
    bg_mask  = imerode(  mask, se );
    dot_mask = imerode( ~mask, se );

    % Lens Shading
    % ===================
    % It is necessary to remove the effects of lens shading from the dot ROI.
    % For example if lens shading is present then it the profile of the dot
    % can be tilted.
    %
    % The effects of lens shading are removed by using the white data (background)
    % to calculate a best-fit linear plane (z = a*x + b*y + c).
    %
    % The dot ROI is as any len shading tilt removed and normalised at
    % the same time by dividing the dot ROI data by the best fit panel.
    % 
    % As there typically a 20x15 array of dot in the image and that
    % lens shading is a low frequency variation the assumption is made
    % that the lens shading for one dot can be approximated by a linear
    % plane

    % Use background mask to make any non background pixels zere
    image_plane( bg_mask < 1 ) = 0;

    % Build a set of simultaneous equations for matlab to solve
    % to get the best fit linear plane
    % 
    % However first the 2-D arrays have to be converted into 1-D
    % arrays
    roi_size = roi_height*roi_width;
    tmp_i   = reshape( image_plane, roi_size, 1 );
    tmp_x   = reshape( xi, roi_size, 1 );
    tmp_y   = reshape( yi, roi_size, 1 );

    % Only use valid data (> 0)
    % Get index list of valid data
    indices = tmp_i > 0;
    x1 = tmp_x( indices );
    x2 = tmp_y( indices );
    y  = tmp_i( indices );

    % Build set of simultaneous equations
    X = [ones(size(x1))  x1  x2];

    % Solve simulation equations - least squares fit
    a = X\double(y);
    % fprintf( logFileId, 'Best Fit Plane: [%s]\n', sprintf( '%.3f ', a' ));

    % Rebuild best fit image plane 2-D array
    best_fit = [ones(size(tmp_x))  tmp_x  tmp_y]*a;
    best_fit = reshape( best_fit, roi_height, roi_width );

    % Correct lens shading and normalise image data
    corrected = double(input) ./ best_fit;

    % After correcting the lens shading the average background level
    % will be 1
    %
    % The next step is to stretch the dot so that the average dot value
    % is zero
    %
    % The average dot value is mean of the pixels selected by the dot
    % mask
    image_plane = corrected;
    image_plane( dot_mask < 1 ) = 0;
    black     = double( sum(sum(image_plane)) ) / double( sum(sum(dot_mask)));

    % Determine re-scale factor
    rescale_factor = 1.0 - black;
    % fprintf( logFileId, 'Rescale factor: %.3f\n', rescale_factor );

    % Re-scale image data
    rescaled = (corrected - black)/rescale_factor;

    % Invert dot
    rescaled = 1.0 - rescaled;

    % 
    clipped = rescaled;
    
	switch centreMethod
        case 2
            % use the above bg and dot filters to
            % remove the non transitional data
            clipped(  bg_mask > 0 )  = 0.0;
            clipped( dot_mask > 0 )  = 1.0;
            % Clip image data
            clipped( clipped > 1.0 ) = 1.0;
            clipped( clipped < 0.0 ) = 0.0;
        case 1
            % Threshold data based on 50%
            clipped( clipped <  0.5 ) = 0.0;
            clipped( clipped >= 0.5 ) = 1.0;
	end


    % Determine the center of mass
    centre    = double( zeros( 1, 2 ) );
    area      = sum(sum(double(clipped)));
    centre(1) = sum(sum(double(clipped).*xi))/area;
    centre(2) = sum(sum(double(clipped).*yi))/area;
                  
end

