function lca_compare_centre_data( xls1FileName, xls2FileName  )
% lca_compare_centre_data( xls1FileName, xls2FileName  )
%
% Calculates the lateral colour aberration from a RGB set of dot centres
%
% Input Parameters:
% fileName           = image file name may be
%                      RGB (bmp, jpeg) or Raw (pgm)
%
% Return Value = Null
%
% Examples:
% lca_compare_centre_data( '.\AMD_Reference_Images\ReferenceCenters.xls', '.\AMD_Reference_Images\NoiseInjected_Blurr_5_bmp\NoiseInjected_Blurr_5_centres.xls' );
% lca_compare_centre_data( '.\AMD_Reference_Images\ReferenceCenters.xls', '.\AMD_Reference_Images\NoiseInjected_Blurr_10_bmp\NoiseInjected_Blurr_10_centres.xls' );
%
% lca_compare_centre_data( '.\DXO_Reference_Images\N70 Tungsten_jpg_dxo\n70_tungsten_centres_dxo.xls',    '.\DXO_Reference_Images\N70 Tungsten_jpg_stm\centres.xls'   );
% lca_compare_centre_data( '.\DXO_Reference_Images\Nokia N90_jpg_dxo\n90_centres_dxo.xls',                '.\DXO_Reference_Images\Nokia N90_jpg_stm\centres.xls'      );
% lca_compare_centre_data( '.\DXO_Reference_Images\NokiaN93-4.5mm_jpg_dxo\n93-4.5mm_centres_dxo.xls.xls', '.\DXO_Reference_Images\NokiaN93-4.5mm_jpg_stm\centres.xls' );
%   
%  Author: Donald BAXTER, donald.baxter@st.com
%  Date:   24 September 2008
%
%  Copyright (c) 2008 STMicroelectronics

    logFileId = 1;

    % Extract output directory from xls File Name
    [outputDirectoryPath, xls2FileNameRoot, xls2FileExt, versn] = fileparts( xls2FileName );
    if( isdir( outputDirectoryPath ) == 0 )
        mkdir( outputDirectoryPath );
    end
    

    % Read in Excel Spreadsheet contain centre 
    fprintf( logFileId, 'Read XLS File 1: "%s\n', xls1FileName );
    centres1 = xlsread( xls1FileName );
    fprintf( logFileId, 'Read XLS File 2: "%s\n', xls2FileName );
    centres2 = xlsread( xls2FileName );
    
    % Find the size of the data
    [samples1 columns1] = size( centres1 );
    [samples2 columns2] = size( centres2 );    

    % Generate x, y and distance deltas
    x_err = centres1(:,1:2:end)-centres2(:,1:2:end);
    y_err = centres1(:,2:2:end)-centres2(:,2:2:end);
    dist  = sqrt( x_err.^2 + y_err.^2 );
    
    % Calculate drg, dbg, drb distances for centres 1
    shift1      = zeros( samples1, 3 );
    shift1(:,1) = sqrt( (centres1(:,1) - centres1(:,3)).^2 + (centres1(:,2) - centres1(:,4)).^2 );
    shift1(:,2) = sqrt( (centres1(:,5) - centres1(:,3)).^2 + (centres1(:,6) - centres1(:,4)).^2 );
    shift1(:,3) = sqrt( (centres1(:,1) - centres1(:,5)).^2 + (centres1(:,2) - centres1(:,6)).^2 );
   
    % Calculate drg, dbg, drb distances for centres 2
    shift2      = zeros( samples2, 3 );
    shift2(:,1) = sqrt( (centres2(:,1) - centres2(:,3)).^2 + (centres2(:,2) - centres2(:,4)).^2 );
    shift2(:,2) = sqrt( (centres2(:,5) - centres2(:,3)).^2 + (centres2(:,6) - centres2(:,4)).^2 );
    shift2(:,3) = sqrt( (centres2(:,1) - centres2(:,5)).^2 + (centres2(:,2) - centres2(:,6)).^2 );
    
    % Delta Shifts
    delta = abs( shift1 - shift2 );
    
    % Define bins for histogram function
    x = 0.005:0.010:1.000;
    
    plane_colours  = cellstr( [ 'r';'g';'b';'k'] );
    title_list     = cellstr( [ 'Red Centre Delta Distance  ';...
                                'Green Centre Delta Distance';...
                                'Blue Centre Delta Distance '] );
    xlabel_list    = cellstr( [ '|r1-r2| distance (pixel)';...
                                '|g1-g2| distance (pixel)';...
                                '|b1-b2| distance (pixel)'] );

    plotFileName = sprintf( '%s\\%s.png', outputDirectoryPath, 'centre_delta_distance_histogram' );
    
    % Create Figure plots
    figureHandle = figure('NumberTitle',  'off',...
                          'Units',        'pixels',...
                          'DockControls', 'off',...
                          'MenuBar',      'figure',...
                          'Position', [64 64 800 240] );

    max_freq=40;
    
    % Calculate Histogram for each colour plane
    for plane=1:3
        
        colour = char( plane_colours( plane ) );
        
        statistics(1) =  min( dist(:,plane) );
        statistics(2) = mean( dist(:,plane) );
        statistics(3) =  max( dist(:,plane) );
        
        [freq xout] = hist( dist(:,plane), x );
        
        
        minString = sprintf( 'Min  = %6.3f',  statistics(1) );
        avgString = sprintf( 'Mean = %6.3f',  statistics(2) );
        maxString = sprintf( 'Max  = %6.3f',  statistics(3) );
        
        fprintf( logFileId, 'Plane %d Deltas: Min = %6.3f, Mean = %6.3f, Max = %6.3f\n',...
                             plane, statistics(1), statistics(2), statistics(3) );
        
        subplot( 1, 3, plane );
        
        bar1 = bar( xout, freq, 'FaceColor', colour, 'EdgeColor', 'k' );
 
        xlim( [0 0.5] );
        ylim( [0 max_freq] );
        set( gca, 'xtick', [0.0 0.1 0.2 0.3 0.4 0.5] );
        title( char( title_list(plane) ) );
        xlabel( char( xlabel_list(plane) ) );
        ylabel( 'Frequency' );
        
        text( 0.45, max_freq*0.9, minString, 'HorizontalAlignment', 'right' );
        text( 0.45, max_freq*0.8, avgString, 'HorizontalAlignment', 'right' );
        text( 0.45, max_freq*0.7, maxString, 'HorizontalAlignment', 'right' );
        
    end

    % Export Figure to file
    fprintf( logFileId, 'Exporting figure to "%s"\n', plotFileName );
    
    frame = getframe( figureHandle ); 

    if isempty( frame.colormap )
       imwrite( frame.cdata,  plotFileName, 'png', 'bitdepth', 8 )
    else
       imwrite( frame.cdata,  frame,colormap, plotFileName, 'png', 'bitdepth', 8 );
    end
 
    % Close figure
    close( figureHandle );

    plane_colours  = cellstr( [ 'r';'b';'k';'k'] );
    title_list     = cellstr( [ 'Red-Green Shift Delta ';...
                                'Blue-Green Shift Delta';...
                                'Red-Blue Shift Delta  '] );
    xlabel_list    = cellstr( [ '|drg1-drg2| distance (pixel)';...
                                '|dbg1-dbg2| distance (pixel)';...
                                '|drb1-drb2| distance (pixel)'] );
    
    plotFileName = sprintf( '%s\\%s.png', outputDirectoryPath, 'shift_delta_histogram' );
    
    % Create Figure plots
    figureHandle = figure('NumberTitle',  'off',...
                          'Units',        'pixels',...
                          'DockControls', 'off',...
                          'MenuBar',      'figure',...
                          'Position', [64 64 800 240] );

    % Calculate Histogram for each colour plane
    for plane=1:3
        
        colour = char( plane_colours( plane ) );
        
        statistics(1) =  min( delta(:,plane) );
        statistics(2) = mean( delta(:,plane) );
        statistics(3) =  max( delta(:,plane) );
        
        [freq xout] = hist( delta(:,plane), x );
                
        minString = sprintf( 'Min  = %6.3f',  statistics(1) );
        avgString = sprintf( 'Mean = %6.3f',  statistics(2) );
        maxString = sprintf( 'Max  = %6.3f',  statistics(3) );
        
        fprintf( logFileId, 'Plane %d Deltas: Min = %6.3f, Mean = %6.3f, Max = %6.3f\n',...
                             plane, statistics(1), statistics(2), statistics(3) );
        
        subplot( 1, 3, plane );
        
        bar1 = bar( xout, freq, 'FaceColor', colour, 'EdgeColor', 'k' );
 
        xlim( [0 0.5] );
        ylim( [0 max_freq] );
        set( gca, 'xtick', [0.0 0.1 0.2 0.3 0.4 0.5] );
        title( char( title_list(plane) ) );
        xlabel( char( xlabel_list(plane) ) );
        ylabel( 'Frequency' );
        
        text( 0.45, max_freq*0.9, minString, 'HorizontalAlignment', 'right' );
        text( 0.45, max_freq*0.8, avgString, 'HorizontalAlignment', 'right' );
        text( 0.45, max_freq*0.7, maxString, 'HorizontalAlignment', 'right' );
        
    end

    % Export Figure to file
    fprintf( logFileId, 'Exporting figure to "%s"\n', plotFileName );
    
    frame = getframe( figureHandle ); 

    if isempty( frame.colormap )
       imwrite( frame.cdata,  plotFileName, 'png', 'bitdepth', 8 )
    else
       imwrite( frame.cdata,  frame,colormap, plotFileName, 'png', 'bitdepth', 8 );
    end
 
    % Close figure
    close( figureHandle );

end

    
    
   
