function lca_extract_dot_centres( imageFileName, xlsFileName, centreMethod )
% lca_extract_dot_centres( imageFileName, centreMethod )
%
% Calculates the lateral colour aberration from a black dot chart
% Dot chart method originally proposed by Dx0
%
% Input Parameters:
% fileName           = image file name may be
%                      RGB (bmp, jpeg) or Raw (pgm)
%
% Return Value = Null
%
% Examples:
% lca_extract_dot_centres( '.\Benchmark_dot_detection\N70 Tungsten.jpg',   2 );
% lca_extract_dot_centres( '.\Benchmark_dot_detection\Nokia N90.jpg',      2 );
% lca_extract_dot_centres( '.\Benchmark_dot_detection\NokiaN93-4.5mm.jpg', 2 );
% lca_extract_dot_centres( '.\Benchmark_dot_detection\vs6556_dot_raw.pgm', 0 );
%
%  Author: Donald BAXTER, donald.baxter@st.com
%  Date:   24 September 2008
%
%  Copyright (c) 2008 STMicroelectronics
 
    % Define Matlab search paths
    addpath( '..\..\..\..\Utils\ImageFileIO' );
    addpath( '..\..\..\..\Utils\Image' );

    % Split File Name to find extension
    [outputDirectoryPath, imageFileNameRoot, imageFileExt, versn] = fileparts( imageFileName );
    
    % Build figure title header
    figureTitleHeader  = sprintf( '%s%s', imageFileNameRoot, imageFileExt );
    
    % Extract output directory from xls File Name
    [outputDirectoryPath, xlsFileNameRoot, xlsFileExt, versn] = fileparts( xlsFileName );
    if( isdir( outputDirectoryPath ) == 0 )
        mkdir( outputDirectoryPath );
    end
      
    % Attempt to open log file
    logFileName = sprintf( '%s\\extract_dot_centres_log.txt', outputDirectoryPath ); 
    logFileId = fopen( logFileName, 'wt' );
    if logFileId < -1
        logFileId = 1;
        fprintf( logFileId, 'WARNING! - Failed to open log file, reverting to stdout\n' );
    end
      
    % Use the appropriate reader
    switch imageFileExt
        case '.pgm'
            % Read ASCII PGM file
            [input, image_width, image_height, max_code] = ReadPgmFile( imageFileName, logFileId );
        otherwise
            % Let imread try to read in the file
            input = imread( imageFileName );
            max_code = 255;
    end
    
    % Extract image dimensions
    [image_height image_width image_planes] = size( input );
    
    % Declare x and y index arrays
    fprintf( logFileId, 'Image Size: %d x %d\n', image_width, image_height );
    fprintf( logFileId, 'Max Code:   %d\n', max_code );
    [input_xi, input_yi] = meshgrid( 0:image_width-1, 0:image_height-1 );
    
    % Declare colour list
    c_red          = [1.0,0.2,0.2];
    c_green        = [0.2,1.0,0.2];
    c_blue         = [0.2,0.2,1.0];
    plane_colours  = [c_red; c_green; c_blue];
    
    % Determine number colour planes in image
    % if 3 then RGB data otherwise Raw Bayer Data
    if image_planes > 1

        % RGB input data - 3 colour planes
        rgb = input;

        xi = zeros( image_height, image_width, 3 );
        xi(:,:,1) = input_xi;
        xi(:,:,2) = input_xi;
        xi(:,:,3) = input_xi;

        yi = zeros( image_height, image_width, 3 );
        yi(:,:,1) = input_yi;
        yi(:,:,2) = input_yi;
        yi(:,:,3) = input_yi;
              
        % Define the indices for the 3 RGB colour planes
        rr_plane    = 1;
        gr_plane    = 2;
        bb_plane    = 3;
        
    else
      
        % Raw Bayer Input Data
        % Split Raw Bayer Data into the 4 bayer planes
        rgb = BayerSplit( input    );
        xi  = BayerSplit( input_xi );
        yi  = BayerSplit( input_yi );
       
        % Define the indices for the 4 bayer colour planes
        gr_plane    = 1;
        rr_plane    = 2;
        bb_plane    = 3;
        
    end
    
    plane_list     = [rr_plane, gr_plane, bb_plane];
    
    % Rescale and save gr_plane
    [plane_height plane_width plane_count ] = size( rgb );
    rescaled = imresize( rgb(:,:,gr_plane), 640/plane_width );
    rescaled = uint8( double(rescaled) * double(255) / double(max_code) );
    jpegFileName = sprintf( '%s\\gr_plane.jpg', outputDirectoryPath );
    imwrite( rescaled, jpegFileName );
    
    % -------------------------
    % Find Centres
    % -------------------------
    
    % Determine the best thresold for binarisation
    threshold = lca_image_threshold( rgb(:,:,gr_plane), max_code, logFileId );

    % Run 1st pass on green channel find ROI for each dot
    % This is used to determine the ROI's for the actuate 2nd Pass
    % circle search
    [roi_list dot_diameter dot_tolerance] = lca_find_roi( rgb(:,:,gr_plane), max_code,...
                                                          threshold, logFileId );
    roi_list = lca_sort_roi( roi_list, image_width, image_height, dot_diameter, logFileId );
                             
    % Plot ROI data
    figure_title = sprintf( '%s - Regions of Interest', figureTitleHeader  );
    plotFileName = sprintf( '%s\\extract_dot_centres_roi.png', outputDirectoryPath );
    lca_regions_of_interest_plot( figure_title,...
                                  rgb(:,:,gr_plane), max_code,...
                                  roi_list,...
                                  plotFileName,...
                                  logFileId );
    
    % Find number of roi's (circles) found
    dot_count = length( roi_list );
    fprintf( logFileId, 'Dots Found = %d\n', dot_count );

    % ----------------------------------------------------------
    % Find the centre of mass for each ROI for each colour plane
    % -----------------------------------------------------------
    % 
    % Each row of the centres is a 2-D array contains the red, green
    % and blue centres for a dot, for example
    %
    % Dot 1: [Red_x, Red_y, Green_x, Green_y, Blue_x, Blue_y]
    % Dot 2: [Red_x, Red_y, Green_x, Green_y, Blue_x, Blue_y]
    % Dot 3: [Red_x, Red_y, Green_x, Green_y, Blue_x, Blue_y]
    
    centres = lca_find_centres( rgb, xi, yi, max_code,...
                                roi_list, centreMethod,...
                                plane_list, plane_colours,...
                                outputDirectoryPath, logFileId );
                            
    % -----------------------------------------------
    % Write the centres data to an excel spreadsheet
    % -----------------------------------------------
    fprintf( logFileId, 'Writing Excel File: "%s"\n', xlsFileName ); 
	xlswrite( xlsFileName, centres );
        
	% Close all open file handles 
	fclose( logFileId );
                                
end

    
    
   
