function [roi_list Config]=lca_find_roi( Config, input, threshold, logFileId )
% [roi_list Config]=lca_find_roi( Config, input, threshold, logFileId )
%
% Returns a list of ROI - one per image dot.
% Each ROI is ~2 the size of each dot and is approximately
% has the same centre as the dot.
%
% Subsequently each ROI is used to crop the colour planes for
% the centre of mass (red/green/blue dot centre) calculation.
%
% Green(R) image plane data should be used as it is generally
% stronger than the red and blue channels.
%
% Input Parameters:
% input              = 2-D array containing grayscale image data
%                      for the green(R) colour plane
% threshold          = binarisation threshold calculated by the
%                      lca_image_threshold function
%                      valid range [0 1.0]
%
% Return Value = array containing the list of ROI for each dot
%
%  Author: Donald BAXTER, donald.baxter@st.com
%  Date:   25 September 2008
%
%  Copyright (c) 2008 STMicroelectronics

    % If logFileId not specified then set to 1 (stdout)
    if nargin < 4, logFileId = 1; end

    fprintf( logFileId, '%%%%Finding the ROIs for each dot\n' );
    
    % Extract Image dimensions
    [image_height image_width] = size( input );
    fprintf( logFileId, 'Image Size: %d x %d\n', image_width, image_height );
    
    %% Pass 1: Binarise the Green Image Plane
    fprintf( logFileId, '%% Pass 1: Binarise the Green Image Plane\n' );
    fprintf( logFileId, 'Binarisation Threshold: %.1f\n', threshold * Config.MaxCode );
    
    % Binarise the image
    bw = im2bw( double(input)/double(Config.MaxCode), threshold );

    % As black dots then need to invert...
    bw = ~bw;
    
    % fill in small gaps
    se = strel('disk',2);
    bw = imclose(bw,se);

    % fill any holes, so that regionprops can be used to estimate
    % the area enclosed by each of the boundaries
    bw = imfill(bw,'holes');
    
    % Find the Boundaries
    [B,L] = bwboundaries(bw,'noholes');
    stats = regionprops(L,'Area','Centroid','BoundingBox');

    % Find the number of regions
    region_count = length(B);
    
    fprintf( logFileId, 'Pass: 1 - Threshold found %d ROIs\n', region_count );
    
    % Assume dot check true until proved otherwise
    dot_check = ones( 1, region_count );
    
    % Initialise arrays
    dot_size   = double( zeros( 1, region_count ));
    area_ratio = dot_size;
    roi_x_ll = dot_size;
    roi_y_ll = dot_size;
    roi_x_ur = dot_size;
    roi_y_ur = dot_size;
                    
    for k = 1:region_count
         area_ratio(k) = (stats(k).BoundingBox(3)* stats(k).BoundingBox(4)) / stats(k).Area;
         dot_size(k)   = mean(stats(k).BoundingBox(3), stats(k).BoundingBox(4));
         roi_x_ll(k)   = int16(floor( stats(k).Centroid(1) - dot_size(k) ));
         roi_y_ll(k)   = int16(floor( stats(k).Centroid(2) - dot_size(k) ));
         roi_x_ur(k)   = int16(ceil(  stats(k).Centroid(1) + dot_size(k) ));
         roi_y_ur(k)   = int16(ceil(  stats(k).Centroid(2) + dot_size(k) ));
    end
    
    fprintf( logFileId, '%% Pass: 2 - Checking the "roundness" of each dot\n' );
    
    %% Check the ratio of the bounding box area to the area of the shape
    % ideally should be 4 / 3.14 = 1.27 check should be 1.05 -> 1.75   
    dot_check(  area_ratio < 1.05 ) = 0;
    dot_check(  area_ratio > 1.75 ) = 0;
    
    dot_count = sum(dot_check, 2);
    fprintf( logFileId, 'Roundness Check = %d ROIs passed\n',  dot_count );
    
    fprintf( logFileId, '%% Pass: 3 - Checking the diameter of each dot\n' );
    
    % Determine the median dot size
    dotSizeList   = sort( dot_size( dot_check > 0 ) );
    Config.MedianDotDiameter  = dotSizeList( uint16( length(dotSizeList)/2 ) );
    fprintf( logFileId, 'Median Dot Diameter:    %8.3f\n', Config.MedianDotDiameter ); 
    
    % Check that the dot size is within 10% of the median
    Config.DotTolerance = Config.MedianDotDiameter * 0.20;
    fprintf( logFileId, 'Dot Diameter Tolerance: %8.3f\n', Config.DotTolerance );
    
    dot_check( dot_size < (Config.MedianDotDiameter-Config.DotTolerance) ) = 0;
    dot_check( dot_size > (Config.MedianDotDiameter+Config.DotTolerance) ) = 0;
    
    dot_count = sum(dot_check, 2);
    fprintf( logFileId, 'Diameter Check: %d ROIs passed\n',  dot_count );
    
    dotSizeList   = sort( dot_size( dot_check > 0 ) );
    Config.MinDotDiameter  = min(  dotSizeList );
    Config.MeanDotDiameter = mean( dotSizeList );
    Config.MaxDotDiameter  = max(  dotSizeList );
    
    fprintf( logFileId, '\n' );
    fprintf( logFileId, 'Min Dot Diameter:       %8.3f\n', Config.MinDotDiameter );
    fprintf( logFileId, 'Mean Dot Diameter:      %8.3f\n', Config.MeanDotDiameter );
    fprintf( logFileId, 'Max Dot Diameter:       %8.3f\n', Config.MaxDotDiameter );
    fprintf( logFileId, '\n' );
    
    fprintf( logFileId, '%% Pass: 4 - Checking that each ROI does not touch edge of image\n' );

    % Check that ROI lie within array if not then clip
    % If clipped then ROI check not valid
    dot_check( roi_x_ll <= 1 )= 0;
    dot_check( roi_y_ll <= 1 )= 0;
    dot_check( roi_x_ur >= image_width  )= 0;
    dot_check( roi_y_ur >= image_height )= 0;
    
    dot_count = sum(dot_check,2);
    fprintf( logFileId, 'ROI Check: %d ROIs passed\n', dot_count );
    
    % Build list of valid ROI's
    roi_list  = zeros(1,4);
    dot_count = 0;
    for k = 1:region_count
        
        % Save the circles to output array if the checks are met
        if dot_check(k) > 0;
              
            dot_count = dot_count + 1;
            roi_list(dot_count,:) = [roi_x_ll(k) roi_y_ll(k) roi_x_ur(k) roi_y_ur(k)];          
            %fprintf( logFileId, 'REGION %03d: Diameter = %8.3f, Ratio: %8.3f, Check = %d, ROI [%s] - OK\n',...
            %                       k, dot_size(k), area_ratio(k), dot_check(k), sprintf( '%5d ', roi_list(dot_count,:))  );
        else
            %fprintf( logFileId, 'REGION %03d: Diameter = %8.3f, Ratio: %8.3f, Check = %d - BAD\n',...
            %                       k, dot_size(k), area_ratio(k), dot_check(k)  );
            
        end
        
    end
    
    fprintf( logFileId, 'Summary Found %d dots\n', dot_count );

end
