function threshold=lca_image_threshold(input, max_code, logFileId)
% threshold=lca_image_threshold(input, max_code, logFileId)
%
% Calculates the binarisation threshold for the grayscale image
% contains black dots. Due to lens shading an dot chart image may
% contain a radial roll-off in the white value.
%
% This roll-off causes problems for Matlab 'graythres' function
% if the whole image is used
%
% To solve this this function calculates the 'graythres' value
% for 5 ROIs the 4 corners and the centre.
%
% The min value of the 5 ROI's is returned a better binarisation
% threshold for the whole image.
%
% Input Parameters:
% input    = 2-D array containing grayscale image data for
%            the green(R) colour plane
%
% Return Value = threshold for binarising grayscale image
%
%  Author: Donald BAXTER, donald.baxter@st.com
%  Date:   24 September 2008
%
%  Copyright (c) 2008 STMicroelectronics

    % If logFileId not specified then set to 1 (stdout)
    if nargin < 3, logFileId = 1; end

    % Extract Image dimensions
    [image_height image_width] = size( input );

    % Calculate ROI which is a 64-th of the array
    roi_width  = uint16( floor((image_width  / 8.0)/2.0)*2-1 );
    roi_height = uint16( floor((image_height / 8.0)/2.0)*2-1 );

    % Calculate Image Centre
    image_x_cent = uint16( floor(image_width/2 ) );
    image_y_cent = uint16( floor(image_height/2) ) ;

    % Build a list which contains ROI for the 4 corners
    % of the image and the centre region
    roi_list      = zeros( 64, 4 );
    for y=0:7
        for x=0:7
            roi = x+(y*8)+1;
            x_ll = uint16( (int16(x)-4)*int16(roi_width)  + int16(image_x_cent));
            y_ll = uint16( (int16(y)-4)*int16(roi_height) + int16(image_y_cent));
            x_ur = x_ll + roi_width -1;
            y_ur = y_ll + roi_height-1;
            roi_list(roi,:) = [x_ll y_ll x_ur y_ur];
            fprintf( logFileId, 'ROI (%d,%d) : [%s]\n', x, y, sprintf( ' %4d ', roi_list(roi,:) ) );
        end
    end

    % Loop through the 5 ROI is 'graythres' threshold of
    % each ROI
    threshold_list = double( zeros( 1,64 ) );
    
    for index=1:length(threshold_list)
        
        % roi
        x_ll  = roi_list(index,1);
        y_ll  = roi_list(index,2);
        x_ur  = roi_list(index,3);
        y_ur  = roi_list(index,4);
        
        roi = double( input( y_ll:y_ur, x_ll:x_ur ) ) / double(max_code);
        
        % Determine the best threshold for the binarising the ROI
        threshold_list(index) = graythresh( roi );
        fprintf( logFileId, 'ROI %02d: (%4d,%4d)->(%4d,%4d), Threshold: %.1f\n',...
                    index, x_ll, y_ll, x_ur, y_ur, threshold_list(index)*double(max_code) );

    end
    
    % Using the minimum of the 4 corners should take into account
    % the lens shading when thresholding the image
    threshold = min( threshold_list );
    fprintf( logFileId, 'Min Binarisation Threshold: %.3f (%.1f)\n', threshold, threshold*double(max_code) );
    
end
