function lca_process_centre_data( imageFileName, xlsFileName  )
% lca_process_centre_data( imageFileName, xlsFileName  )
%
% Calculates the lateral colour aberration from a RGB set of dot centres
%
% Input Parameters:
% fileName           = image file name may be
%                      RGB (bmp, jpeg) or Raw (pgm)
%
% Return Value = Null
%
% Examples:
% lca_process_centre_data( '.\750_dot_images\VB6750_D01_lens_L_big_dots.pgm', '.\750_dot_images\VB6750_D01_lens_L_big_dots_dot_centres.xls' );
%
%  Author: Donald BAXTER, donald.baxter@st.com
%  Date:   24 September 2008
%
%  Copyright (c) 2008 STMicroelectronics
    
    % Define Matlab search paths
    addpath( '..\..\..\..\Utils\ImageFileIO' );
    addpath( '..\..\..\..\Utils\Image' );

    % Split File Name to find extension
    [outputDirectoryPath, imageFileNameRoot, imageFileExt, versn] = fileparts( imageFileName );
    
    % Build figure title header
    figureTitleHeader  = sprintf( '%s%s', imageFileNameRoot, imageFileExt );
    
    % Extract output directory from xls File Name
    [outputDirectoryPath, xlsFileNameRoot, xlsFileExt, versn] = fileparts( xlsFileName );
    if( isdir( outputDirectoryPath ) == 0 )
        mkdir( outputDirectoryPath );
    end
    
    % Attempt to open log file
    logFileName = sprintf( '%s\\process_centre_data_log.txt', outputDirectoryPath ); 
    logFileId = fopen( logFileName, 'wt' );
    if logFileId < -1
        logFileId = 1;
        fprintf( logFileId, 'WARNING! - Failed to open log file, reverting to stdout\n' );
    end

    % Use the appropriate reader
    switch imageFileExt
        case '.pgm'
            % Read ASCII PGM file
            [input, image_width, image_height, max_code] = ReadPgmFile( imageFileName, logFileId );
        otherwise
            % Let imread try to read in the file
            input = imread( imageFileName );
            max_code = 255;
    end
    
    % Extract image dimensions
    [image_height image_width image_planes] = size( input );

    % Assume centre is in the middle of the input image
    x_image_centre = double( image_width  / 2 );
    y_image_centre = double( image_height / 2 );
    
    % Read in Excel Spreadsheet contain centre data
    centres = xlsread( xlsFileName );
    
    % Find the size of the data
    [samples columns] = size( centres );
    plane_count = columns/2;
    
    % Define the indices for the 3 RGB colour planes
    rr_plane    = 1;
    gr_plane    = 2;
    bb_plane    = 3;
    plot_list = [rr_plane bb_plane];

    % Generate Radius data
    radius = sqrt((centres(:,1:2:end)-x_image_centre).^2 + (centres(:,2:2:end)-y_image_centre).^2);
    
    % Generate Radial shifts
    shift  = zeros( size(radius) );
    shift(:,rr_plane) = radius(:,rr_plane) - radius(:,gr_plane);
    shift(:,bb_plane) = radius(:,bb_plane) - radius(:,gr_plane);
    
    % Find the maximum radial distance
    max_radius = floor( max(radius(:,gr_plane)) );

    % Build Best Fit Data
    best_fit   = zeros( max_radius+1, plane_count );
  
    % Fit a 3rd order polynomial to the radial shift data for each plane
    for index=1:length(plot_list)
       plane = plot_list(index);
       polynomial = polyfit( radius(:,gr_plane), shift(:,plane), 3 );
       polyString = 'y = ';
       for order=1:length(polynomial);
           if order > 1
               polyString = sprintf( '%s + ', polyString);
           end
           polyString = sprintf( '%s%10.6g x^%d', polyString, polynomial(order), length(polynomial)-order );
       end
       fprintf( logFileId, 'Plane %d best fit polynominal: %s\n', plane, polyString );
       best_fit(:,plane) = polyval( polynomial,  0:max_radius );
    end
    
    % Lateral Colour Aberration plots
    plane_markers  = cellstr( [ 'r*';'g*';'b+';'k+'] );
    plane_lines    = cellstr( [ '-r';'-g';'-b';'-k'] );
    
    figure_title = sprintf( '%s - Lateral Colour Aberration Plot', figureTitleHeader  );
    plotFileName = sprintf( '%s\\lca_shift_vs_image_height.png', outputDirectoryPath );
    lca_shift_vs_image_height_plot( figure_title,...
                                    radius, shift, best_fit, max_radius,...
                                    plot_list, gr_plane,...
                                    plane_lines, plane_markers,...
                                    plotFileName,...
                                    logFileId );
                                
	% Close all open file handles 
	fclose( logFileId );
    
end

    
    
   
