function output=lca_sort_roi( input, image_width, image_height, dot_diameter, logFileId )
% output=lca_sort_roi(input, image_width, image_height, logFileId )
%
% Re-orders an array of x,y values into assending left to right and 
% top to bottom raster order
%
% Input Parameters:
% input              = 2-D array containing grayscale image data
%                      for the green(R) colour plane
% image_width        = width of original image
% image_height       = height of original image
%
% Return Value = array containing the sorted values
%
%  Author: Donald BAXTER, donald.baxter@st.com
%  Date:   24 September 2008
%
%  Copyright (c) 2008 STMicroelectronics

    % If logFileId not specified then set to 1 (stdout)
    if nargin < 5, logFileId = 1; end

    fprintf( logFileId, '%% - Sorting Array into a left to right and top to bottom order\n' );

    % Extract and sort Y data
    y_data = sort( input(:,2) );

    % Generate the average y value for each row
    % a new row is defined if the delta between the
    % current and the last y value is > dot_diameter
    sum       = double( 0.0 );
    count     =  0;
    avg_count =  0;
    avg       = double( zeros( 1 ) );

    for y_index=1:length(y_data)
        if y_index == 1
            sum  = y_data( y_index );
            count = count + 1;
        else
            y_delta = abs( y_data(y_index) - y_data(y_index-1) );
            if y_delta > dot_diameter
                avg_count = avg_count + 1;
                avg(avg_count) = sum / double( count );
                fprintf( logFileId, '# %2d: Sum = %8.3f, Count = %3d, Mean = %8.3f\n',...
                                avg_count, sum, count, avg(avg_count) );
                sum   = 0.0;
                count = 0;
            end
            sum = sum + y_data( y_index );
            count = count + 1;
        end
    end
    if count > 0
        avg_count      = avg_count + 1;
        avg(avg_count) = sum / double( count );
        fprintf( logFileId, '# %2d: Sum = %8.3f, Count = %3d, Mean = %8.3f\n',...
                          avg_count, sum, count, avg(avg_count) );
    end

    fprintf( logFileId, 'Means: [%s]\n', sprintf( '%8.3f ', avg ) );
    
    % Generate mid point data for binning the real y data
    bins = cat( 2, [0.0], (avg(1:end-1) + avg(2:end))/2, [double(image_height)] );
    
    fprintf( logFileId, 'Bins:  [%s]\n', sprintf( '%8.3f ', bins ) );

    % Bin the y_data values
    y_data = input(:,2);

    for y_index=1:length(y_data)
        a_index = bins < y_data(y_index);
        y_data(y_index) = max( bins(a_index) );
    end

    % Build a raster value - x + y*image_width
    raster  = input(:,1) + (y_data * double(image_width));
    % Sort raster values into asscending order
    [r_list r_index] = sort( raster );
    % Use sort index values to reorder the input array
    output = input( r_index, : );

end
