function output=ImageToDiagonal( input, x_grid_size, lineDirection, logFileId )
% output=ImageToDiagonal( rgb, x_grid_size, logFileId )
%
% Divides the inage into x_grid_size by XX equal sized
% ROIs. Returns an array containing the mean of each ROI
%
% Automatically determines the y_grid_size required - each
% ROI is square
%
% Input Parameters:
% input        = input image
% x_grid_size = height of the colour plane in pixels
%
% Return Value: image_diagonal (array)
%
% Examples:
% grid_data = ImageToDiagonal( rgb, 32 );
%
%  Author: Donald BAXTER, donald.baxter@st.com
%  Date:   24 September 2008
%
%  Copyright (c) 2008 STMicroelectronics

    % If logFileId not specified then set to 1 (stdout)
    if nargin < 4, logFileId = 1; end

    % Extract Colour Plane Dimensions
    plane_width  = size( input, 2 );
    plane_height = size( input, 1 );
    plane_count  = size( input, 3 );
    fprintf( logFileId, 'Image: %d x %d x %d\n', plane_width, plane_height, plane_count );

    % Define Grid Size, Spacing and X/Y Offsets
    grid_spacing  = floor( plane_width / x_grid_size );
    x_grid_offset = floor((plane_width  - (x_grid_size *  grid_spacing ))/2); 
    y_grid_size   = floor( plane_height / grid_spacing );
    y_grid_offset = floor((plane_height - (y_grid_size *  grid_spacing ))/2);
    
    fprintf( logFileId, 'Grid: Size = %d x %d, Spacing = %d, XOff = %d, YOff = %d\n',...
                     x_grid_size, y_grid_size, grid_spacing, x_grid_offset, y_grid_offset );
                 
    % Intialise output array
    output = zeros( x_grid_size, plane_count );
    
    % Average each region generate grid
    for plane=1:plane_count;
        
        % Extract Diagonal
        for x_index=1:x_grid_size;
            
             switch lower( lineDirection )
                 case 'line4'
                     x = double(x_grid_offset) + double(grid_spacing)*(double(x_index)-0.5);
                     y = double(x * plane_height)/double(plane_width);
                     x = double(plane_width) - x;
                 case 'line2'
                     x = double(x_grid_offset) + double(grid_spacing)*(double(x_index)-0.5);
                     y = double(x * plane_height)/double(plane_width);
                 otherwise
                     x = double(x_grid_offset) + double(grid_spacing)*(double(x_index)-0.5);
                     y = double(plane_height/2);
             end
             
             x_ll = uint16( floor(x - double(grid_spacing)/2.0) ) + 1.0;
             y_ll = uint16( floor(y - double(grid_spacing)/2.0) ) + 1.0;
             
             % Lower left Checks
             if x_ll < 1
                 x_ll = 0;
             end
             if y_ll < 1
                 y_ll = 1;
             end
             
             x_ur = x_ll + grid_spacing - 1;
             y_ur = y_ll + grid_spacing - 1;

             % Upper Right Checks
             if x_ur > plane_width
                 x_ur = plane_width;
                 x_ll = x_ur - grid_spacing + 1;
             end
             if y_ur > plane_height
                 y_ur = plane_height;
                 y_ll = y_ur - grid_spacing + 1;
             end
             
             % Calculate ROI Mean
             output( x_index, plane ) = mean2( double(input(y_ll:y_ur, x_ll:x_ur, plane) ) );
             
             % fprintf( logFileId, 'Plane %d: ROI %d: (%4d,%4d)->(%4d,%4d) = %.1f\n', plane, x_index, x_ll, y_ll, x_ur, y_ur, output( x_index, plane ) ) );

        end
        
    end
        
end

    
    
   
