function output=ImageToGrid( input, x_grid_size, logFileId )
% output=ImageToGrid( rgb, x_grid_size, logFileId )
%
% Divides the inage into x_grid_size by XX equal sized
% ROIs. Returns an array containing the mean of each ROI
%
% Automatically determines the y_grid_size required - each
% ROI is square
%
% Input Parameters:
% input        = input image
% x_grid_size = height of the colour plane in pixels
%
% Return Value: image_grid (array)
%
% Examples:
% grid_data = ImageToGrid( rgb, 32 );
%
%  Author: Donald BAXTER, donald.baxter@st.com
%  Date:   24 September 2008
%
%  Copyright (c) 2008 STMicroelectronics

    % If logFileId not specified then set to 1 (stdout)
    if nargin < 3, logFileId = 1; end

    % Extract Colour Plane Dimensions
    plane_width  = size( input, 2 );
    plane_height = size( input, 1 );
    plane_count  = size( input, 3 );
    fprintf( logFileId, 'Image: %d x %d x %d\n', plane_width, plane_height, plane_count );

    % Define Grid Size, Spacing and X/Y Offsets
    grid_spacing  = floor( plane_width / x_grid_size );
    x_grid_offset = floor((plane_width  - (x_grid_size *  grid_spacing ))/2); 
    y_grid_size   = floor( plane_height / grid_spacing );
    y_grid_offset = floor((plane_height - (y_grid_size *  grid_spacing ))/2);
    
    fprintf( logFileId, 'Grid: Size = %d x %d, Spacing = %d, XOff = %d, YOff = %d\n',...
                     x_grid_size, y_grid_size, grid_spacing, x_grid_offset, y_grid_offset );

    % Intialise output array
    output = zeros( y_grid_size, x_grid_size, plane_count );
    
    % Average each region generate grid
    for plane=1:plane_count;
        
        for y_index=1:y_grid_size;
            for x_index=1:x_grid_size;
                
                x_ll = x_grid_offset + (grid_spacing * (x_index-1)) + 1;
                y_ll = y_grid_offset + (grid_spacing * (y_index-1)) + 1;
                x_ur = x_ll + grid_spacing - 1;
                y_ur = y_ll + grid_spacing - 1;
                
                output( y_index, x_index, plane ) = mean2( double(input(y_ll:y_ur, x_ll:x_ur, plane) ) );
            end
        end
    end
        
end

    
    
   
