function FileList = DirectoryFileSearch( FileList, FilePath, FileMatchPattern, logFileId )
%FileList = DirectoryFileSearch( FileList, FilePath, FileMatchPattern, logFileId )
%
% Performs a recursive file search to find all file with the specified file
% extensions
%
% Input Parameters:
% FileList         = Input CELL array - maybe empty or with data
%                    If already contains data then appends new data to the end
% FilePath         = FileRoot to start the recusive search
%                    For current directory enter '.\'
% FileMatchPattern = File Name Match Pattern for regexp
% logFileId        = file identifier for logging output (optional)
%                    If not specified then logFileId = 1 (stdout)
%
% Return Value(s):
% FileList = CELL array containing a list of all of the files found
%
% Examples:
%
% FileList = {};  % Note MUST be a cell array
% FileList = DirectoryFileSearch( FileList, '.\', '10frame.pgm' );
%
%  Author: Donald BAXTER, donald.baxter@st.com
%  Date:   24 September 2008
%
%  Copyright (c) 2008 STMicroelectronics

	% If logFileId not specified then set to 1 (stdout)
	if nargin < 4, logFileId = 1; end

    % List all files in current directir
    directoryFileList = dir( FilePath );
     
    % Convert into a cell array to sort into ASCII dictionary order
    cellArrayList                    = struct2cell( directoryFileList );
    
    % Sort the File names (field 1) into  ASCII dictionary order
    [sortedCellList sortedIndexList] = sort( cellArrayList( 1,: ) );
    
	% Loop through directory file list
	for index=1:length(sortedIndexList)
        
        % Get sortIndex value
        sortedIndex = sortedIndexList( index );
        
        % Extract file details
        fileName     = directoryFileList(sortedIndex).name;      
        fileType     = directoryFileList(sortedIndex).isdir;
        
        % Build full file name
        fullFileName = fullfile( FilePath, fileName );
                  
        % If not '.' or '..' then proceed
        if( strcmp(fileName,'.')==0 && strcmp(fileName,'..')==0 )
             
            % If fileType = 1 then directory otherwise a file
            if( fileType == 1 )
                % fprintf( logFileId, 'Directory: "%s"\n', fullFileName );
                FileList = DirectoryFileSearch( FileList, fullFileName, FileMatchPattern, logFileId );
            elseif( regexpi( fullFileName, FileMatchPattern ) )
                fprintf( logFileId, '%s\n', fullFileName );
                fileCount = length( FileList );
                FileList{fileCount+1} = fullFileName;
            end
        end
        
    end
    
end
