function [image, image_width, image_height, max_code]=ReadPgmFile( pgmFileName, endian, logFileId )
% [image, image_width, image_height, max_code]=ReadPgmFile( pgmFileName, endian, logFileId )
% Reads in an Binary/ASCII PGM Raw Image either 8-bit or 16-bit
%
% Input Parameters:
% pgmFileName  = input ASCII/Binary PGM file name
% endian       = endianness of input file
%                0 - little endian
%                1 - big endian
% logFileId    = file identifier for logging output (optional)
%                If not specified then logFileId = 1 (stdout)
%
% Return Parameters:
% image = 2-D integer array containing raw bayer image data
%
% Examples:
% [image, image_width, image_height, max_code] = ReadPgmFile( 'DUT_1_AV_On_D65_10bit_10.bmp_AvgOut.pgm', 1 )
%
%  Author: Donald BAXTER, donald.baxter@st.com
%  Date:   24 September 2008
%
%  Copyright (c) 2008 STMicroelectronics

    % If logFileId not specified then set to 1 (stdout)
    if nargin < 3, logFileId = 1; end
    
    % Attempt to open PGM Raw Bayer Image File
    fid = fopen( pgmFileName, 'rb' );
    
    if fid > -1
       fprintf( logFileId, 'Reading PGM File "%s"\n', pgmFileName );
    
       % Read in pgm file infomation header
       info           = uint16(zeros( 4 ) );
       index          = 1;
       prev_data_flag = 0;
       commentFlag    = 0;
       
       while( index <= 4 )
           
           % Read in a character 
           byte = fread( fid, 1, 'uint8' );
           
           if byte == uint8('#')  % # Comment Character
               commentFlag = 1;
               data_flag = 0;
           elseif byte == 10 || byte == 13  % 10=NL, 13=CR
               commentFlag = 0;
               data_flag = 0;
           elseif commentFlag == 0 && byte >= uint8('0') && byte <= uint8('9')
               info(index) = info(index)*10 + (uint16(byte)-uint16('0'));
               data_flag = 1;
           else
               data_flag = 0;
           end    
            
           if prev_data_flag == 1 && data_flag == 0
               index = index + 1;
           end
           
           prev_data_flag = data_flag;          
       end
       
       % Print output Pgm file information
       pgmFileType  = info(1);
       image_width  = info(2);
       image_height = info(3);
       max_code     = info(4);

       fprintf( logFileId, 'PGM File Type = "P%d"\n', pgmFileType );
       fprintf( logFileId, 'Image Info: %d x %d, Max: %d\n', image_width, image_height, max_code );
    
       % Endianess
       if endian > 0
            machineFormat = 'ieee-be'; % Big Endian
       else
            machineFormat = 'ieee-le'; % Little Endian
       end
       fprintf( logFileId, 'Format: "%s"\n', machineFormat );

        % Read in data values
       if pgmFileType == 5
           if max_code > 255
               image = fread( fid, 'uint16', machineFormat );
           else
               image = fread( fid, 'uint8',  machineFormat );
           end
       else
           image = fscanf( fid, '%d', inf );
       end

       if max_code > 255
           image = uint16( image );
       else
           image = uint8( image );
       end
       
       fclose( fid );
    
       % Determine the number of values read
       valuesRead = length( image );
       valuesType = class( image );
       
       fprintf( logFileId, 'Read %d values\n',   valuesRead );  
       fprintf( logFileId, 'Array type: "%s"\n', valuesType );  
            
       % transform image
       image = reshape(image, info(2), info(3) )';
       
    else
        fprintf( logFileId, 'ERROR! - Failed to open PGM File "%s"\n', pgmFileName );
        image        = [];
        image_width  = 0;
        image_height = 0;
        max_code     = 0;
    end
    
end

    

