function [image image_width image_height]=ReadRawFile(rawFileName, image_width, image_height, max_code, endian, logFileId )
% image=ReadRawFile(rawFileName, image_width, image_height, max_code, endian, logFileId )
% Reads in an Binary Raw Bayer Image either 8-bit or 16-bit
%
% Input Parameters:
% rawFileName  = input Binary Raw file name
% image_width  = width of raw image file in pixels
% image_height = height of raw image file in pixels
% max_code     = maximum code value
%                 8-bit = 255
%                10-bit = 1023
% endian       = endianness of input file
%                0 = little endian
%                1 = big endian
%% logFileId    = file identifier for logging output (optional)
%                If not specified then logFileId = 1 (stdout)
%
% Return Parameters:
% image = 2-D integer array containing raw image data
%
% Examples:
% image = ReadRawFile( 'DUT_1_AV_On_D65_10bit_10.bmp_AvgOut.raw', 2048, 1536, 1023, 1 )
%
%  Author: Donald BAXTER, donald.baxter@st.com
%  Date:   24 September 2008
%
%  Copyright (c) 2008 STMicroelectronics

    % If logFileId not specified then set to 1 (stdout)
    if nargin < 6, logFileId = 1; end

    % Attempt to open Binary  Raw Image File
    fid = fopen(rawFileName, 'rb' );
    
    if fid > -1
        
        fprintf( logFileId, 'Reading Raw File "%s"\n', rawFileName );
    
        % Endianess
        if endian > 0
            machineFormat = 'ieee-be'; % Big Endian
        else
            machineFormat = 'ieee-le'; % Little Endian
        end
        fprintf( logFileId, 'Format: "%s"\n', machineFormat );

        % Read in raw data
        if max_code > 255
            image=fread( fid, 'uint16', machineFormat);
        else
            image=fread( fid, 'uint8',  machineFormat);
        end
        fclose( fid );

        if max_code > 255
           image = uint16( image );
        else
           image = uint8( image );
        end
       
        % Determine the number of values read
        valuesRead = length( image );
        valuesType = class( image );
        
        fprintf( logFileId, 'Read %d values\n',   valuesRead );  
        fprintf( logFileId, 'Array type: "%s"\n', valuesType );  

        % transform image
        image_size = uint32(image_width) * uint32(image_height);
        
        % Clip as approriate
        if valuesRead > image_size
             fprintf( 2, 'WARNING!: Too Many values - Read %d  expecting %d!\n',...
                         valuesRead, image_size );
             fprintf( 2, 'WARNING!: Too Many values - Only reading input image size!\n' );
        elseif valuesRead < image_size
             fprintf( 2, 'WARNING!: Too Few values - Read %d  expecting %d!\n',...
                         valuesRead, image_size );
             image_height = uint16( floor( double(valuesRead) / double(image_width) ) );
             fprintf( 2, 'WARNING!: Too Few values - Clipping image height to %d!\n',...
                         image_height  );
        end
        
        image_size = uint32(image_width) * uint32(image_height);
        image = reshape( image(1:image_size), image_width, image_height )';
    
    else
        fprintf( logFileId, 'ERROR! - Failed to open Raw  File "%s"\n', rawFileName );
        image = [];
    end
    
end

