function WritePgmFile( image, max_code, image_file_name, pgmFileType, logFileId )
% WritePgmFile( image, max_code, image_file_name, pgmFileType, logFileId )
%
% Writes a Binary Pgm Bayer Image either 8-bit or 16-bit
%
% Input Parameters:
% image        = 2-D image array
%                must be either uint8 or unit16
% max_code     = maximum valid value
%                 8-bit data =  255
%                10-bit data = 1023
% pgmFileName  = output Binary Pgm file name
% pgmFileType  = 2 (Ascii) or 5 (Binary)
% logFileId    = file identifier for logging output (optional)
%                If not specified then logFileId = 1 (stdout)
%
% Return Parameters:
% None
%
% Examples:
% WritePgmFile( image, 'DUT_1_AV_On_D65_10bit_10.bmp_AvgOut.raw', 5 )
% WritePgmFile( image, 'DUT_1_AV_On_D65_10bit_10.bmp_AvgOut.raw', 5, logFileId )
%
%  Author: Donald BAXTER, donald.baxter@st.com
%  Date:   24 September 2008
%
%  Copyright (c) 2008 STMicroelectronics

    % If logFileId not specified then set to 1 (stdout)
    if nargin < 5, logFileId = 1; end

    % Attempt to open Binary Pgm Image File
    fid = fopen( image_file_name, 'wb' );
 
    % Determine Image width and height
    [image_height image_width] = size( image );
    
    if fid > -1
        
        fprintf( logFileId, 'Writing Pgm File "%s"\n', image_file_name );
    
        % transform image
        image = image';

        % Print Output PGM File header
        % P2 = ASCII Data, P5 = Binary Data
        fprintf( fid, 'P%d\n',    pgmFileType );
        fprintf( fid, '# "%s"\n', image_file_name );
        fprintf( fid, '%d %d\n',  image_width, image_height );
        fprintf( fid, '%d\n',     max_code );
        
        % Endianess
        machineFormat = 'ieee-be'; % Big Endian
        % machineFormat = 'ieee-le'; % Little Endian
        
        % Read in raw data
        if pgmFileType == 5
            switch class( image )
                case 'uint16'
                    fwrite( fid, image, 'uint16', machineFormat );
                case 'uint8'
                    fwrite( fid, image, 'uint8',  machineFormat );
                otherwise
                    fprintf( logFileId, 'ERROR! - Unsupported array class = must be either uint8 or uint16 data\n' );
            end
        else
            fprintf( fid, '%d\n', image );
        end
        
        % Close output file
        fclose( fid );
    
    else
        fprintf( logFileId, 'ERROR! - Failed to open Pgm File "%s"\n', image_file_name );
    end
    
end

