/* -*- Mode:C++; c-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 *  Copyright � 2014 by IEEE.
 *
 *  This source file is an essential part of IEEE P1906.1,
 *  Recommended Practice for Nanoscale and Molecular
 *  Communication Framework.
 *  Verbatim copies of this source file may be used and
 *  distributed without restriction. Modifications to this source
 *  file as permitted in IEEE P1906.1 may also be made and
 *  distributed. All other uses require permission from the IEEE
 *  Standards Department (stds-ipr@ieee.org). All other rights
 *  reserved.
 *
 *  This source file is provided on an AS IS basis.
 *  The IEEE disclaims ANY WARRANTY EXPRESS OR IMPLIED INCLUDING
 *  ANY WARRANTY OF MERCHANTABILITY AND FITNESS FOR USE FOR A
 *  PARTICULAR PURPOSE.
 *  The user of the source file shall indemnify and hold
 *  IEEE harmless from any damages or liability arising out of
 *  the use thereof.
 *
 * Author: Giuseppe Piro - Telematics Lab Research Group
 *                         Politecnico di Bari
 *                         giuseppe.piro@poliba.it
 *                         telematics.poliba.it/piro
 */

#include "ns3/log.h"

#include "p1906-em-motion.h"
#include "ns3/p1906-communication-interface.h"
#include "ns3/p1906-message-carrier.h"
#include "ns3/p1906-field.h"
#include "ns3/mobility-model.h"
#include "ns3/p1906-net-device.h"
#include <ns3/spectrum-value.h>
#include "p1906-em-message-carrier.h"

namespace ns3 {

NS_LOG_COMPONENT_DEFINE ("P1906EMMotion");

TypeId P1906EMMotion::GetTypeId (void)
{
  static TypeId tid = TypeId ("ns3::P1906EMMotion")
    .SetParent<P1906Motion> ();
  return tid;
}

P1906EMMotion::P1906EMMotion ()
{
  NS_LOG_FUNCTION (this);
}

P1906EMMotion::~P1906EMMotion ()
{
  NS_LOG_FUNCTION (this);
}


double
P1906EMMotion::ComputePropagationDelay (Ptr<P1906CommunicationInterface> src,
		                                Ptr<P1906CommunicationInterface> dst,
		                                Ptr<P1906MessageCarrier> message,
		                                Ptr<P1906Field> field)
{
  NS_LOG_FUNCTION (this);

  Ptr<MobilityModel> srcMobility = src->GetP1906NetDevice ()->GetNode ()->GetObject<MobilityModel> ();
  Ptr<MobilityModel> dstMobility = dst->GetP1906NetDevice ()->GetNode ()->GetObject<MobilityModel> ();

  double distance = dstMobility->GetDistanceFrom (srcMobility);
  double delay = distance/GetWaveSpeed ();

  NS_LOG_FUNCTION (this << "[d,speed,delay]" << distance << GetWaveSpeed() << delay);

  return delay;
}


Ptr<P1906MessageCarrier>
P1906EMMotion::CalculateReceivedMessageCarrier(Ptr<P1906CommunicationInterface> src,
		                                       Ptr<P1906CommunicationInterface> dst,
		                                       Ptr<P1906MessageCarrier> message,
		                                       Ptr<P1906Field> field)
{
  NS_LOG_FUNCTION (this << "Compute the path loss model through Akram's contribution");

  double distances[1000] = {
  		 0.0001, 0.0006004, 0.0011008, 0.0016012, 0.0021016, 0.002602, 0.0031024, 0.0036028, 0.0041032, 0.0046036, 0.005104, 0.0056044, 0.0061048, 0.00660521, 0.00710561, 0.00760601, 0.00810641, 0.00860681, 0.00910721, 0.00960761, 0.010108, 0.0106084, 0.0111088, 0.0116092, 0.0121096, 0.01261, 0.0131104, 0.0136108, 0.0141112, 0.0146116, 0.015112, 0.0156124, 0.0161128, 0.0166132, 0.0171136, 0.017614, 0.0181144, 0.0186148, 0.0191152, 0.0196156, 0.020116, 0.0206164, 0.0211168, 0.0216172, 0.0221176, 0.022618, 0.0231184, 0.0236188, 0.0241192, 0.0246196, 0.02512, 0.0256204, 0.0261208, 0.0266212, 0.0271216, 0.027622, 0.0281224, 0.0286228, 0.0291232, 0.0296236, 0.030124, 0.0306244, 0.0311248, 0.0316252, 0.0321256, 0.032626, 0.0331264, 0.0336268, 0.0341272, 0.0346276, 0.035128, 0.0356284, 0.0361288, 0.0366292, 0.0371296, 0.03763, 0.0381304, 0.0386308, 0.0391312, 0.0396316, 0.040132, 0.0406324, 0.0411328, 0.0416332, 0.0421336, 0.042634, 0.0431344, 0.0436348, 0.0441352, 0.0446356, 0.045136, 0.0456364, 0.0461368, 0.0466372, 0.0471376, 0.047638, 0.0481384, 0.0486388, 0.0491392, 0.0496396, 0.05014, 0.0506404, 0.0511408, 0.0516412, 0.0521416, 0.052642, 0.0531424, 0.0536428, 0.0541432, 0.0546436, 0.055144, 0.0556444, 0.0561448, 0.0566452, 0.0571456, 0.057646, 0.0581464, 0.0586468, 0.0591472, 0.0596476, 0.060148, 0.0606484, 0.0611488, 0.0616492, 0.0621496, 0.0626501, 0.0631505, 0.0636509, 0.0641513, 0.0646517, 0.0651521, 0.0656525, 0.0661529, 0.0666533, 0.0671537, 0.0676541, 0.0681545, 0.0686549, 0.0691553, 0.0696557, 0.0701561, 0.0706565, 0.0711569, 0.0716573, 0.0721577, 0.0726581, 0.0731585, 0.0736589, 0.0741593, 0.0746597, 0.0751601, 0.0756605, 0.0761609, 0.0766613, 0.0771617, 0.0776621, 0.0781625, 0.0786629, 0.0791633, 0.0796637, 0.0801641, 0.0806645, 0.0811649, 0.0816653, 0.0821657, 0.0826661, 0.0831665, 0.0836669, 0.0841673, 0.0846677, 0.0851681, 0.0856685, 0.0861689, 0.0866693, 0.0871697, 0.0876701, 0.0881705, 0.0886709, 0.0891713, 0.0896717, 0.0901721, 0.0906725, 0.0911729, 0.0916733, 0.0921737, 0.0926741, 0.0931745, 0.0936749, 0.0941753, 0.0946757, 0.0951761, 0.0956765, 0.0961769, 0.0966773, 0.0971777, 0.0976781, 0.0981785, 0.0986789, 0.0991793, 0.0996797, 0.10018, 0.10068, 0.101181, 0.101681, 0.102182, 0.102682, 0.103182, 0.103683, 0.104183, 0.104684, 0.105184, 0.105684, 0.106185, 0.106685, 0.107186, 0.107686, 0.108186, 0.108687, 0.109187, 0.109688, 0.110188, 0.110688, 0.111189, 0.111689, 0.11219, 0.11269, 0.11319, 0.113691, 0.114191, 0.114692, 0.115192, 0.115692, 0.116193, 0.116693, 0.117194, 0.117694, 0.118194, 0.118695, 0.119195, 0.119696, 0.120196, 0.120696, 0.121197, 0.121697, 0.122198, 0.122698, 0.123198, 0.123699, 0.124199, 0.1247, 0.1252, 0.125701, 0.126201, 0.126701, 0.127202, 0.127702, 0.128203, 0.128703, 0.129203, 0.129704, 0.130204, 0.130705, 0.131205, 0.131705, 0.132206, 0.132706, 0.133207, 0.133707, 0.134207, 0.134708, 0.135208, 0.135709, 0.136209, 0.136709, 0.13721, 0.13771, 0.138211, 0.138711, 0.139211, 0.139712, 0.140212, 0.140713, 0.141213, 0.141713, 0.142214, 0.142714, 0.143215, 0.143715, 0.144215, 0.144716, 0.145216, 0.145717, 0.146217, 0.146717, 0.147218, 0.147718, 0.148219, 0.148719, 0.149219, 0.14972, 0.15022, 0.150721, 0.151221, 0.151721, 0.152222, 0.152722, 0.153223, 0.153723, 0.154223, 0.154724, 0.155224, 0.155725, 0.156225, 0.156725, 0.157226, 0.157726, 0.158227, 0.158727, 0.159227, 0.159728, 0.160228, 0.160729, 0.161229, 0.161729, 0.16223, 0.16273, 0.163231, 0.163731, 0.164231, 0.164732, 0.165232, 0.165733, 0.166233, 0.166733, 0.167234, 0.167734, 0.168235, 0.168735, 0.169235, 0.169736, 0.170236, 0.170737, 0.171237, 0.171737, 0.172238, 0.172738, 0.173239, 0.173739, 0.174239, 0.17474, 0.17524, 0.175741, 0.176241, 0.176741, 0.177242, 0.177742, 0.178243, 0.178743, 0.179243, 0.179744, 0.180244, 0.180745, 0.181245, 0.181745, 0.182246, 0.182746, 0.183247, 0.183747, 0.184247, 0.184748, 0.185248, 0.185749, 0.186249, 0.186749, 0.18725, 0.18775, 0.188251, 0.188751, 0.189251, 0.189752, 0.190252, 0.190753, 0.191253, 0.191753, 0.192254, 0.192754, 0.193255, 0.193755, 0.194255, 0.194756, 0.195256, 0.195757, 0.196257, 0.196757, 0.197258, 0.197758, 0.198259, 0.198759, 0.199259, 0.19976, 0.20026, 0.200761, 0.201261, 0.201761, 0.202262, 0.202762, 0.203263, 0.203763, 0.204263, 0.204764, 0.205264, 0.205765, 0.206265, 0.206765, 0.207266, 0.207766, 0.208267, 0.208767, 0.209267, 0.209768, 0.210268, 0.210769, 0.211269, 0.211769, 0.21227, 0.21277, 0.213271, 0.213771, 0.214271, 0.214772, 0.215272, 0.215773, 0.216273, 0.216773, 0.217274, 0.217774, 0.218275, 0.218775, 0.219275, 0.219776, 0.220276, 0.220777, 0.221277, 0.221777, 0.222278, 0.222778, 0.223279, 0.223779, 0.224279, 0.22478, 0.22528, 0.225781, 0.226281, 0.226781, 0.227282, 0.227782, 0.228283, 0.228783, 0.229283, 0.229784, 0.230284, 0.230785, 0.231285, 0.231785, 0.232286, 0.232786, 0.233287, 0.233787, 0.234287, 0.234788, 0.235288, 0.235789, 0.236289, 0.236789, 0.23729, 0.23779, 0.238291, 0.238791, 0.239291, 0.239792, 0.240292, 0.240793, 0.241293, 0.241793, 0.242294, 0.242794, 0.243295, 0.243795, 0.244295, 0.244796, 0.245296, 0.245797, 0.246297, 0.246797, 0.247298, 0.247798, 0.248299, 0.248799, 0.249299, 0.2498, 0.2503, 0.250801, 0.251301, 0.251801, 0.252302, 0.252802, 0.253303, 0.253803, 0.254303, 0.254804, 0.255304, 0.255805, 0.256305, 0.256805, 0.257306, 0.257806, 0.258307, 0.258807, 0.259307, 0.259808, 0.260308, 0.260809, 0.261309, 0.261809, 0.26231, 0.26281, 0.263311, 0.263811, 0.264311, 0.264812, 0.265312, 0.265813, 0.266313, 0.266813, 0.267314, 0.267814, 0.268315, 0.268815, 0.269315, 0.269816, 0.270316, 0.270817, 0.271317, 0.271817, 0.272318, 0.272818, 0.273319, 0.273819, 0.274319, 0.27482, 0.27532, 0.275821, 0.276321, 0.276821, 0.277322, 0.277822, 0.278323, 0.278823, 0.279323, 0.279824, 0.280324, 0.280825, 0.281325, 0.281825, 0.282326, 0.282826, 0.283327, 0.283827, 0.284327, 0.284828, 0.285328, 0.285829, 0.286329, 0.286829, 0.28733, 0.28783, 0.288331, 0.288831, 0.289331, 0.289832, 0.290332, 0.290833, 0.291333, 0.291833, 0.292334, 0.292834, 0.293335, 0.293835, 0.294335, 0.294836, 0.295336, 0.295837, 0.296337, 0.296837, 0.297338, 0.297838, 0.298339, 0.298839, 0.299339, 0.29984, 0.30034, 0.300841, 0.301341, 0.301841, 0.302342, 0.302842, 0.303343, 0.303843, 0.304343, 0.304844, 0.305344, 0.305845, 0.306345, 0.306845, 0.307346, 0.307846, 0.308347, 0.308847, 0.309347, 0.309848, 0.310348, 0.310849, 0.311349, 0.311849, 0.31235, 0.31285, 0.313351, 0.313851, 0.314351, 0.314852, 0.315352, 0.315853, 0.316353, 0.316853, 0.317354, 0.317854, 0.318355, 0.318855, 0.319355, 0.319856, 0.320356, 0.320857, 0.321357, 0.321857, 0.322358, 0.322858, 0.323359, 0.323859, 0.324359, 0.32486, 0.32536, 0.325861, 0.326361, 0.326861, 0.327362, 0.327862, 0.328363, 0.328863, 0.329363, 0.329864, 0.330364, 0.330865, 0.331365, 0.331865, 0.332366, 0.332866, 0.333367, 0.333867, 0.334367, 0.334868, 0.335368, 0.335869, 0.336369, 0.336869, 0.33737, 0.33787, 0.338371, 0.338871, 0.339371, 0.339872, 0.340372, 0.340873, 0.341373, 0.341873, 0.342374, 0.342874, 0.343375, 0.343875, 0.344375, 0.344876, 0.345376, 0.345877, 0.346377, 0.346877, 0.347378, 0.347878, 0.348379, 0.348879, 0.349379, 0.34988, 0.35038, 0.350881, 0.351381, 0.351881, 0.352382, 0.352882, 0.353383, 0.353883, 0.354383, 0.354884, 0.355384, 0.355885, 0.356385, 0.356885, 0.357386, 0.357886, 0.358387, 0.358887, 0.359387, 0.359888, 0.360388, 0.360889, 0.361389, 0.361889, 0.36239, 0.36289, 0.363391, 0.363891, 0.364391, 0.364892, 0.365392, 0.365893, 0.366393, 0.366893, 0.367394, 0.367894, 0.368395, 0.368895, 0.369395, 0.369896, 0.370396, 0.370897, 0.371397, 0.371897, 0.372398, 0.372898, 0.373399, 0.373899, 0.374399, 0.3749, 0.3754, 0.375901, 0.376401, 0.376902, 0.377402, 0.377902, 0.378403, 0.378903, 0.379404, 0.379904, 0.380404, 0.380905, 0.381405, 0.381906, 0.382406, 0.382906, 0.383407, 0.383907, 0.384408, 0.384908, 0.385408, 0.385909, 0.386409, 0.38691, 0.38741, 0.38791, 0.388411, 0.388911, 0.389412, 0.389912, 0.390412, 0.390913, 0.391413, 0.391914, 0.392414, 0.392914, 0.393415, 0.393915, 0.394416, 0.394916, 0.395416, 0.395917, 0.396417, 0.396918, 0.397418, 0.397918, 0.398419, 0.398919, 0.39942, 0.39992, 0.40042, 0.400921, 0.401421, 0.401922, 0.402422, 0.402922, 0.403423, 0.403923, 0.404424, 0.404924, 0.405424, 0.405925, 0.406425, 0.406926, 0.407426, 0.407926, 0.408427, 0.408927, 0.409428, 0.409928, 0.410428, 0.410929, 0.411429, 0.41193, 0.41243, 0.41293, 0.413431, 0.413931, 0.414432, 0.414932, 0.415432, 0.415933, 0.416433, 0.416934, 0.417434, 0.417934, 0.418435, 0.418935, 0.419436, 0.419936, 0.420436, 0.420937, 0.421437, 0.421938, 0.422438, 0.422938, 0.423439, 0.423939, 0.42444, 0.42494, 0.42544, 0.425941, 0.426441, 0.426942, 0.427442, 0.427942, 0.428443, 0.428943, 0.429444, 0.429944, 0.430444, 0.430945, 0.431445, 0.431946, 0.432446, 0.432946, 0.433447, 0.433947, 0.434448, 0.434948, 0.435448, 0.435949, 0.436449, 0.43695, 0.43745, 0.43795, 0.438451, 0.438951, 0.439452, 0.439952, 0.440452, 0.440953, 0.441453, 0.441954, 0.442454, 0.442954, 0.443455, 0.443955, 0.444456, 0.444956, 0.445456, 0.445957, 0.446457, 0.446958, 0.447458, 0.447958, 0.448459, 0.448959, 0.44946, 0.44996, 0.45046, 0.450961, 0.451461, 0.451962, 0.452462, 0.452962, 0.453463, 0.453963, 0.454464, 0.454964, 0.455464, 0.455965, 0.456465, 0.456966, 0.457466, 0.457966, 0.458467, 0.458967, 0.459468, 0.459968, 0.460468, 0.460969, 0.461469, 0.46197, 0.46247, 0.46297, 0.463471, 0.463971, 0.464472, 0.464972, 0.465472, 0.465973, 0.466473, 0.466974, 0.467474, 0.467974, 0.468475, 0.468975, 0.469476, 0.469976, 0.470476, 0.470977, 0.471477, 0.471978, 0.472478, 0.472978, 0.473479, 0.473979, 0.47448, 0.47498, 0.47548, 0.475981, 0.476481, 0.476982, 0.477482, 0.477982, 0.478483, 0.478983, 0.479484, 0.479984, 0.480484, 0.480985, 0.481485, 0.481986, 0.482486, 0.482986, 0.483487, 0.483987, 0.484488, 0.484988, 0.485488, 0.485989, 0.486489, 0.48699, 0.48749, 0.48799, 0.488491, 0.488991, 0.489492, 0.489992, 0.490492, 0.490993, 0.491493, 0.491994, 0.492494, 0.492994, 0.493495, 0.493995, 0.494496, 0.494996, 0.495496, 0.495997, 0.496497, 0.496998, 0.497498, 0.497998, 0.498499, 0.498999, 0.4995, 0.5,
  		 };

  double pathloss[1000][11] = {
   {
  12.3394, 14.5749, 15.9138, 17.291, 18.314, 19.4465, 20.9262, 21.8993, 23.0291, 24.3247, 24.2721
   },
  {
  39.8692, 45.3669, 46.7058, 49.1703, 50.1933, 52.4132, 57.155, 59.2154, 62.52, 67.0777, 63.763
   },
  {
  57.0956, 65.8553, 67.1943, 70.7461, 71.7692, 75.0764, 83.0803, 86.2281, 91.7074, 99.5272, 92.9504
   },
  {
  72.3114, 84.3332, 85.6722, 90.3114, 91.3344, 95.729, 106.995, 111.23, 118.884, 129.966, 120.127
   },
  {
  86.6345, 101.918, 103.257, 108.984, 110.007, 115.489, 130.017, 135.34, 145.168, 159.512, 146.411
   },
  {
  100.451, 118.997, 120.336, 127.15, 128.173, 134.742, 152.532, 158.942, 170.946, 188.552, 172.189
   },
  {
  113.94, 135.748, 137.087, 144.988, 146.011, 153.668, 174.72, 182.217, 196.396, 217.264, 197.638
   },
  {
  127.199, 152.27, 153.609, 162.597, 163.62, 172.364, 196.679, 205.263, 221.616, 245.747, 222.859
   },
  {
  140.29, 168.622, 169.961, 180.037, 181.06, 190.892, 218.468, 228.14, 246.668, 274.061, 247.911
   },
  {
  153.251, 184.845, 186.184, 197.347, 198.37, 209.289, 240.128, 250.887, 271.59, 302.244, 272.833
   },
  {
  166.108, 200.964, 202.303, 214.554, 215.577, 227.583, 261.684, 273.531, 296.408, 330.325, 297.651
   },
  {
  178.881, 217, 218.339, 231.677, 232.7, 245.794, 283.156, 296.091, 321.142, 358.321, 322.385
   },
  {
  191.585, 232.966, 234.305, 248.73, 249.753, 263.934, 304.559, 318.581, 345.807, 386.248, 347.05
   },
  {
  204.231, 248.873, 250.212, 265.725, 266.748, 282.016, 325.904, 341.012, 370.414, 414.117, 371.657
   },
  {
  216.826, 264.731, 266.07, 282.67, 283.693, 300.049, 347.198, 363.394, 394.97, 441.935, 396.213
   },
  {
  229.378, 280.545, 281.884, 299.572, 300.595, 318.038, 368.449, 385.732, 419.483, 469.71, 420.726
   },
  {
  241.893, 296.322, 297.661, 316.436, 317.459, 335.989, 389.662, 408.033, 443.959, 497.448, 445.202
   },
  {
  254.374, 312.065, 313.404, 333.266, 334.289, 353.907, 410.843, 430.301, 468.401, 525.152, 469.644
   },
  {
  266.826, 327.779, 329.118, 350.068, 351.091, 371.796, 431.993, 452.539, 492.814, 552.827, 494.057
   },
  {
  279.251, 343.467, 344.806, 366.843, 367.866, 389.658, 453.118, 474.751, 517.201, 580.476, 518.444
   },
  {
  291.653, 359.131, 360.47, 383.594, 384.617, 407.497, 474.219, 496.939, 541.564, 608.101, 542.807
   },
  {
  304.034, 374.774, 376.113, 400.325, 401.348, 425.315, 495.299, 519.106, 565.906, 635.705, 567.149
   },
  {
  316.395, 390.397, 391.736, 417.035, 418.058, 443.113, 516.359, 541.254, 590.228, 663.29, 591.471
   },
  {
  328.739, 406.003, 407.342, 433.729, 434.752, 460.894, 537.402, 563.384, 614.533, 690.857, 615.776
   },
  {
  341.067, 421.593, 422.932, 450.406, 451.429, 478.658, 558.428, 585.498, 638.821, 718.407, 640.064
   },
  {
  353.38, 437.168, 438.507, 467.068, 468.091, 496.408, 579.44, 607.597, 663.095, 745.943, 664.338
   },
  {
  365.679, 452.729, 454.068, 483.716, 484.739, 514.143, 600.438, 629.682, 687.355, 773.465, 688.598
   },
  {
  377.965, 468.277, 469.616, 500.352, 501.375, 531.867, 621.423, 651.755, 711.603, 800.975, 712.846
   },
  {
  390.24, 483.814, 485.153, 516.976, 517.999, 549.578, 642.397, 673.816, 735.838, 828.473, 737.081
   },
  {
  402.503, 499.34, 500.679, 533.589, 534.613, 567.279, 663.359, 695.866, 760.063, 855.96, 761.306
   },
  {
  414.757, 514.855, 516.194, 550.192, 551.216, 584.969, 684.312, 717.906, 784.278, 883.436, 785.521
   },
  {
  427.001, 530.361, 531.7, 566.786, 567.809, 602.65, 705.255, 739.936, 808.483, 910.903, 809.726
   },
  {
  439.236, 545.859, 547.198, 583.37, 584.394, 620.322, 726.189, 761.957, 832.679, 938.362, 833.922
   },
  {
  451.463, 561.347, 562.686, 599.947, 600.97, 637.985, 747.114, 783.97, 856.867, 965.811, 858.109
   },
  {
  463.681, 576.828, 578.167, 616.515, 617.538, 655.641, 768.032, 805.976, 881.046, 993.253, 882.289
   },
  {
  475.893, 592.302, 593.641, 633.076, 634.099, 673.289, 788.942, 827.973, 905.219, 1020.69, 906.462
   },
  {
  488.097, 607.768, 609.107, 649.63, 650.653, 690.93, 809.846, 849.964, 929.384, 1048.12, 930.627
   },
  {
  500.295, 623.228, 624.567, 666.177, 667.2, 708.565, 830.742, 871.948, 953.543, 1075.54, 954.786
   },
  {
  512.486, 638.682, 640.021, 682.718, 683.741, 726.193, 851.633, 893.926, 977.695, 1102.95, 978.938
   },
  {
  524.672, 654.129, 655.468, 699.253, 700.276, 743.815, 872.517, 915.897, 1001.84, 1130.36, 1003.08
   },
  {
  536.852, 669.571, 670.91, 715.782, 716.805, 761.432, 893.396, 937.864, 1025.98, 1157.76, 1027.23
   },
  {
  549.026, 685.008, 686.347, 732.306, 733.329, 779.043, 914.269, 959.824, 1050.12, 1185.16, 1051.36
   },
  {
  561.195, 700.439, 701.778, 748.825, 749.848, 796.65, 935.138, 981.78, 1074.25, 1212.55, 1075.49
   },
  {
  573.36, 715.866, 717.205, 765.339, 766.362, 814.251, 956.001, 1003.73, 1098.37, 1239.94, 1099.62
   },
  {
  585.52, 731.288, 732.627, 781.848, 782.871, 831.847, 976.86, 1025.68, 1122.49, 1267.32, 1123.74
   },
  {
  597.675, 746.705, 748.044, 798.353, 799.376, 849.44, 997.714, 1047.62, 1146.61, 1294.7, 1147.85
   },
  {
  609.826, 762.118, 763.457, 814.853, 815.876, 867.028, 1018.56, 1069.56, 1170.72, 1322.08, 1171.97
   },
  {
  621.973, 777.527, 778.866, 831.35, 832.373, 884.611, 1039.41, 1091.49, 1194.83, 1349.45, 1196.07
   },
  {
  634.116, 792.933, 794.272, 847.842, 848.865, 902.191, 1060.25, 1113.42, 1218.94, 1376.81, 1220.18
   },
  {
  646.256, 808.334, 809.673, 864.331, 865.354, 919.768, 1081.09, 1135.34, 1243.04, 1404.17, 1244.28
   },
  {
  658.392, 823.732, 825.071, 880.816, 881.84, 937.34, 1101.93, 1157.27, 1267.13, 1431.53, 1268.38
   },
  {
  670.524, 839.127, 840.465, 897.298, 898.321, 954.91, 1122.76, 1179.19, 1291.23, 1458.89, 1292.47
   },
  {
  682.653, 854.518, 855.857, 913.777, 914.8, 972.475, 1143.58, 1201.1, 1315.32, 1486.24, 1316.56
   },
  {
  694.779, 869.906, 871.245, 930.252, 931.275, 990.038, 1164.41, 1223.01, 1339.4, 1513.59, 1340.65
   },
  {
  706.902, 885.291, 886.629, 946.724, 947.747, 1007.6, 1185.23, 1244.92, 1363.49, 1540.94, 1364.73
   },
  {
  719.021, 900.672, 902.011, 963.194, 964.217, 1025.15, 1206.05, 1266.83, 1387.57, 1568.28, 1388.81
   },
  {
  731.138, 916.052, 917.39, 979.66, 980.683, 1042.71, 1226.87, 1288.73, 1411.65, 1595.62, 1412.89
   },
  {
  743.253, 931.428, 932.767, 996.124, 997.147, 1060.26, 1247.68, 1310.63, 1435.72, 1622.96, 1436.96
   },
  {
  755.364, 946.802, 948.14, 1012.58, 1013.61, 1077.81, 1268.49, 1332.53, 1459.79, 1650.29, 1461.04
   },
  {
  767.473, 962.173, 963.512, 1029.04, 1030.07, 1095.35, 1289.3, 1354.42, 1483.86, 1677.62, 1485.11
   },
  {
  779.58, 977.541, 978.88, 1045.5, 1046.52, 1112.9, 1310.1, 1376.32, 1507.93, 1704.95, 1509.17
   },
  {
  791.684, 992.908, 994.246, 1061.95, 1062.98, 1130.44, 1330.91, 1398.21, 1532, 1732.28, 1533.24
   },
  {
  803.786, 1008.27, 1009.61, 1078.4, 1079.43, 1147.98, 1351.71, 1420.1, 1556.06, 1759.61, 1557.3
   },
  {
  815.885, 1023.63, 1024.97, 1094.85, 1095.88, 1165.51, 1372.5, 1441.98, 1580.12, 1786.93, 1581.36
   },
  {
  827.983, 1038.99, 1040.33, 1111.3, 1112.32, 1183.05, 1393.3, 1463.87, 1604.18, 1814.25, 1605.42
   },
  {
  840.078, 1054.35, 1055.69, 1127.74, 1128.77, 1200.58, 1414.1, 1485.75, 1628.23, 1841.57, 1629.48
   },
  {
  852.171, 1069.71, 1071.04, 1144.19, 1145.21, 1218.11, 1434.89, 1507.63, 1652.29, 1868.88, 1653.53
   },
  {
  864.262, 1085.06, 1086.4, 1160.63, 1161.65, 1235.64, 1455.68, 1529.5, 1676.34, 1896.2, 1677.58
   },
  {
  876.352, 1100.41, 1101.75, 1177.07, 1178.09, 1253.16, 1476.47, 1551.38, 1700.39, 1923.51, 1701.63
   },
  {
  888.439, 1115.76, 1117.1, 1193.5, 1194.53, 1270.69, 1497.25, 1573.25, 1724.44, 1950.82, 1725.68
   },
  {
  900.525, 1131.11, 1132.45, 1209.94, 1210.96, 1288.21, 1518.04, 1595.13, 1748.49, 1978.13, 1749.73
   },
  {
  912.609, 1146.45, 1147.79, 1226.37, 1227.4, 1305.73, 1538.82, 1617, 1772.53, 2005.44, 1773.77
   },
  {
  924.691, 1161.8, 1163.14, 1242.8, 1243.83, 1323.25, 1559.6, 1638.86, 1796.58, 2032.74, 1797.82
   },
  {
  936.771, 1177.14, 1178.48, 1259.23, 1260.26, 1340.77, 1580.38, 1660.73, 1820.62, 2060.05, 1821.86
   },
  {
  948.85, 1192.48, 1193.82, 1275.66, 1276.69, 1358.28, 1601.16, 1682.6, 1844.66, 2087.35, 1845.9
   },
  {
  960.928, 1207.82, 1209.16, 1292.09, 1293.11, 1375.8, 1621.93, 1704.46, 1868.7, 2114.65, 1869.94
   },
  {
  973.003, 1223.16, 1224.5, 1308.51, 1309.54, 1393.31, 1642.71, 1726.32, 1892.73, 2141.95, 1893.97
   },
  {
  985.078, 1238.5, 1239.83, 1324.94, 1325.96, 1410.82, 1663.48, 1748.18, 1916.77, 2169.24, 1918.01
   },
  {
  997.151, 1253.83, 1255.17, 1341.36, 1342.38, 1428.33, 1684.25, 1770.04, 1940.8, 2196.54, 1942.04
   },
  {
  1009.22, 1269.16, 1270.5, 1357.78, 1358.8, 1445.84, 1705.02, 1791.9, 1964.83, 2223.84, 1966.08
   },
  {
  1021.29, 1284.5, 1285.83, 1374.2, 1375.22, 1463.35, 1725.79, 1813.76, 1988.86, 2251.13, 1990.11
   },
  {
  1033.36, 1299.83, 1301.17, 1390.62, 1391.64, 1480.85, 1746.56, 1835.61, 2012.89, 2278.42, 2014.14
   },
  {
  1045.43, 1315.16, 1316.49, 1407.04, 1408.06, 1498.36, 1767.33, 1857.46, 2036.92, 2305.71, 2038.17
   },
  {
  1057.49, 1330.48, 1331.82, 1423.45, 1424.47, 1515.86, 1788.09, 1879.32, 2060.95, 2333, 2062.19
   },
  {
  1069.56, 1345.81, 1347.15, 1439.87, 1440.89, 1533.36, 1808.86, 1901.17, 2084.98, 2360.29, 2086.22
   },
  {
  1081.62, 1361.14, 1362.48, 1456.28, 1457.3, 1550.86, 1829.62, 1923.02, 2109, 2387.57, 2110.24
   },
  {
  1093.68, 1376.46, 1377.8, 1472.69, 1473.71, 1568.36, 1850.38, 1944.87, 2133.02, 2414.86, 2134.27
   },
  {
  1105.75, 1391.78, 1393.12, 1489.1, 1490.12, 1585.86, 1871.14, 1966.71, 2157.05, 2442.14, 2158.29
   },
  {
  1117.81, 1407.11, 1408.45, 1505.51, 1506.53, 1603.35, 1891.9, 1988.56, 2181.07, 2469.43, 2182.31
   },
  {
  1129.87, 1422.43, 1423.77, 1521.92, 1522.94, 1620.85, 1912.66, 2010.41, 2205.09, 2496.71, 2206.33
   },
  {
  1141.92, 1437.75, 1439.09, 1538.33, 1539.35, 1638.35, 1933.41, 2032.25, 2229.11, 2523.99, 2230.35
   },
  {
  1153.98, 1453.07, 1454.41, 1554.73, 1555.76, 1655.84, 1954.17, 2054.09, 2253.12, 2551.27, 2254.37
   },
  {
  1166.04, 1468.38, 1469.72, 1571.14, 1572.16, 1673.33, 1974.92, 2075.94, 2277.14, 2578.55, 2278.38
   },
  {
  1178.09, 1483.7, 1485.04, 1587.54, 1588.57, 1690.82, 1995.68, 2097.78, 2301.16, 2605.83, 2302.4
   },
  {
  1190.14, 1499.02, 1500.36, 1603.95, 1604.97, 1708.31, 2016.43, 2119.62, 2325.17, 2633.1, 2326.41
   },
  {
  1202.2, 1514.33, 1515.67, 1620.35, 1621.37, 1725.8, 2037.18, 2141.46, 2349.18, 2660.38, 2350.43
   },
  {
  1214.25, 1529.65, 1530.98, 1636.75, 1637.77, 1743.29, 2057.93, 2163.29, 2373.2, 2687.65, 2374.44
   },
  {
  1226.3, 1544.96, 1546.3, 1653.15, 1654.17, 1760.78, 2078.68, 2185.13, 2397.21, 2714.93, 2398.45
   },
  {
  1238.35, 1560.27, 1561.61, 1669.55, 1670.57, 1778.27, 2099.43, 2206.97, 2421.22, 2742.2, 2422.46
   },
  {
  1250.4, 1575.58, 1576.92, 1685.95, 1686.97, 1795.75, 2120.18, 2228.8, 2445.23, 2769.47, 2446.47
   },
  {
  1262.45, 1590.89, 1592.23, 1702.34, 1703.37, 1813.24, 2140.93, 2250.64, 2469.24, 2796.75, 2470.48
   },
  {
  1274.49, 1606.2, 1607.54, 1718.74, 1719.76, 1830.72, 2161.67, 2272.47, 2493.25, 2824.02, 2494.49
   },
  {
  1286.54, 1621.51, 1622.85, 1735.14, 1736.16, 1848.2, 2182.42, 2294.3, 2517.26, 2851.29, 2518.5
   },
  {
  1298.59, 1636.82, 1638.16, 1751.53, 1752.56, 1865.69, 2203.16, 2316.13, 2541.26, 2878.55, 2542.51
   },
  {
  1310.63, 1652.12, 1653.46, 1767.93, 1768.95, 1883.17, 2223.91, 2337.97, 2565.27, 2905.82, 2566.51
   },
  {
  1322.67, 1667.43, 1668.77, 1784.32, 1785.34, 1900.65, 2244.65, 2359.8, 2589.27, 2933.09, 2590.52
   },
  {
  1334.72, 1682.74, 1684.08, 1800.71, 1801.74, 1918.13, 2265.39, 2381.63, 2613.28, 2960.36, 2614.52
   },
  {
  1346.76, 1698.04, 1699.38, 1817.1, 1818.13, 1935.61, 2286.13, 2403.45, 2637.28, 2987.62, 2638.52
   },
  {
  1358.8, 1713.34, 1714.68, 1833.5, 1834.52, 1953.09, 2306.87, 2425.28, 2661.28, 3014.89, 2662.53
   },
  {
  1370.84, 1728.65, 1729.99, 1849.89, 1850.91, 1970.56, 2327.61, 2447.11, 2685.29, 3042.15, 2686.53
   },
  {
  1382.88, 1743.95, 1745.29, 1866.28, 1867.3, 1988.04, 2348.35, 2468.94, 2709.29, 3069.41, 2710.53
   },
  {
  1394.92, 1759.25, 1760.59, 1882.67, 1883.69, 2005.52, 2369.09, 2490.76, 2733.29, 3096.68, 2734.53
   },
  {
  1406.96, 1774.55, 1775.89, 1899.05, 1900.08, 2022.99, 2389.83, 2512.59, 2757.29, 3123.94, 2758.53
   },
  {
  1419, 1789.85, 1791.19, 1915.44, 1916.46, 2040.47, 2410.57, 2534.41, 2781.29, 3151.2, 2782.53
   },
  {
  1431.04, 1805.15, 1806.49, 1931.83, 1932.85, 2057.94, 2431.3, 2556.24, 2805.28, 3178.46, 2806.53
   },
  {
  1443.07, 1820.45, 1821.79, 1948.21, 1949.24, 2075.42, 2452.04, 2578.06, 2829.28, 3205.72, 2830.53
   },
  {
  1455.11, 1835.75, 1837.09, 1964.6, 1965.62, 2092.89, 2472.77, 2599.88, 2853.28, 3232.98, 2854.52
   },
  {
  1467.15, 1851.05, 1852.39, 1980.98, 1982.01, 2110.36, 2493.51, 2621.7, 2877.28, 3260.24, 2878.52
   },
  {
  1479.18, 1866.34, 1867.68, 1997.37, 1998.39, 2127.83, 2514.24, 2643.52, 2901.27, 3287.5, 2902.52
   },
  {
  1491.21, 1881.64, 1882.98, 2013.75, 2014.78, 2145.3, 2534.97, 2665.34, 2925.27, 3314.75, 2926.51
   },
  {
  1503.25, 1896.94, 1898.27, 2030.14, 2031.16, 2162.78, 2555.71, 2687.16, 2949.26, 3342.01, 2950.5
   },
  {
  1515.28, 1912.23, 1913.57, 2046.52, 2047.54, 2180.25, 2576.44, 2708.98, 2973.26, 3369.27, 2974.5
   },
  {
  1527.31, 1927.53, 1928.86, 2062.9, 2063.92, 2197.71, 2597.17, 2730.8, 2997.25, 3396.52, 2998.49
   },
  {
  1539.35, 1942.82, 1944.16, 2079.28, 2080.3, 2215.18, 2617.9, 2752.62, 3021.24, 3423.78, 3022.49
   },
  {
  1551.38, 1958.11, 1959.45, 2095.66, 2096.69, 2232.65, 2638.63, 2774.44, 3045.23, 3451.03, 3046.48
   },
  {
  1563.41, 1973.41, 1974.74, 2112.04, 2113.07, 2250.12, 2659.36, 2796.25, 3069.23, 3478.28, 3070.47
   },
  {
  1575.44, 1988.7, 1990.04, 2128.42, 2129.44, 2267.59, 2680.09, 2818.07, 3093.22, 3505.54, 3094.46
   },
  {
  1587.47, 2003.99, 2005.33, 2144.8, 2145.82, 2285.05, 2700.82, 2839.89, 3117.21, 3532.79, 3118.45
   },
  {
  1599.5, 2019.28, 2020.62, 2161.18, 2162.2, 2302.52, 2721.55, 2861.7, 3141.2, 3560.04, 3142.44
   },
  {
  1611.52, 2034.57, 2035.91, 2177.56, 2178.58, 2319.98, 2742.27, 2883.52, 3165.19, 3587.29, 3166.43
   },
  {
  1623.55, 2049.86, 2051.2, 2193.93, 2194.96, 2337.45, 2763, 2905.33, 3189.18, 3614.55, 3190.42
   },
  {
  1635.58, 2065.15, 2066.49, 2210.31, 2211.33, 2354.91, 2783.73, 2927.15, 3213.17, 3641.8, 3214.41
   },
  {
  1647.61, 2080.44, 2081.78, 2226.69, 2227.71, 2372.38, 2804.45, 2948.96, 3237.15, 3669.05, 3238.4
   },
  {
  1659.63, 2095.73, 2097.07, 2243.06, 2244.09, 2389.84, 2825.18, 2970.77, 3261.14, 3696.3, 3262.38
   },
  {
  1671.66, 2111.02, 2112.36, 2259.44, 2260.46, 2407.3, 2845.9, 2992.58, 3285.13, 3723.54, 3286.37
   },
  {
  1683.68, 2126.3, 2127.64, 2275.81, 2276.84, 2424.76, 2866.63, 3014.4, 3309.11, 3750.79, 3310.36
   },
  {
  1695.71, 2141.59, 2142.93, 2292.19, 2293.21, 2442.23, 2887.35, 3036.21, 3333.1, 3778.04, 3334.34
   },
  {
  1707.73, 2156.88, 2158.22, 2308.56, 2309.59, 2459.69, 2908.08, 3058.02, 3357.09, 3805.29, 3358.33
   },
  {
  1719.76, 2172.16, 2173.5, 2324.94, 2325.96, 2477.15, 2928.8, 3079.83, 3381.07, 3832.54, 3382.31
   },
  {
  1731.78, 2187.45, 2188.79, 2341.31, 2342.33, 2494.61, 2949.52, 3101.64, 3405.06, 3859.78, 3406.3
   },
  {
  1743.81, 2202.74, 2204.07, 2357.68, 2358.71, 2512.07, 2970.24, 3123.45, 3429.04, 3887.03, 3430.28
   },
  {
  1755.83, 2218.02, 2219.36, 2374.05, 2375.08, 2529.53, 2990.96, 3145.26, 3453.02, 3914.28, 3454.27
   },
  {
  1767.85, 2233.3, 2234.64, 2390.43, 2391.45, 2546.99, 3011.69, 3167.07, 3477.01, 3941.52, 3478.25
   },
  {
  1779.87, 2248.59, 2249.93, 2406.8, 2407.82, 2564.45, 3032.41, 3188.87, 3500.99, 3968.77, 3502.23
   },
  {
  1791.89, 2263.87, 2265.21, 2423.17, 2424.19, 2581.91, 3053.13, 3210.68, 3524.97, 3996.01, 3526.22
   },
  {
  1803.92, 2279.16, 2280.49, 2439.54, 2440.56, 2599.36, 3073.85, 3232.49, 3548.95, 4023.26, 3550.2
   },
  {
  1815.94, 2294.44, 2295.78, 2455.91, 2456.93, 2616.82, 3094.57, 3254.3, 3572.94, 4050.5, 3574.18
   },
  {
  1827.96, 2309.72, 2311.06, 2472.28, 2473.3, 2634.28, 3115.29, 3276.1, 3596.92, 4077.74, 3598.16
   },
  {
  1839.98, 2325, 2326.34, 2488.65, 2489.67, 2651.73, 3136.01, 3297.91, 3620.9, 4104.99, 3622.14
   },
  {
  1852, 2340.28, 2341.62, 2505.02, 2506.04, 2669.19, 3156.72, 3319.71, 3644.88, 4132.23, 3646.12
   },
  {
  1864.01, 2355.56, 2356.9, 2521.39, 2522.41, 2686.65, 3177.44, 3341.52, 3668.86, 4159.47, 3670.1
   },
  {
  1876.03, 2370.85, 2372.18, 2537.75, 2538.78, 2704.1, 3198.16, 3363.32, 3692.84, 4186.71, 3694.08
   },
  {
  1888.05, 2386.13, 2387.47, 2554.12, 2555.15, 2721.56, 3218.88, 3385.13, 3716.82, 4213.95, 3718.06
   },
  {
  1900.07, 2401.41, 2402.75, 2570.49, 2571.51, 2739.01, 3239.59, 3406.93, 3740.8, 4241.19, 3742.04
   },
  {
  1912.09, 2416.69, 2418.02, 2586.86, 2587.88, 2756.47, 3260.31, 3428.74, 3764.78, 4268.44, 3766.02
   },
  {
  1924.1, 2431.97, 2433.3, 2603.22, 2604.25, 2773.92, 3281.03, 3450.54, 3788.75, 4295.68, 3790
   },
  {
  1936.12, 2447.24, 2448.58, 2619.59, 2620.61, 2791.37, 3301.74, 3472.34, 3812.73, 4322.92, 3813.97
   },
  {
  1948.14, 2462.52, 2463.86, 2635.96, 2636.98, 2808.83, 3322.46, 3494.15, 3836.71, 4350.15, 3837.95
   },
  {
  1960.15, 2477.8, 2479.14, 2652.32, 2653.34, 2826.28, 3343.17, 3515.95, 3860.69, 4377.39, 3861.93
   },
  {
  1972.17, 2493.08, 2494.42, 2668.69, 2669.71, 2843.73, 3363.89, 3537.75, 3884.66, 4404.63, 3885.91
   },
  {
  1984.18, 2508.36, 2509.7, 2685.05, 2686.07, 2861.19, 3384.6, 3559.55, 3908.64, 4431.87, 3909.88
   },
  {
  1996.2, 2523.63, 2524.97, 2701.42, 2702.44, 2878.64, 3405.31, 3581.35, 3932.62, 4459.11, 3933.86
   },
  {
  2008.21, 2538.91, 2540.25, 2717.78, 2718.8, 2896.09, 3426.03, 3603.16, 3956.59, 4486.35, 3957.83
   },
  {
  2020.23, 2554.19, 2555.53, 2734.14, 2735.17, 2913.54, 3446.74, 3624.96, 3980.57, 4513.59, 3981.81
   },
  {
  2032.24, 2569.46, 2570.8, 2750.51, 2751.53, 2930.99, 3467.46, 3646.76, 4004.54, 4540.82, 4005.79
   },
  {
  2044.26, 2584.74, 2586.08, 2766.87, 2767.89, 2948.44, 3488.17, 3668.56, 4028.52, 4568.06, 4029.76
   },
  {
  2056.27, 2600.01, 2601.35, 2783.23, 2784.26, 2965.89, 3508.88, 3690.36, 4052.49, 4595.3, 4053.73
   },
  {
  2068.28, 2615.29, 2616.63, 2799.6, 2800.62, 2983.34, 3529.59, 3712.16, 4076.47, 4622.53, 4077.71
   },
  {
  2080.3, 2630.56, 2631.9, 2815.96, 2816.98, 3000.79, 3550.3, 3733.95, 4100.44, 4649.77, 4101.68
   },
  {
  2092.31, 2645.84, 2647.18, 2832.32, 2833.34, 3018.24, 3571.02, 3755.75, 4124.41, 4677, 4125.66
   },
  {
  2104.32, 2661.11, 2662.45, 2848.68, 2849.7, 3035.69, 3591.73, 3777.55, 4148.39, 4704.24, 4149.63
   },
  {
  2116.33, 2676.39, 2677.73, 2865.04, 2866.07, 3053.14, 3612.44, 3799.35, 4172.36, 4731.47, 4173.6
   },
  {
  2128.34, 2691.66, 2693, 2881.4, 2882.43, 3070.59, 3633.15, 3821.15, 4196.33, 4758.71, 4197.58
   },
  {
  2140.36, 2706.93, 2708.27, 2897.77, 2898.79, 3088.04, 3653.86, 3842.95, 4220.3, 4785.94, 4221.55
   },
  {
  2152.37, 2722.21, 2723.55, 2914.13, 2915.15, 3105.48, 3674.57, 3864.74, 4244.28, 4813.18, 4245.52
   },
  {
  2164.38, 2737.48, 2738.82, 2930.49, 2931.51, 3122.93, 3695.28, 3886.54, 4268.25, 4840.41, 4269.49
   },
  {
  2176.39, 2752.75, 2754.09, 2946.85, 2947.87, 3140.38, 3715.99, 3908.34, 4292.22, 4867.65, 4293.46
   },
  {
  2188.4, 2768.03, 2769.36, 2963.21, 2964.23, 3157.82, 3736.7, 3930.13, 4316.19, 4894.88, 4317.43
   },
  {
  2200.41, 2783.3, 2784.64, 2979.56, 2980.59, 3175.27, 3757.41, 3951.93, 4340.16, 4922.11, 4341.41
   },
  {
  2212.42, 2798.57, 2799.91, 2995.92, 2996.95, 3192.72, 3778.11, 3973.73, 4364.13, 4949.34, 4365.38
   },
  {
  2224.43, 2813.84, 2815.18, 3012.28, 3013.31, 3210.16, 3798.82, 3995.52, 4388.1, 4976.58, 4389.35
   },
  {
  2236.44, 2829.11, 2830.45, 3028.64, 3029.66, 3227.61, 3819.53, 4017.32, 4412.07, 5003.81, 4413.32
   },
  {
  2248.45, 2844.38, 2845.72, 3045, 3046.02, 3245.06, 3840.24, 4039.11, 4436.04, 5031.04, 4437.29
   },
  {
  2260.45, 2859.65, 2860.99, 3061.36, 3062.38, 3262.5, 3860.95, 4060.91, 4460.01, 5058.27, 4461.26
   },
  {
  2272.46, 2874.92, 2876.26, 3077.72, 3078.74, 3279.95, 3881.65, 4082.7, 4483.98, 5085.5, 4485.23
   },
  {
  2284.47, 2890.19, 2891.53, 3094.07, 3095.1, 3297.39, 3902.36, 4104.5, 4507.95, 5112.74, 4509.19
   },
  {
  2296.48, 2905.46, 2906.8, 3110.43, 3111.45, 3314.84, 3923.07, 4126.29, 4531.92, 5139.97, 4533.16
   },
  {
  2308.49, 2920.73, 2922.07, 3126.79, 3127.81, 3332.28, 3943.77, 4148.08, 4555.89, 5167.2, 4557.13
   },
  {
  2320.49, 2936, 2937.34, 3143.14, 3144.17, 3349.72, 3964.48, 4169.88, 4579.86, 5194.43, 4581.1
   },
  {
  2332.5, 2951.27, 2952.61, 3159.5, 3160.52, 3367.17, 3985.19, 4191.67, 4603.83, 5221.66, 4605.07
   },
  {
  2344.51, 2966.54, 2967.88, 3175.86, 3176.88, 3384.61, 4005.89, 4213.46, 4627.79, 5248.89, 4629.04
   },
  {
  2356.51, 2981.81, 2983.15, 3192.21, 3193.24, 3402.06, 4026.6, 4235.26, 4651.76, 5276.12, 4653
   },
  {
  2368.52, 2997.08, 2998.42, 3208.57, 3209.59, 3419.5, 4047.3, 4257.05, 4675.73, 5303.35, 4676.97
   },
  {
  2380.53, 3012.35, 3013.69, 3224.92, 3225.95, 3436.94, 4068.01, 4278.84, 4699.7, 5330.58, 4700.94
   },
  {
  2392.53, 3027.61, 3028.95, 3241.28, 3242.3, 3454.38, 4088.71, 4300.63, 4723.66, 5357.81, 4724.91
   },
  {
  2404.54, 3042.88, 3044.22, 3257.63, 3258.66, 3471.83, 4109.42, 4322.43, 4747.63, 5385.03, 4748.87
   },
  {
  2416.54, 3058.15, 3059.49, 3273.99, 3275.01, 3489.27, 4130.12, 4344.22, 4771.6, 5412.26, 4772.84
   },
  {
  2428.55, 3073.42, 3074.76, 3290.34, 3291.37, 3506.71, 4150.82, 4366.01, 4795.56, 5439.49, 4796.8
   },
  {
  2440.55, 3088.68, 3090.02, 3306.7, 3307.72, 3524.15, 4171.53, 4387.8, 4819.53, 5466.72, 4820.77
   },
  {
  2452.56, 3103.95, 3105.29, 3323.05, 3324.08, 3541.59, 4192.23, 4409.59, 4843.49, 5493.95, 4844.74
   },
  {
  2464.56, 3119.22, 3120.56, 3339.41, 3340.43, 3559.04, 4212.94, 4431.38, 4867.46, 5521.17, 4868.7
   },
  {
  2476.57, 3134.48, 3135.82, 3355.76, 3356.78, 3576.48, 4233.64, 4453.17, 4891.42, 5548.4, 4892.67
   },
  {
  2488.57, 3149.75, 3151.09, 3372.11, 3373.14, 3593.92, 4254.34, 4474.96, 4915.39, 5575.63, 4916.63
   },
  {
  2500.57, 3165.02, 3166.36, 3388.47, 3389.49, 3611.36, 4275.04, 4496.75, 4939.35, 5602.86, 4940.6
   },
  {
  2512.58, 3180.28, 3181.62, 3404.82, 3405.84, 3628.8, 4295.75, 4518.54, 4963.32, 5630.08, 4964.56
   },
  {
  2524.58, 3195.55, 3196.89, 3421.17, 3422.2, 3646.24, 4316.45, 4540.33, 4987.28, 5657.31, 4988.53
   },
  {
  2536.58, 3210.81, 3212.15, 3437.53, 3438.55, 3663.68, 4337.15, 4562.12, 5011.25, 5684.54, 5012.49
   },
  {
  2548.59, 3226.08, 3227.42, 3453.88, 3454.9, 3681.12, 4357.85, 4583.91, 5035.21, 5711.76, 5036.45
   },
  {
  2560.59, 3241.34, 3242.68, 3470.23, 3471.25, 3698.56, 4378.56, 4605.7, 5059.18, 5738.99, 5060.42
   },
  {
  2572.59, 3256.61, 3257.95, 3486.58, 3487.61, 3716, 4399.26, 4627.49, 5083.14, 5766.21, 5084.38
   },
  {
  2584.6, 3271.87, 3273.21, 3502.94, 3503.96, 3733.44, 4419.96, 4649.28, 5107.1, 5793.44, 5108.35
   },
  {
  2596.6, 3287.14, 3288.48, 3519.29, 3520.31, 3750.88, 4440.66, 4671.07, 5131.07, 5820.66, 5132.31
   },
  {
  2608.6, 3302.4, 3303.74, 3535.64, 3536.66, 3768.32, 4461.36, 4692.86, 5155.03, 5847.89, 5156.27
   },
  {
  2620.6, 3317.66, 3319, 3551.99, 3553.01, 3785.75, 4482.06, 4714.64, 5178.99, 5875.12, 5180.24
   },
  {
  2632.6, 3332.93, 3334.27, 3568.34, 3569.36, 3803.19, 4502.76, 4736.43, 5202.95, 5902.34, 5204.2
   },
  {
  2644.61, 3348.19, 3349.53, 3584.69, 3585.72, 3820.63, 4523.46, 4758.22, 5226.92, 5929.56, 5228.16
   },
  {
  2656.61, 3363.46, 3364.79, 3601.04, 3602.07, 3838.07, 4544.16, 4780.01, 5250.88, 5956.79, 5252.12
   },
  {
  2668.61, 3378.72, 3380.06, 3617.39, 3618.42, 3855.51, 4564.86, 4801.79, 5274.84, 5984.01, 5276.08
   },
  {
  2680.61, 3393.98, 3395.32, 3633.74, 3634.77, 3872.95, 4585.56, 4823.58, 5298.8, 6011.24, 5300.05
   },
  {
  2692.61, 3409.24, 3410.58, 3650.09, 3651.12, 3890.38, 4606.26, 4845.37, 5322.77, 6038.46, 5324.01
   },
  {
  2704.61, 3424.51, 3425.85, 3666.44, 3667.47, 3907.82, 4626.96, 4867.16, 5346.73, 6065.68, 5347.97
   },
  {
  2716.61, 3439.77, 3441.11, 3682.79, 3683.82, 3925.26, 4647.66, 4888.94, 5370.69, 6092.91, 5371.93
   },
  {
  2728.61, 3455.03, 3456.37, 3699.14, 3700.17, 3942.69, 4668.36, 4910.73, 5394.65, 6120.13, 5395.89
   },
  {
  2740.61, 3470.29, 3471.63, 3715.49, 3716.52, 3960.13, 4689.06, 4932.52, 5418.61, 6147.35, 5419.85
   },
  {
  2752.61, 3485.56, 3486.9, 3731.84, 3732.87, 3977.57, 4709.76, 4954.3, 5442.57, 6174.58, 5443.81
   },
  {
  2764.61, 3500.82, 3502.16, 3748.19, 3749.21, 3995, 4730.46, 4976.09, 5466.53, 6201.8, 5467.78
   },
  {
  2776.61, 3516.08, 3517.42, 3764.54, 3765.56, 4012.44, 4751.16, 4997.87, 5490.49, 6229.02, 5491.74
   },
  {
  2788.61, 3531.34, 3532.68, 3780.89, 3781.91, 4029.88, 4771.85, 5019.66, 5514.45, 6256.25, 5515.7
   },
  {
  2800.61, 3546.6, 3547.94, 3797.24, 3798.26, 4047.31, 4792.55, 5041.44, 5538.41, 6283.47, 5539.66
   },
  {
  2812.61, 3561.86, 3563.2, 3813.59, 3814.61, 4064.75, 4813.25, 5063.23, 5562.37, 6310.69, 5563.62
   },
  {
  2824.61, 3577.12, 3578.46, 3829.93, 3830.96, 4082.18, 4833.95, 5085.02, 5586.33, 6337.91, 5587.58
   },
  {
  2836.6, 3592.39, 3593.72, 3846.28, 3847.31, 4099.62, 4854.65, 5106.8, 5610.29, 6365.13, 5611.54
   },
  {
  2848.6, 3607.65, 3608.98, 3862.63, 3863.65, 4117.06, 4875.34, 5128.58, 5634.25, 6392.36, 5635.5
   },
  {
  2860.6, 3622.91, 3624.25, 3878.98, 3880, 4134.49, 4896.04, 5150.37, 5658.21, 6419.58, 5659.46
   },
  {
  2872.6, 3638.17, 3639.51, 3895.33, 3896.35, 4151.93, 4916.74, 5172.15, 5682.17, 6446.8, 5683.41
   },
  {
  2884.6, 3653.43, 3654.77, 3911.67, 3912.7, 4169.36, 4937.43, 5193.94, 5706.13, 6474.02, 5707.37
   },
  {
  2896.6, 3668.69, 3670.03, 3928.02, 3929.04, 4186.8, 4958.13, 5215.72, 5730.09, 6501.24, 5731.33
   },
  {
  2908.59, 3683.95, 3685.29, 3944.37, 3945.39, 4204.23, 4978.83, 5237.51, 5754.05, 6528.46, 5755.29
   },
  {
  2920.59, 3699.21, 3700.55, 3960.72, 3961.74, 4221.66, 4999.52, 5259.29, 5778.01, 6555.68, 5779.25
   },
  {
  2932.59, 3714.47, 3715.8, 3977.06, 3978.09, 4239.1, 5020.22, 5281.07, 5801.97, 6582.9, 5803.21
   },
  {
  2944.59, 3729.73, 3731.06, 3993.41, 3994.43, 4256.53, 5040.92, 5302.86, 5825.92, 6610.12, 5827.17
   },
  {
  2956.58, 3744.98, 3746.32, 4009.76, 4010.78, 4273.97, 5061.61, 5324.64, 5849.88, 6637.34, 5851.12
   },
  {
  2968.58, 3760.24, 3761.58, 4026.1, 4027.13, 4291.4, 5082.31, 5346.42, 5873.84, 6664.56, 5875.08
   },
  {
  2980.58, 3775.5, 3776.84, 4042.45, 4043.47, 4308.83, 5103, 5368.21, 5897.8, 6691.78, 5899.04
   },
  {
  2992.57, 3790.76, 3792.1, 4058.79, 4059.82, 4326.27, 5123.7, 5389.99, 5921.76, 6719, 5923
   },
  {
  3004.57, 3806.02, 3807.36, 4075.14, 4076.16, 4343.7, 5144.4, 5411.77, 5945.71, 6746.22, 5946.96
   },
  {
  3016.57, 3821.28, 3822.62, 4091.49, 4092.51, 4361.13, 5165.09, 5433.56, 5969.67, 6773.44, 5970.91
   },
  {
  3028.56, 3836.54, 3837.88, 4107.83, 4108.86, 4378.57, 5185.79, 5455.34, 5993.63, 6800.66, 5994.87
   },
  {
  3040.56, 3851.79, 3853.13, 4124.18, 4125.2, 4396, 5206.48, 5477.12, 6017.58, 6827.88, 6018.83
   },
  {
  3052.55, 3867.05, 3868.39, 4140.52, 4141.55, 4413.43, 5227.18, 5498.9, 6041.54, 6855.1, 6042.78
   },
  {
  3064.55, 3882.31, 3883.65, 4156.87, 4157.89, 4430.87, 5247.87, 5520.69, 6065.5, 6882.32, 6066.74
   },
  {
  3076.55, 3897.57, 3898.91, 4173.21, 4174.24, 4448.3, 5268.57, 5542.47, 6089.46, 6909.54, 6090.7
   },
  {
  3088.54, 3912.83, 3914.17, 4189.56, 4190.58, 4465.73, 5289.26, 5564.25, 6113.41, 6936.76, 6114.65
   },
  {
  3100.54, 3928.08, 3929.42, 4205.9, 4206.93, 4483.16, 5309.95, 5586.03, 6137.37, 6963.98, 6138.61
   },
  {
  3112.53, 3943.34, 3944.68, 4222.25, 4223.27, 4500.6, 5330.65, 5607.81, 6161.32, 6991.19, 6162.57
   },
  {
  3124.53, 3958.6, 3959.94, 4238.59, 4239.62, 4518.03, 5351.34, 5629.59, 6185.28, 7018.41, 6186.52
   },
  {
  3136.52, 3973.86, 3975.19, 4254.94, 4255.96, 4535.46, 5372.04, 5651.38, 6209.24, 7045.63, 6210.48
   },
  {
  3148.52, 3989.11, 3990.45, 4271.28, 4272.3, 4552.89, 5392.73, 5673.16, 6233.19, 7072.85, 6234.44
   },
  {
  3160.51, 4004.37, 4005.71, 4287.63, 4288.65, 4570.32, 5413.42, 5694.94, 6257.15, 7100.07, 6258.39
   },
  {
  3172.51, 4019.63, 4020.96, 4303.97, 4304.99, 4587.75, 5434.12, 5716.72, 6281.1, 7127.28, 6282.35
   },
  {
  3184.5, 4034.88, 4036.22, 4320.31, 4321.34, 4605.18, 5454.81, 5738.5, 6305.06, 7154.5, 6306.3
   },
  {
  3196.5, 4050.14, 4051.48, 4336.66, 4337.68, 4622.62, 5475.5, 5760.28, 6329.02, 7181.72, 6330.26
   },
  {
  3208.49, 4065.4, 4066.73, 4353, 4354.02, 4640.05, 5496.2, 5782.06, 6352.97, 7208.94, 6354.21
   },
  {
  3220.48, 4080.65, 4081.99, 4369.35, 4370.37, 4657.48, 5516.89, 5803.84, 6376.93, 7236.15, 6378.17
   },
  {
  3232.48, 4095.91, 4097.25, 4385.69, 4386.71, 4674.91, 5537.58, 5825.62, 6400.88, 7263.37, 6402.12
   },
  {
  3244.47, 4111.16, 4112.5, 4402.03, 4403.05, 4692.34, 5558.28, 5847.4, 6424.84, 7290.59, 6426.08
   },
  {
  3256.47, 4126.42, 4127.76, 4418.38, 4419.4, 4709.77, 5578.97, 5869.18, 6448.79, 7317.8, 6450.03
   },
  {
  3268.46, 4141.68, 4143.01, 4434.72, 4435.74, 4727.2, 5599.66, 5890.96, 6472.74, 7345.02, 6473.99
   },
  {
  3280.45, 4156.93, 4158.27, 4451.06, 4452.08, 4744.63, 5620.35, 5912.74, 6496.7, 7372.24, 6497.94
   },
  {
  3292.45, 4172.19, 4173.53, 4467.4, 4468.43, 4762.06, 5641.05, 5934.52, 6520.65, 7399.45, 6521.9
   },
  {
  3304.44, 4187.44, 4188.78, 4483.75, 4484.77, 4779.49, 5661.74, 5956.3, 6544.61, 7426.67, 6545.85
   },
  {
  3316.43, 4202.7, 4204.04, 4500.09, 4501.11, 4796.92, 5682.43, 5978.08, 6568.56, 7453.89, 6569.81
   },
  {
  3328.43, 4217.95, 4219.29, 4516.43, 4517.45, 4814.35, 5703.12, 5999.86, 6592.52, 7481.1, 6593.76
   },
  {
  3340.42, 4233.21, 4234.55, 4532.77, 4533.8, 4831.78, 5723.81, 6021.64, 6616.47, 7508.32, 6617.71
   },
  {
  3352.41, 4248.46, 4249.8, 4549.12, 4550.14, 4849.21, 5744.51, 6043.42, 6640.42, 7535.54, 6641.67
   },
  {
  3364.4, 4263.72, 4265.06, 4565.46, 4566.48, 4866.64, 5765.2, 6065.2, 6664.38, 7562.75, 6665.62
   },
  {
  3376.4, 4278.97, 4280.31, 4581.8, 4582.82, 4884.07, 5785.89, 6086.97, 6688.33, 7589.97, 6689.57
   },
  {
  3388.39, 4294.23, 4295.56, 4598.14, 4599.17, 4901.5, 5806.58, 6108.75, 6712.28, 7617.18, 6713.53
   },
  {
  3400.38, 4309.48, 4310.82, 4614.48, 4615.51, 4918.93, 5827.27, 6130.53, 6736.24, 7644.4, 6737.48
   },
  {
  3412.37, 4324.73, 4326.07, 4630.83, 4631.85, 4936.36, 5847.96, 6152.31, 6760.19, 7671.61, 6761.43
   },
  {
  3424.36, 4339.99, 4341.33, 4647.17, 4648.19, 4953.79, 5868.65, 6174.09, 6784.14, 7698.83, 6785.39
   },
  {
  3436.36, 4355.24, 4356.58, 4663.51, 4664.53, 4971.21, 5889.34, 6195.87, 6808.1, 7726.04, 6809.34
   },
  {
  3448.35, 4370.5, 4371.84, 4679.85, 4680.87, 4988.64, 5910.04, 6217.65, 6832.05, 7753.26, 6833.29
   },
  {
  3460.34, 4385.75, 4387.09, 4696.19, 4697.21, 5006.07, 5930.73, 6239.42, 6856, 7780.47, 6857.25
   },
  {
  3472.33, 4401, 4402.34, 4712.53, 4713.56, 5023.5, 5951.42, 6261.2, 6879.96, 7807.69, 6881.2
   },
  {
  3484.32, 4416.26, 4417.6, 4728.87, 4729.9, 5040.93, 5972.11, 6282.98, 6903.91, 7834.9, 6905.15
   },
  {
  3496.32, 4431.51, 4432.85, 4745.21, 4746.24, 5058.36, 5992.8, 6304.76, 6927.86, 7862.12, 6929.1
   },
  {
  3508.31, 4446.76, 4448.1, 4761.56, 4762.58, 5075.78, 6013.49, 6326.53, 6951.81, 7889.33, 6953.06
   },
  {
  3520.3, 4462.02, 4463.36, 4777.9, 4778.92, 5093.21, 6034.18, 6348.31, 6975.77, 7916.55, 6977.01
   },
  {
  3532.29, 4477.27, 4478.61, 4794.24, 4795.26, 5110.64, 6054.87, 6370.09, 6999.72, 7943.76, 7000.96
   },
  {
  3544.28, 4492.52, 4493.86, 4810.58, 4811.6, 5128.07, 6075.56, 6391.87, 7023.67, 7970.97, 7024.91
   },
  {
  3556.27, 4507.78, 4509.12, 4826.92, 4827.94, 5145.5, 6096.25, 6413.64, 7047.62, 7998.19, 7048.86
   },
  {
  3568.26, 4523.03, 4524.37, 4843.26, 4844.28, 5162.92, 6116.94, 6435.42, 7071.57, 8025.4, 7072.82
   },
  {
  3580.25, 4538.28, 4539.62, 4859.6, 4860.62, 5180.35, 6137.63, 6457.2, 7095.53, 8052.62, 7096.77
   },
  {
  3592.24, 4553.54, 4554.87, 4875.94, 4876.96, 5197.78, 6158.32, 6478.98, 7119.48, 8079.83, 7120.72
   },
  {
  3604.23, 4568.79, 4570.13, 4892.28, 4893.3, 5215.21, 6179.01, 6500.75, 7143.43, 8107.04, 7144.67
   },
  {
  3616.22, 4584.04, 4585.38, 4908.62, 4909.64, 5232.63, 6199.7, 6522.53, 7167.38, 8134.26, 7168.62
   },
  {
  3628.21, 4599.29, 4600.63, 4924.96, 4925.98, 5250.06, 6220.38, 6544.31, 7191.33, 8161.47, 7192.57
   },
  {
  3640.2, 4614.55, 4615.88, 4941.3, 4942.32, 5267.49, 6241.07, 6566.08, 7215.28, 8188.68, 7216.53
   },
  {
  3652.19, 4629.8, 4631.14, 4957.64, 4958.66, 5284.91, 6261.76, 6587.86, 7239.23, 8215.9, 7240.48
   },
  {
  3664.18, 4645.05, 4646.39, 4973.98, 4975, 5302.34, 6282.45, 6609.63, 7263.18, 8243.11, 7264.43
   },
  {
  3676.17, 4660.3, 4661.64, 4990.32, 4991.34, 5319.77, 6303.14, 6631.41, 7287.14, 8270.32, 7288.38
   },
  {
  3688.16, 4675.55, 4676.89, 5006.65, 5007.68, 5337.2, 6323.83, 6653.19, 7311.09, 8297.54, 7312.33
   },
  {
  3700.15, 4690.81, 4692.14, 5022.99, 5024.02, 5354.62, 6344.52, 6674.96, 7335.04, 8324.75, 7336.28
   },
  {
  3712.14, 4706.06, 4707.4, 5039.33, 5040.36, 5372.05, 6365.21, 6696.74, 7358.99, 8351.96, 7360.23
   },
  {
  3724.13, 4721.31, 4722.65, 5055.67, 5056.7, 5389.47, 6385.89, 6718.51, 7382.94, 8379.17, 7384.18
   },
  {
  3736.12, 4736.56, 4737.9, 5072.01, 5073.03, 5406.9, 6406.58, 6740.29, 7406.89, 8406.39, 7408.13
   },
  {
  3748.11, 4751.81, 4753.15, 5088.35, 5089.37, 5424.33, 6427.27, 6762.07, 7430.84, 8433.6, 7432.08
   },
  {
  3760.1, 4767.06, 4768.4, 5104.69, 5105.71, 5441.75, 6447.96, 6783.84, 7454.79, 8460.81, 7456.03
   },
  {
  3772.09, 4782.31, 4783.65, 5121.03, 5122.05, 5459.18, 6468.65, 6805.62, 7478.74, 8488.02, 7479.98
   },
  {
  3784.08, 4797.57, 4798.91, 5137.37, 5138.39, 5476.61, 6489.34, 6827.39, 7502.69, 8515.24, 7503.93
   },
  {
  3796.07, 4812.82, 4814.16, 5153.7, 5154.73, 5494.03, 6510.02, 6849.17, 7526.64, 8542.45, 7527.88
   },
  {
  3808.06, 4828.07, 4829.41, 5170.04, 5171.07, 5511.46, 6530.71, 6870.94, 7550.59, 8569.66, 7551.83
   },
  {
  3820.05, 4843.32, 4844.66, 5186.38, 5187.4, 5528.88, 6551.4, 6892.72, 7574.54, 8596.87, 7575.78
   },
  {
  3832.03, 4858.57, 4859.91, 5202.72, 5203.74, 5546.31, 6572.09, 6914.49, 7598.49, 8624.08, 7599.73
   },
  {
  3844.02, 4873.82, 4875.16, 5219.06, 5220.08, 5563.73, 6592.77, 6936.27, 7622.44, 8651.3, 7623.68
   },
  {
  3856.01, 4889.07, 4890.41, 5235.4, 5236.42, 5581.16, 6613.46, 6958.04, 7646.39, 8678.51, 7647.63
   },
  {
  3868, 4904.32, 4905.66, 5251.73, 5252.76, 5598.58, 6634.15, 6979.82, 7670.34, 8705.72, 7671.58
   },
  {
  3879.99, 4919.57, 4920.91, 5268.07, 5269.09, 5616.01, 6654.84, 7001.59, 7694.29, 8732.93, 7695.53
   },
  {
  3891.98, 4934.82, 4936.16, 5284.41, 5285.43, 5633.43, 6675.52, 7023.37, 7718.24, 8760.14, 7719.48
   },
  {
  3903.97, 4950.07, 4951.41, 5300.75, 5301.77, 5650.86, 6696.21, 7045.14, 7742.19, 8787.35, 7743.43
   },
  {
  3915.95, 4965.32, 4966.66, 5317.08, 5318.11, 5668.28, 6716.9, 7066.92, 7766.14, 8814.57, 7767.38
   },
  {
  3927.94, 4980.57, 4981.91, 5333.42, 5334.44, 5685.71, 6737.59, 7088.69, 7790.08, 8841.78, 7791.33
   },
  {
  3939.93, 4995.82, 4997.16, 5349.76, 5350.78, 5703.13, 6758.27, 7110.46, 7814.03, 8868.99, 7815.28
   },
  {
  3951.92, 5011.07, 5012.41, 5366.1, 5367.12, 5720.56, 6778.96, 7132.24, 7837.98, 8896.2, 7839.23
   },
  {
  3963.91, 5026.32, 5027.66, 5382.43, 5383.46, 5737.98, 6799.65, 7154.01, 7861.93, 8923.41, 7863.17
   },
  {
  3975.89, 5041.57, 5042.91, 5398.77, 5399.79, 5755.41, 6820.33, 7175.79, 7885.88, 8950.62, 7887.12
   },
  {
  3987.88, 5056.82, 5058.16, 5415.11, 5416.13, 5772.83, 6841.02, 7197.56, 7909.83, 8977.83, 7911.07
   },
  {
  3999.87, 5072.07, 5073.41, 5431.44, 5432.47, 5790.26, 6861.71, 7219.33, 7933.78, 9005.04, 7935.02
   },
  {
  4011.86, 5087.32, 5088.66, 5447.78, 5448.8, 5807.68, 6882.39, 7241.11, 7957.73, 9032.25, 7958.97
   },
  {
  4023.84, 5102.57, 5103.91, 5464.12, 5465.14, 5825.11, 6903.08, 7262.88, 7981.67, 9059.46, 7982.92
   },
  {
  4035.83, 5117.82, 5119.16, 5480.46, 5481.48, 5842.53, 6923.76, 7284.66, 8005.62, 9086.67, 8006.87
   },
  {
  4047.82, 5133.07, 5134.41, 5496.79, 5497.82, 5859.95, 6944.45, 7306.43, 8029.57, 9113.88, 8030.81
   },
  {
  4059.8, 5148.32, 5149.66, 5513.13, 5514.15, 5877.38, 6965.14, 7328.2, 8053.52, 9141.09, 8054.76
   },
  {
  4071.79, 5163.57, 5164.91, 5529.47, 5530.49, 5894.8, 6985.82, 7349.98, 8077.47, 9168.3, 8078.71
   },
  {
  4083.78, 5178.82, 5180.16, 5545.8, 5546.82, 5912.23, 7006.51, 7371.75, 8101.42, 9195.51, 8102.66
   },
  {
  4095.77, 5194.07, 5195.41, 5562.14, 5563.16, 5929.65, 7027.19, 7393.52, 8125.36, 9222.72, 8126.61
   },
  {
  4107.75, 5209.32, 5210.65, 5578.47, 5579.5, 5947.07, 7047.88, 7415.3, 8149.31, 9249.93, 8150.55
   },
  {
  4119.74, 5224.56, 5225.9, 5594.81, 5595.83, 5964.5, 7068.57, 7437.07, 8173.26, 9277.14, 8174.5
   },
  {
  4131.73, 5239.81, 5241.15, 5611.15, 5612.17, 5981.92, 7089.25, 7458.84, 8197.21, 9304.35, 8198.45
   },
  {
  4143.71, 5255.06, 5256.4, 5627.48, 5628.51, 5999.34, 7109.94, 7480.61, 8221.15, 9331.56, 8222.4
   },
  {
  4155.7, 5270.31, 5271.65, 5643.82, 5644.84, 6016.77, 7130.62, 7502.39, 8245.1, 9358.77, 8246.34
   },
  {
  4167.69, 5285.56, 5286.9, 5660.15, 5661.18, 6034.19, 7151.31, 7524.16, 8269.05, 9385.98, 8270.29
   },
  {
  4179.67, 5300.81, 5302.15, 5676.49, 5677.51, 6051.61, 7171.99, 7545.93, 8293, 9413.19, 8294.24
   },
  {
  4191.66, 5316.06, 5317.4, 5692.83, 5693.85, 6069.04, 7192.68, 7567.71, 8316.94, 9440.4, 8318.19
   },
  {
  4203.64, 5331.3, 5332.64, 5709.16, 5710.19, 6086.46, 7213.36, 7589.48, 8340.89, 9467.61, 8342.13
   },
  {
  4215.63, 5346.55, 5347.89, 5725.5, 5726.52, 6103.88, 7234.05, 7611.25, 8364.84, 9494.82, 8366.08
   },
  {
  4227.62, 5361.8, 5363.14, 5741.83, 5742.86, 6121.3, 7254.73, 7633.02, 8388.79, 9522.03, 8390.03
   },
  {
  4239.6, 5377.05, 5378.39, 5758.17, 5759.19, 6138.73, 7275.42, 7654.8, 8412.73, 9549.24, 8413.98
   },
  {
  4251.59, 5392.3, 5393.64, 5774.5, 5775.53, 6156.15, 7296.1, 7676.57, 8436.68, 9576.45, 8437.92
   },
  {
  4263.57, 5407.54, 5408.88, 5790.84, 5791.86, 6173.57, 7316.79, 7698.34, 8460.63, 9603.66, 8461.87
   },
  {
  4275.56, 5422.79, 5424.13, 5807.17, 5808.2, 6191, 7337.47, 7720.11, 8484.57, 9630.87, 8485.82
   },
  {
  4287.55, 5438.04, 5439.38, 5823.51, 5824.53, 6208.42, 7358.16, 7741.88, 8508.52, 9658.07, 8509.76
   },
  {
  4299.53, 5453.29, 5454.63, 5839.85, 5840.87, 6225.84, 7378.84, 7763.66, 8532.47, 9685.28, 8533.71
   },
  {
  4311.52, 5468.54, 5469.88, 5856.18, 5857.2, 6243.26, 7399.53, 7785.43, 8556.41, 9712.49, 8557.66
   },
  {
  4323.5, 5483.78, 5485.12, 5872.52, 5873.54, 6260.69, 7420.21, 7807.2, 8580.36, 9739.7, 8581.6
   },
  {
  4335.49, 5499.03, 5500.37, 5888.85, 5889.87, 6278.11, 7440.9, 7828.97, 8604.31, 9766.91, 8605.55
   },
  {
  4347.47, 5514.28, 5515.62, 5905.18, 5906.21, 6295.53, 7461.58, 7850.74, 8628.25, 9794.12, 8629.5
   },
  {
  4359.46, 5529.53, 5530.87, 5921.52, 5922.54, 6312.95, 7482.26, 7872.51, 8652.2, 9821.33, 8653.44
   },
  {
  4371.44, 5544.77, 5546.11, 5937.85, 5938.88, 6330.37, 7502.95, 7894.29, 8676.15, 9848.54, 8677.39
   },
  {
  4383.43, 5560.02, 5561.36, 5954.19, 5955.21, 6347.8, 7523.63, 7916.06, 8700.09, 9875.74, 8701.33
   },
  {
  4395.41, 5575.27, 5576.61, 5970.52, 5971.55, 6365.22, 7544.32, 7937.83, 8724.04, 9902.95, 8725.28
   },
  {
  4407.4, 5590.52, 5591.85, 5986.86, 5987.88, 6382.64, 7565, 7959.6, 8747.98, 9930.16, 8749.23
   },
  {
  4419.38, 5605.76, 5607.1, 6003.19, 6004.22, 6400.06, 7585.68, 7981.37, 8771.93, 9957.37, 8773.17
   },
  {
  4431.37, 5621.01, 5622.35, 6019.53, 6020.55, 6417.48, 7606.37, 8003.14, 8795.88, 9984.58, 8797.12
   },
  {
  4443.35, 5636.26, 5637.6, 6035.86, 6036.88, 6434.91, 7627.05, 8024.91, 8819.82, 10011.8, 8821.06
   },
  {
  4455.34, 5651.5, 5652.84, 6052.2, 6053.22, 6452.33, 7647.74, 8046.69, 8843.77, 10039, 8845.01
   },
  {
  4467.32, 5666.75, 5668.09, 6068.53, 6069.55, 6469.75, 7668.42, 8068.46, 8867.71, 10066.2, 8868.96
   },
  {
  4479.31, 5682, 5683.34, 6084.86, 6085.89, 6487.17, 7689.1, 8090.23, 8891.66, 10093.4, 8892.9
   },
  {
  4491.29, 5697.24, 5698.58, 6101.2, 6102.22, 6504.59, 7709.79, 8112, 8915.6, 10120.6, 8916.85
   },
  {
  4503.28, 5712.49, 5713.83, 6117.53, 6118.55, 6522.01, 7730.47, 8133.77, 8939.55, 10147.8, 8940.79
   },
  {
  4515.26, 5727.74, 5729.08, 6133.87, 6134.89, 6539.43, 7751.15, 8155.54, 8963.5, 10175, 8964.74
   },
  {
  4527.25, 5742.98, 5744.32, 6150.2, 6151.22, 6556.86, 7771.84, 8177.31, 8987.44, 10202.2, 8988.68
   },
  {
  4539.23, 5758.23, 5759.57, 6166.53, 6167.56, 6574.28, 7792.52, 8199.08, 9011.39, 10229.4, 9012.63
   },
  {
  4551.21, 5773.48, 5774.81, 6182.87, 6183.89, 6591.7, 7813.2, 8220.85, 9035.33, 10256.7, 9036.57
   },
  {
  4563.2, 5788.72, 5790.06, 6199.2, 6200.22, 6609.12, 7833.89, 8242.62, 9059.28, 10283.9, 9060.52
   },
  {
  4575.18, 5803.97, 5805.31, 6215.53, 6216.56, 6626.54, 7854.57, 8264.39, 9083.22, 10311.1, 9084.47
   },
  {
  4587.17, 5819.21, 5820.55, 6231.87, 6232.89, 6643.96, 7875.25, 8286.16, 9107.17, 10338.3, 9108.41
   },
  {
  4599.15, 5834.46, 5835.8, 6248.2, 6249.22, 6661.38, 7895.94, 8307.93, 9131.11, 10365.5, 9132.36
   },
  {
  4611.14, 5849.71, 5851.05, 6264.54, 6265.56, 6678.8, 7916.62, 8329.7, 9155.06, 10392.7, 9156.3
   },
  {
  4623.12, 5864.95, 5866.29, 6280.87, 6281.89, 6696.22, 7937.3, 8351.47, 9179, 10419.9, 9180.25
   },
  {
  4635.1, 5880.2, 5881.54, 6297.2, 6298.22, 6713.64, 7957.98, 8373.24, 9202.95, 10447.1, 9204.19
   },
  {
  4647.09, 5895.44, 5896.78, 6313.54, 6314.56, 6731.06, 7978.67, 8395.01, 9226.89, 10474.3, 9228.14
   },
  {
  4659.07, 5910.69, 5912.03, 6329.87, 6330.89, 6748.49, 7999.35, 8416.78, 9250.84, 10501.5, 9252.08
   },
  {
  4671.05, 5925.94, 5927.28, 6346.2, 6347.22, 6765.91, 8020.03, 8438.55, 9274.78, 10528.7, 9276.02
   },
  {
  4683.04, 5941.18, 5942.52, 6362.53, 6363.56, 6783.33, 8040.71, 8460.32, 9298.73, 10555.9, 9299.97
   },
  {
  4695.02, 5956.43, 5957.77, 6378.87, 6379.89, 6800.75, 8061.4, 8482.09, 9322.67, 10583.1, 9323.91
   },
  {
  4707, 5971.67, 5973.01, 6395.2, 6396.22, 6818.17, 8082.08, 8503.86, 9346.62, 10610.3, 9347.86
   },
  {
  4718.99, 5986.92, 5988.26, 6411.53, 6412.56, 6835.59, 8102.76, 8525.63, 9370.56, 10637.6, 9371.8
   },
  {
  4730.97, 6002.16, 6003.5, 6427.87, 6428.89, 6853.01, 8123.44, 8547.4, 9394.5, 10664.8, 9395.75
   },
  {
  4742.96, 6017.41, 6018.75, 6444.2, 6445.22, 6870.43, 8144.13, 8569.17, 9418.45, 10692, 9419.69
   },
  {
  4754.94, 6032.66, 6033.99, 6460.53, 6461.55, 6887.85, 8164.81, 8590.94, 9442.39, 10719.2, 9443.64
   },
  {
  4766.92, 6047.9, 6049.24, 6476.86, 6477.89, 6905.27, 8185.49, 8612.71, 9466.34, 10746.4, 9467.58
   },
  {
  4778.9, 6063.15, 6064.48, 6493.2, 6494.22, 6922.69, 8206.17, 8634.48, 9490.28, 10773.6, 9491.52
   },
  {
  4790.89, 6078.39, 6079.73, 6509.53, 6510.55, 6940.11, 8226.85, 8656.25, 9514.23, 10800.8, 9515.47
   },
  {
  4802.87, 6093.64, 6094.98, 6525.86, 6526.89, 6957.53, 8247.54, 8678.02, 9538.17, 10828, 9539.41
   },
  {
  4814.85, 6108.88, 6110.22, 6542.19, 6543.22, 6974.95, 8268.22, 8699.79, 9562.11, 10855.2, 9563.36
   },
  {
  4826.84, 6124.13, 6125.47, 6558.53, 6559.55, 6992.37, 8288.9, 8721.56, 9586.06, 10882.4, 9587.3
   },
  {
  4838.82, 6139.37, 6140.71, 6574.86, 6575.88, 7009.79, 8309.58, 8743.33, 9610, 10909.6, 9611.24
   },
  {
  4850.8, 6154.62, 6155.96, 6591.19, 6592.21, 7027.21, 8330.26, 8765.1, 9633.95, 10936.8, 9635.19
   },
  {
  4862.79, 6169.86, 6171.2, 6607.52, 6608.55, 7044.63, 8350.95, 8786.87, 9657.89, 10964, 9659.13
   },
  {
  4874.77, 6185.11, 6186.44, 6623.86, 6624.88, 7062.05, 8371.63, 8808.63, 9681.83, 10991.2, 9683.08
   },
  {
  4886.75, 6200.35, 6201.69, 6640.19, 6641.21, 7079.47, 8392.31, 8830.4, 9705.78, 11018.4, 9707.02
   },
  {
  4898.73, 6215.6, 6216.93, 6656.52, 6657.54, 7096.88, 8412.99, 8852.17, 9729.72, 11045.6, 9730.96
   },
  {
  4910.72, 6230.84, 6232.18, 6672.85, 6673.88, 7114.3, 8433.67, 8873.94, 9753.66, 11072.8, 9754.91
   },
  {
  4922.7, 6246.08, 6247.42, 6689.18, 6690.21, 7131.72, 8454.35, 8895.71, 9777.61, 11100.1, 9778.85
   },
  {
  4934.68, 6261.33, 6262.67, 6705.52, 6706.54, 7149.14, 8475.04, 8917.48, 9801.55, 11127.3, 9802.79
   },
  {
  4946.66, 6276.57, 6277.91, 6721.85, 6722.87, 7166.56, 8495.72, 8939.25, 9825.49, 11154.5, 9826.74
   },
  {
  4958.65, 6291.82, 6293.16, 6738.18, 6739.2, 7183.98, 8516.4, 8961.02, 9849.44, 11181.7, 9850.68
   },
  {
  4970.63, 6307.06, 6308.4, 6754.51, 6755.53, 7201.4, 8537.08, 8982.78, 9873.38, 11208.9, 9874.62
   },
  {
  4982.61, 6322.31, 6323.65, 6770.84, 6771.87, 7218.82, 8557.76, 9004.55, 9897.32, 11236.1, 9898.57
   },
  {
  4994.59, 6337.55, 6338.89, 6787.17, 6788.2, 7236.24, 8578.44, 9026.32, 9921.27, 11263.3, 9922.51
   },
  {
  5006.57, 6352.8, 6354.13, 6803.51, 6804.53, 7253.66, 8599.12, 9048.09, 9945.21, 11290.5, 9946.45
   },
  {
  5018.56, 6368.04, 6369.38, 6819.84, 6820.86, 7271.08, 8619.8, 9069.86, 9969.15, 11317.7, 9970.4
   },
  {
  5030.54, 6383.28, 6384.62, 6836.17, 6837.19, 7288.49, 8640.48, 9091.63, 9993.1, 11344.9, 9994.34
   },
  {
  5042.52, 6398.53, 6399.87, 6852.5, 6853.52, 7305.91, 8661.17, 9113.4, 10017, 11372.1, 10018.3
   },
  {
  5054.5, 6413.77, 6415.11, 6868.83, 6869.86, 7323.33, 8681.85, 9135.16, 10041, 11399.3, 10042.2
   },
  {
  5066.48, 6429.02, 6430.35, 6885.16, 6886.19, 7340.75, 8702.53, 9156.93, 10064.9, 11426.5, 10066.2
   },
  {
  5078.47, 6444.26, 6445.6, 6901.49, 6902.52, 7358.17, 8723.21, 9178.7, 10088.9, 11453.7, 10090.1
   },
  {
  5090.45, 6459.5, 6460.84, 6917.83, 6918.85, 7375.59, 8743.89, 9200.47, 10112.8, 11480.9, 10114.1
   },
  {
  5102.43, 6474.75, 6476.09, 6934.16, 6935.18, 7393.01, 8764.57, 9222.24, 10136.8, 11508.1, 10138
   },
  {
  5114.41, 6489.99, 6491.33, 6950.49, 6951.51, 7410.43, 8785.25, 9244, 10160.7, 11535.3, 10161.9
   },
  {
  5126.39, 6505.23, 6506.57, 6966.82, 6967.84, 7427.84, 8805.93, 9265.77, 10184.6, 11562.5, 10185.9
   },
  {
  5138.37, 6520.48, 6521.82, 6983.15, 6984.17, 7445.26, 8826.61, 9287.54, 10208.6, 11589.7, 10209.8
   },
  {
  5150.36, 6535.72, 6537.06, 6999.48, 7000.5, 7462.68, 8847.29, 9309.31, 10232.5, 11617, 10233.8
   },
  {
  5162.34, 6550.97, 6552.3, 7015.81, 7016.84, 7480.1, 8867.97, 9331.08, 10256.5, 11644.2, 10257.7
   },
  {
  5174.32, 6566.21, 6567.55, 7032.14, 7033.17, 7497.52, 8888.65, 9352.84, 10280.4, 11671.4, 10281.7
   },
  {
  5186.3, 6581.45, 6582.79, 7048.47, 7049.5, 7514.94, 8909.33, 9374.61, 10304.4, 11698.6, 10305.6
   },
  {
  5198.28, 6596.7, 6598.04, 7064.81, 7065.83, 7532.35, 8930.01, 9396.38, 10328.3, 11725.8, 10329.5
   },
  {
  5210.26, 6611.94, 6613.28, 7081.14, 7082.16, 7549.77, 8950.69, 9418.15, 10352.2, 11753, 10353.5
   },
  {
  5222.24, 6627.18, 6628.52, 7097.47, 7098.49, 7567.19, 8971.37, 9439.91, 10376.2, 11780.2, 10377.4
   },
  {
  5234.23, 6642.43, 6643.77, 7113.8, 7114.82, 7584.61, 8992.05, 9461.68, 10400.1, 11807.4, 10401.4
   },
  {
  5246.21, 6657.67, 6659.01, 7130.13, 7131.15, 7602.03, 9012.73, 9483.45, 10424.1, 11834.6, 10425.3
   },
  {
  5258.19, 6672.91, 6674.25, 7146.46, 7147.48, 7619.44, 9033.41, 9505.22, 10448, 11861.8, 10449.2
   },
  {
  5270.17, 6688.16, 6689.49, 7162.79, 7163.81, 7636.86, 9054.09, 9526.98, 10471.9, 11889, 10473.2
   },
  {
  5282.15, 6703.4, 6704.74, 7179.12, 7180.14, 7654.28, 9074.77, 9548.75, 10495.9, 11916.2, 10497.1
   },
  {
  5294.13, 6718.64, 6719.98, 7195.45, 7196.47, 7671.7, 9095.45, 9570.52, 10519.8, 11943.4, 10521.1
   },
  {
  5306.11, 6733.89, 6735.22, 7211.78, 7212.8, 7689.12, 9116.13, 9592.29, 10543.8, 11970.6, 10545
   },
  {
  5318.09, 6749.13, 6750.47, 7228.11, 7229.13, 7706.53, 9136.81, 9614.05, 10567.7, 11997.8, 10569
   },
  {
  5330.07, 6764.37, 6765.71, 7244.44, 7245.46, 7723.95, 9157.49, 9635.82, 10591.7, 12025, 10592.9
   },
  {
  5342.05, 6779.61, 6780.95, 7260.77, 7261.79, 7741.37, 9178.17, 9657.59, 10615.6, 12052.2, 10616.8
   },
  {
  5354.03, 6794.86, 6796.2, 7277.1, 7278.12, 7758.79, 9198.85, 9679.35, 10639.5, 12079.4, 10640.8
   },
  {
  5366.02, 6810.1, 6811.44, 7293.43, 7294.45, 7776.2, 9219.53, 9701.12, 10663.5, 12106.6, 10664.7
   },
  {
  5378, 6825.34, 6826.68, 7309.76, 7310.78, 7793.62, 9240.21, 9722.89, 10687.4, 12133.8, 10688.7
   },
  {
  5389.98, 6840.58, 6841.92, 7326.09, 7327.11, 7811.04, 9260.89, 9744.65, 10711.4, 12161, 10712.6
   },
  {
  5401.96, 6855.83, 6857.17, 7342.42, 7343.44, 7828.46, 9281.57, 9766.42, 10735.3, 12188.2, 10736.6
   },
  {
  5413.94, 6871.07, 6872.41, 7358.75, 7359.77, 7845.87, 9302.25, 9788.19, 10759.2, 12215.4, 10760.5
   },
  {
  5425.92, 6886.31, 6887.65, 7375.08, 7376.1, 7863.29, 9322.93, 9809.96, 10783.2, 12242.6, 10784.4
   },
  {
  5437.9, 6901.56, 6902.89, 7391.41, 7392.43, 7880.71, 9343.61, 9831.72, 10807.1, 12269.8, 10808.4
   },
  {
  5449.88, 6916.8, 6918.14, 7407.74, 7408.76, 7898.12, 9364.29, 9853.49, 10831.1, 12297.1, 10832.3
   },
  {
  5461.86, 6932.04, 6933.38, 7424.07, 7425.09, 7915.54, 9384.97, 9875.26, 10855, 12324.3, 10856.3
   },
  {
  5473.84, 6947.28, 6948.62, 7440.4, 7441.42, 7932.96, 9405.65, 9897.02, 10879, 12351.5, 10880.2
   },
  {
  5485.82, 6962.53, 6963.86, 7456.73, 7457.75, 7950.38, 9426.33, 9918.79, 10902.9, 12378.7, 10904.1
   },
  {
  5497.8, 6977.77, 6979.11, 7473.06, 7474.08, 7967.79, 9447, 9940.55, 10926.8, 12405.9, 10928.1
   },
  {
  5509.78, 6993.01, 6994.35, 7489.39, 7490.41, 7985.21, 9467.68, 9962.32, 10950.8, 12433.1, 10952
   },
  {
  5521.76, 7008.25, 7009.59, 7505.72, 7506.74, 8002.63, 9488.36, 9984.09, 10974.7, 12460.3, 10976
   },
  {
  5533.74, 7023.49, 7024.83, 7522.05, 7523.07, 8020.04, 9509.04, 10005.9, 10998.7, 12487.5, 10999.9
   },
  {
  5545.72, 7038.74, 7040.08, 7538.38, 7539.4, 8037.46, 9529.72, 10027.6, 11022.6, 12514.7, 11023.8
   },
  {
  5557.7, 7053.98, 7055.32, 7554.71, 7555.73, 8054.88, 9550.4, 10049.4, 11046.5, 12541.9, 11047.8
   },
  {
  5569.68, 7069.22, 7070.56, 7571.04, 7572.06, 8072.29, 9571.08, 10071.2, 11070.5, 12569.1, 11071.7
   },
  {
  5581.66, 7084.46, 7085.8, 7587.37, 7588.39, 8089.71, 9591.76, 10092.9, 11094.4, 12596.3, 11095.7
   },
  {
  5593.64, 7099.7, 7101.04, 7603.7, 7604.72, 8107.13, 9612.44, 10114.7, 11118.4, 12623.5, 11119.6
   },
  {
  5605.62, 7114.95, 7116.29, 7620.03, 7621.05, 8124.54, 9633.12, 10136.5, 11142.3, 12650.7, 11143.6
   },
  {
  5617.6, 7130.19, 7131.53, 7636.36, 7637.38, 8141.96, 9653.79, 10158.2, 11166.2, 12677.9, 11167.5
   },
  {
  5629.58, 7145.43, 7146.77, 7652.68, 7653.71, 8159.38, 9674.47, 10180, 11190.2, 12705.1, 11191.4
   },
  {
  5641.56, 7160.67, 7162.01, 7669.01, 7670.04, 8176.79, 9695.15, 10201.7, 11214.1, 12732.3, 11215.4
   },
  {
  5653.54, 7175.91, 7177.25, 7685.34, 7686.37, 8194.21, 9715.83, 10223.5, 11238.1, 12759.5, 11239.3
   },
  {
  5665.52, 7191.16, 7192.49, 7701.67, 7702.7, 8211.63, 9736.51, 10245.3, 11262, 12786.7, 11263.3
   },
  {
  5677.5, 7206.4, 7207.74, 7718, 7719.02, 8229.04, 9757.19, 10267, 11286, 12813.9, 11287.2
   },
  {
  5689.48, 7221.64, 7222.98, 7734.33, 7735.35, 8246.46, 9777.87, 10288.8, 11309.9, 12841.1, 11311.1
   },
  {
  5701.46, 7236.88, 7238.22, 7750.66, 7751.68, 8263.88, 9798.54, 10310.6, 11333.8, 12868.3, 11335.1
   },
  {
  5713.44, 7252.12, 7253.46, 7766.99, 7768.01, 8281.29, 9819.22, 10332.3, 11357.8, 12895.5, 11359
   },
  {
  5725.42, 7267.36, 7268.7, 7783.32, 7784.34, 8298.71, 9839.9, 10354.1, 11381.7, 12922.7, 11383
   },
  {
  5737.4, 7282.61, 7283.94, 7799.65, 7800.67, 8316.13, 9860.58, 10375.9, 11405.7, 12949.9, 11406.9
   },
  {
  5749.38, 7297.85, 7299.19, 7815.97, 7817, 8333.54, 9881.26, 10397.6, 11429.6, 12977.1, 11430.8
   },
  {
  5761.35, 7313.09, 7314.43, 7832.3, 7833.33, 8350.96, 9901.94, 10419.4, 11453.5, 13004.3, 11454.8
   },
  {
  5773.33, 7328.33, 7329.67, 7848.63, 7849.66, 8368.37, 9922.61, 10441.2, 11477.5, 13031.5, 11478.7
   },
  {
  5785.31, 7343.57, 7344.91, 7864.96, 7865.98, 8385.79, 9943.29, 10462.9, 11501.4, 13058.7, 11502.7
   },
  {
  5797.29, 7358.81, 7360.15, 7881.29, 7882.31, 8403.21, 9963.97, 10484.7, 11525.4, 13085.9, 11526.6
   },
  {
  5809.27, 7374.05, 7375.39, 7897.62, 7898.64, 8420.62, 9984.65, 10506.5, 11549.3, 13113.1, 11550.5
   },
  {
  5821.25, 7389.29, 7390.63, 7913.95, 7914.97, 8438.04, 10005.3, 10528.2, 11573.2, 13140.3, 11574.5
   },
  {
  5833.23, 7404.54, 7405.87, 7930.28, 7931.3, 8455.45, 10026, 10550, 11597.2, 13167.5, 11598.4
   },
  {
  5845.21, 7419.78, 7421.12, 7946.6, 7947.63, 8472.87, 10046.7, 10571.8, 11621.1, 13194.7, 11622.4
   },
  {
  5857.19, 7435.02, 7436.36, 7962.93, 7963.96, 8490.29, 10067.4, 10593.5, 11645.1, 13221.9, 11646.3
   },
  {
  5869.17, 7450.26, 7451.6, 7979.26, 7980.28, 8507.7, 10088, 10615.3, 11669, 13249.2, 11670.2
   },
  {
  5881.15, 7465.5, 7466.84, 7995.59, 7996.61, 8525.12, 10108.7, 10637.1, 11692.9, 13276.4, 11694.2
   },
  {
  5893.13, 7480.74, 7482.08, 8011.92, 8012.94, 8542.53, 10129.4, 10658.8, 11716.9, 13303.6, 11718.1
   },
  {
  5905.1, 7495.98, 7497.32, 8028.25, 8029.27, 8559.95, 10150.1, 10680.6, 11740.8, 13330.8, 11742.1
   },
  {
  5917.08, 7511.22, 7512.56, 8044.57, 8045.6, 8577.36, 10170.7, 10702.4, 11764.8, 13358, 11766
   },
  {
  5929.06, 7526.46, 7527.8, 8060.9, 8061.93, 8594.78, 10191.4, 10724.1, 11788.7, 13385.2, 11789.9
   },
  {
  5941.04, 7541.71, 7543.04, 8077.23, 8078.25, 8612.2, 10212.1, 10745.9, 11812.6, 13412.4, 11813.9
   },
  {
  5953.02, 7556.95, 7558.28, 8093.56, 8094.58, 8629.61, 10232.8, 10767.7, 11836.6, 13439.6, 11837.8
   },
  {
  5965, 7572.19, 7573.53, 8109.89, 8110.91, 8647.03, 10253.5, 10789.4, 11860.5, 13466.8, 11861.8
   },
  {
  5976.98, 7587.43, 7588.77, 8126.22, 8127.24, 8664.44, 10274.1, 10811.2, 11884.5, 13494, 11885.7
   },
  {
  5988.96, 7602.67, 7604.01, 8142.54, 8143.57, 8681.86, 10294.8, 10832.9, 11908.4, 13521.2, 11909.6
   },
  {
  6000.93, 7617.91, 7619.25, 8158.87, 8159.89, 8699.27, 10315.5, 10854.7, 11932.3, 13548.4, 11933.6
   },
  {
  6012.91, 7633.15, 7634.49, 8175.2, 8176.22, 8716.69, 10336.2, 10876.5, 11956.3, 13575.6, 11957.5
   },
  {
  6024.89, 7648.39, 7649.73, 8191.53, 8192.55, 8734.1, 10356.8, 10898.2, 11980.2, 13602.8, 11981.5
   },
  {
  6036.87, 7663.63, 7664.97, 8207.86, 8208.88, 8751.52, 10377.5, 10920, 12004.2, 13630, 12005.4
   },
  {
  6048.85, 7678.87, 7680.21, 8224.18, 8225.21, 8768.94, 10398.2, 10941.8, 12028.1, 13657.2, 12029.3
   },
  {
  6060.83, 7694.11, 7695.45, 8240.51, 8241.53, 8786.35, 10418.9, 10963.5, 12052, 13684.4, 12053.3
   },
  {
  6072.8, 7709.35, 7710.69, 8256.84, 8257.86, 8803.77, 10439.6, 10985.3, 12076, 13711.6, 12077.2
   },
  {
  6084.78, 7724.59, 7725.93, 8273.17, 8274.19, 8821.18, 10460.2, 11007.1, 12099.9, 13738.8, 12101.2
   },
  {
  6096.76, 7739.83, 7741.17, 8289.49, 8290.52, 8838.6, 10480.9, 11028.8, 12123.9, 13766, 12125.1
   },
  {
  6108.74, 7755.07, 7756.41, 8305.82, 8306.85, 8856.01, 10501.6, 11050.6, 12147.8, 13793.2, 12149
   },
  {
  6120.72, 7770.31, 7771.65, 8322.15, 8323.17, 8873.43, 10522.3, 11072.4, 12171.7, 13820.4, 12173
   },
  {
  6132.7, 7785.55, 7786.89, 8338.48, 8339.5, 8890.84, 10542.9, 11094.1, 12195.7, 13847.6, 12196.9
   },
  {
  6144.67, 7800.79, 7802.13, 8354.81, 8355.83, 8908.26, 10563.6, 11115.9, 12219.6, 13874.8, 12220.9
   },
  {
  6156.65, 7816.03, 7817.37, 8371.13, 8372.16, 8925.67, 10584.3, 11137.7, 12243.5, 13902, 12244.8
   },
  {
  6168.63, 7831.27, 7832.61, 8387.46, 8388.48, 8943.09, 10605, 11159.4, 12267.5, 13929.2, 12268.7
   },
  {
  6180.61, 7846.52, 7847.85, 8403.79, 8404.81, 8960.5, 10625.7, 11181.2, 12291.4, 13956.4, 12292.7
   },
  {
  6192.59, 7861.76, 7863.09, 8420.12, 8421.14, 8977.92, 10646.3, 11202.9, 12315.4, 13983.6, 12316.6
   },
  {
  6204.56, 7877, 7878.33, 8436.44, 8437.47, 8995.33, 10667, 11224.7, 12339.3, 14010.8, 12340.5
   },
  {
  6216.54, 7892.24, 7893.57, 8452.77, 8453.79, 9012.75, 10687.7, 11246.5, 12363.2, 14038, 12364.5
   },
  {
  6228.52, 7907.48, 7908.81, 8469.1, 8470.12, 9030.16, 10708.4, 11268.2, 12387.2, 14065.2, 12388.4
   },
  {
  6240.5, 7922.72, 7924.05, 8485.43, 8486.45, 9047.57, 10729, 11290, 12411.1, 14092.4, 12412.4
   },
  {
  6252.48, 7937.96, 7939.29, 8501.75, 8502.78, 9064.99, 10749.7, 11311.8, 12435.1, 14119.6, 12436.3
   },
  {
  6264.45, 7953.2, 7954.53, 8518.08, 8519.1, 9082.4, 10770.4, 11333.5, 12459, 14146.8, 12460.2
   },
  {
  6276.43, 7968.44, 7969.77, 8534.41, 8535.43, 9099.82, 10791.1, 11355.3, 12482.9, 14174, 12484.2
   },
  {
  6288.41, 7983.68, 7985.01, 8550.73, 8551.76, 9117.23, 10811.7, 11377.1, 12506.9, 14201.2, 12508.1
   },
  {
  6300.39, 7998.92, 8000.25, 8567.06, 8568.08, 9134.65, 10832.4, 11398.8, 12530.8, 14228.4, 12532.1
   },
  {
  6312.37, 8014.15, 8015.49, 8583.39, 8584.41, 9152.06, 10853.1, 11420.6, 12554.8, 14255.6, 12556
   },
  {
  6324.34, 8029.39, 8030.73, 8599.72, 8600.74, 9169.48, 10873.8, 11442.4, 12578.7, 14282.8, 12579.9
   },
  {
  6336.32, 8044.63, 8045.97, 8616.04, 8617.07, 9186.89, 10894.5, 11464.1, 12602.6, 14310, 12603.9
   },
  {
  6348.3, 8059.87, 8061.21, 8632.37, 8633.39, 9204.31, 10915.1, 11485.9, 12626.6, 14337.2, 12627.8
   },
  {
  6360.28, 8075.11, 8076.45, 8648.7, 8649.72, 9221.72, 10935.8, 11507.6, 12650.5, 14364.4, 12651.8
   },
  {
  6372.25, 8090.35, 8091.69, 8665.02, 8666.05, 9239.13, 10956.5, 11529.4, 12674.4, 14391.6, 12675.7
   },
  {
  6384.23, 8105.59, 8106.93, 8681.35, 8682.37, 9256.55, 10977.2, 11551.2, 12698.4, 14418.8, 12699.6
   },
  {
  6396.21, 8120.83, 8122.17, 8697.68, 8698.7, 9273.96, 10997.8, 11572.9, 12722.3, 14446, 12723.6
   },
  {
  6408.19, 8136.07, 8137.41, 8714.01, 8715.03, 9291.38, 11018.5, 11594.7, 12746.3, 14473.2, 12747.5
   },
  {
  6420.16, 8151.31, 8152.65, 8730.33, 8731.36, 9308.79, 11039.2, 11616.5, 12770.2, 14500.4, 12771.4
   },
  {
  6432.14, 8166.55, 8167.89, 8746.66, 8747.68, 9326.21, 11059.9, 11638.2, 12794.1, 14527.6, 12795.4
   },
  {
  6444.12, 8181.79, 8183.13, 8762.99, 8764.01, 9343.62, 11080.5, 11660, 12818.1, 14554.8, 12819.3
   },
  {
  6456.1, 8197.03, 8198.37, 8779.31, 8780.34, 9361.03, 11101.2, 11681.8, 12842, 14582, 12843.3
   },
  {
  6468.07, 8212.27, 8213.61, 8795.64, 8796.66, 9378.45, 11121.9, 11703.5, 12866, 14609.2, 12867.2
   },
  {
  6480.05, 8227.51, 8228.85, 8811.97, 8812.99, 9395.86, 11142.6, 11725.3, 12889.9, 14636.4, 12891.1
   },
  {
  6492.03, 8242.75, 8244.09, 8828.29, 8829.32, 9413.28, 11163.2, 11747, 12913.8, 14663.6, 12915.1
   },
  {
  6504, 8257.99, 8259.33, 8844.62, 8845.64, 9430.69, 11183.9, 11768.8, 12937.8, 14690.8, 12939
   },
  {
  6515.98, 8273.23, 8274.57, 8860.95, 8861.97, 9448.11, 11204.6, 11790.6, 12961.7, 14718, 12963
   },
  {
  6527.96, 8288.47, 8289.81, 8877.27, 8878.3, 9465.52, 11225.3, 11812.3, 12985.6, 14745.2, 12986.9
   },
  {
  6539.94, 8303.71, 8305.04, 8893.6, 8894.62, 9482.93, 11245.9, 11834.1, 13009.6, 14772.4, 13010.8
   },
  {
  6551.91, 8318.94, 8320.28, 8909.93, 8910.95, 9500.35, 11266.6, 11855.9, 13033.5, 14799.6, 13034.8
   },
  {
  6563.89, 8334.18, 8335.52, 8926.25, 8927.28, 9517.76, 11287.3, 11877.6, 13057.5, 14826.8, 13058.7
   },
  {
  6575.87, 8349.42, 8350.76, 8942.58, 8943.6, 9535.17, 11308, 11899.4, 13081.4, 14854, 13082.6
   },
  {
  6587.84, 8364.66, 8366, 8958.91, 8959.93, 9552.59, 11328.7, 11921.2, 13105.3, 14881.2, 13106.6
   },
  {
  6599.82, 8379.9, 8381.24, 8975.23, 8976.26, 9570, 11349.3, 11942.9, 13129.3, 14908.4, 13130.5
   },
  {
  6611.8, 8395.14, 8396.48, 8991.56, 8992.58, 9587.42, 11370, 11964.7, 13153.2, 14935.6, 13154.5
   },
  {
  6623.77, 8410.38, 8411.72, 9007.88, 9008.91, 9604.83, 11390.7, 11986.4, 13177.2, 14962.8, 13178.4
   },
  {
  6635.75, 8425.62, 8426.96, 9024.21, 9025.23, 9622.24, 11411.4, 12008.2, 13201.1, 14990, 13202.3
   },
  {
  6647.73, 8440.86, 8442.2, 9040.54, 9041.56, 9639.66, 11432, 12030, 13225, 15017.2, 13226.3
   },
  {
  6659.71, 8456.1, 8457.44, 9056.86, 9057.89, 9657.07, 11452.7, 12051.7, 13249, 15044.4, 13250.2
   },
  {
  6671.68, 8471.34, 8472.67, 9073.19, 9074.21, 9674.48, 11473.4, 12073.5, 13272.9, 15071.6, 13274.1
   },
  {
  6683.66, 8486.57, 8487.91, 9089.52, 9090.54, 9691.9, 11494.1, 12095.3, 13296.8, 15098.8, 13298.1
   },
  {
  6695.64, 8501.81, 8503.15, 9105.84, 9106.87, 9709.31, 11514.7, 12117, 13320.8, 15126, 13322
   },
  {
  6707.61, 8517.05, 8518.39, 9122.17, 9123.19, 9726.73, 11535.4, 12138.8, 13344.7, 15153.2, 13346
   },
  {
  6719.59, 8532.29, 8533.63, 9138.5, 9139.52, 9744.14, 11556.1, 12160.5, 13368.7, 15180.4, 13369.9
   },
  {
  6731.57, 8547.53, 8548.87, 9154.82, 9155.84, 9761.55, 11576.8, 12182.3, 13392.6, 15207.6, 13393.8
   },
  {
  6743.54, 8562.77, 8564.11, 9171.15, 9172.17, 9778.97, 11597.4, 12204.1, 13416.5, 15234.8, 13417.8
   },
  {
  6755.52, 8578.01, 8579.35, 9187.47, 9188.5, 9796.38, 11618.1, 12225.8, 13440.5, 15262, 13441.7
   },
  {
  6767.5, 8593.25, 8594.58, 9203.8, 9204.82, 9813.79, 11638.8, 12247.6, 13464.4, 15289.2, 13465.6
   },
  {
  6779.47, 8608.48, 8609.82, 9220.13, 9221.15, 9831.21, 11659.5, 12269.4, 13488.3, 15316.4, 13489.6
   },
  {
  6791.45, 8623.72, 8625.06, 9236.45, 9237.47, 9848.62, 11680.1, 12291.1, 13512.3, 15343.6, 13513.5
   },
  {
  6803.43, 8638.96, 8640.3, 9252.78, 9253.8, 9866.03, 11700.8, 12312.9, 13536.2, 15370.8, 13537.5
   },
  {
  6815.4, 8654.2, 8655.54, 9269.1, 9270.13, 9883.45, 11721.5, 12334.6, 13560.2, 15398, 13561.4
   },
  {
  6827.38, 8669.44, 8670.78, 9285.43, 9286.45, 9900.86, 11742.2, 12356.4, 13584.1, 15425.2, 13585.3
   },
  {
  6839.35, 8684.68, 8686.02, 9301.76, 9302.78, 9918.27, 11762.8, 12378.2, 13608, 15452.4, 13609.3
   },
  {
  6851.33, 8699.92, 8701.25, 9318.08, 9319.1, 9935.69, 11783.5, 12399.9, 13632, 15479.6, 13633.2
   },
  {
  6863.31, 8715.15, 8716.49, 9334.41, 9335.43, 9953.1, 11804.2, 12421.7, 13655.9, 15506.8, 13657.1
   },
  {
  6875.28, 8730.39, 8731.73, 9350.73, 9351.76, 9970.51, 11824.9, 12443.5, 13679.8, 15534, 13681.1
   },
  {
  6887.26, 8745.63, 8746.97, 9367.06, 9368.08, 9987.93, 11845.5, 12465.2, 13703.8, 15561.2, 13705
   },
  {
  6899.24, 8760.87, 8762.21, 9383.38, 9384.41, 10005.3, 11866.2, 12487, 13727.7, 15588.4, 13729
   },
  {
  6911.21, 8776.11, 8777.45, 9399.71, 9400.73, 10022.8, 11886.9, 12508.8, 13751.7, 15615.6, 13752.9
   },
  {
  6923.19, 8791.35, 8792.68, 9416.04, 9417.06, 10040.2, 11907.6, 12530.5, 13775.6, 15642.8, 13776.8
   },
  {
  6935.16, 8806.58, 8807.92, 9432.36, 9433.38, 10057.6, 11928.2, 12552.3, 13799.5, 15670, 13800.8
   },
  {
  6947.14, 8821.82, 8823.16, 9448.69, 9449.71, 10075, 11948.9, 12574, 13823.5, 15697.2, 13824.7
   },
  {
  6959.12, 8837.06, 8838.4, 9465.01, 9466.04, 10092.4, 11969.6, 12595.8, 13847.4, 15724.4, 13848.6
   },
  {
  6971.09, 8852.3, 8853.64, 9481.34, 9482.36, 10109.8, 11990.3, 12617.6, 13871.3, 15751.6, 13872.6
   },
  {
  6983.07, 8867.54, 8868.88, 9497.66, 9498.69, 10127.2, 12010.9, 12639.3, 13895.3, 15778.8, 13896.5
   },
  {
  6995.05, 8882.78, 8884.11, 9513.99, 9515.01, 10144.6, 12031.6, 12661.1, 13919.2, 15806, 13920.5
   },
  {
  7007.02, 8898.01, 8899.35, 9530.31, 9531.34, 10162.1, 12052.3, 12682.8, 13943.2, 15833.2, 13944.4
   },
  {
  7019, 8913.25, 8914.59, 9546.64, 9547.66, 10179.5, 12073, 12704.6, 13967.1, 15860.4, 13968.3
   },
  {
  7030.97, 8928.49, 8929.83, 9562.97, 9563.99, 10196.9, 12093.6, 12726.4, 13991, 15887.6, 13992.3
   },
  {
  7042.95, 8943.73, 8945.07, 9579.29, 9580.31, 10214.3, 12114.3, 12748.1, 14015, 15914.8, 14016.2
   },
  {
  7054.93, 8958.97, 8960.3, 9595.62, 9596.64, 10231.7, 12135, 12769.9, 14038.9, 15942, 14040.1
   },
  {
  7066.9, 8974.2, 8975.54, 9611.94, 9612.96, 10249.1, 12155.7, 12791.7, 14062.8, 15969.2, 14064.1
   },
  {
  7078.88, 8989.44, 8990.78, 9628.27, 9629.29, 10266.5, 12176.3, 12813.4, 14086.8, 15996.4, 14088
   },
  {
  7090.85, 9004.68, 9006.02, 9644.59, 9645.62, 10283.9, 12197, 12835.2, 14110.7, 16023.6, 14112
   },
  {
  7102.83, 9019.92, 9021.26, 9660.92, 9661.94, 10301.4, 12217.7, 12856.9, 14134.6, 16050.8, 14135.9
   },
  {
  7114.8, 9035.16, 9036.49, 9677.24, 9678.27, 10318.8, 12238.4, 12878.7, 14158.6, 16078, 14159.8
   },
  {
  7126.78, 9050.39, 9051.73, 9693.57, 9694.59, 10336.2, 12259, 12900.5, 14182.5, 16105.2, 14183.8
   },
  {
  7138.76, 9065.63, 9066.97, 9709.89, 9710.92, 10353.6, 12279.7, 12922.2, 14206.5, 16132.4, 14207.7
   },
  {
  7150.73, 9080.87, 9082.21, 9726.22, 9727.24, 10371, 12300.4, 12944, 14230.4, 16159.6, 14231.6
   },
  {
  7162.71, 9096.11, 9097.45, 9742.54, 9743.57, 10388.4, 12321.1, 12965.8, 14254.3, 16186.8, 14255.6
   },
  {
  7174.68, 9111.34, 9112.68, 9758.87, 9759.89, 10405.8, 12341.7, 12987.5, 14278.3, 16214, 14279.5
   },
  {
  7186.66, 9126.58, 9127.92, 9775.19, 9776.22, 10423.2, 12362.4, 13009.3, 14302.2, 16241.2, 14303.4
   },
  {
  7198.63, 9141.82, 9143.16, 9791.52, 9792.54, 10440.7, 12383.1, 13031, 14326.1, 16268.4, 14327.4
   },
  {
  7210.61, 9157.06, 9158.4, 9807.84, 9808.87, 10458.1, 12403.8, 13052.8, 14350.1, 16295.6, 14351.3
   },
  {
  7222.59, 9172.3, 9173.63, 9824.17, 9825.19, 10475.5, 12424.4, 13074.6, 14374, 16322.8, 14375.3
   },
  {
  7234.56, 9187.53, 9188.87, 9840.49, 9841.52, 10492.9, 12445.1, 13096.3, 14398, 16350, 14399.2
   },
  {
  7246.54, 9202.77, 9204.11, 9856.82, 9857.84, 10510.3, 12465.8, 13118.1, 14421.9, 16377.2, 14423.1
   },
  {
  7258.51, 9218.01, 9219.35, 9873.14, 9874.17, 10527.7, 12486.5, 13139.9, 14445.8, 16404.4, 14447.1
   },
  {
  7270.49, 9233.25, 9234.58, 9889.47, 9890.49, 10545.1, 12507.1, 13161.6, 14469.8, 16431.6, 14471
   },
  {
  7282.46, 9248.48, 9249.82, 9905.79, 9906.82, 10562.5, 12527.8, 13183.4, 14493.7, 16458.8, 14494.9
   },
  {
  7294.44, 9263.72, 9265.06, 9922.12, 9923.14, 10580, 12548.5, 13205.1, 14517.6, 16486, 14518.9
   },
  {
  7306.41, 9278.96, 9280.3, 9938.44, 9939.47, 10597.4, 12569.2, 13226.9, 14541.6, 16513.2, 14542.8
   },
  {
  7318.39, 9294.2, 9295.54, 9954.77, 9955.79, 10614.8, 12589.8, 13248.7, 14565.5, 16540.4, 14566.7
   },
  {
  7330.37, 9309.43, 9310.77, 9971.09, 9972.12, 10632.2, 12610.5, 13270.4, 14589.4, 16567.6, 14590.7
   },
  {
  7342.34, 9324.67, 9326.01, 9987.42, 9988.44, 10649.6, 12631.2, 13292.2, 14613.4, 16594.8, 14614.6
   },
  {
  7354.32, 9339.91, 9341.25, 10003.7, 10004.8, 10667, 12651.9, 13313.9, 14637.3, 16622, 14638.6
   },
  {
  7366.29, 9355.15, 9356.48, 10020.1, 10021.1, 10684.4, 12672.5, 13335.7, 14661.3, 16649.2, 14662.5
   },
  {
  7378.27, 9370.38, 9371.72, 10036.4, 10037.4, 10701.8, 12693.2, 13357.5, 14685.2, 16676.4, 14686.4
   },
  {
  7390.24, 9385.62, 9386.96, 10052.7, 10053.7, 10719.3, 12713.9, 13379.2, 14709.1, 16703.6, 14710.4
   },
  {
  7402.22, 9400.86, 9402.2, 10069, 10070.1, 10736.7, 12734.6, 13401, 14733.1, 16730.8, 14734.3
   },
  {
  7414.19, 9416.1, 9417.43, 10085.4, 10086.4, 10754.1, 12755.2, 13422.8, 14757, 16758, 14758.2
   },
  {
  7426.17, 9431.33, 9432.67, 10101.7, 10102.7, 10771.5, 12775.9, 13444.5, 14780.9, 16785.2, 14782.2
   },
  {
  7438.14, 9446.57, 9447.91, 10118, 10119, 10788.9, 12796.6, 13466.3, 14804.9, 16812.4, 14806.1
   },
  {
  7450.12, 9461.81, 9463.15, 10134.3, 10135.4, 10806.3, 12817.2, 13488, 14828.8, 16839.6, 14830
   },
  {
  7462.09, 9477.04, 9478.38, 10150.7, 10151.7, 10823.7, 12837.9, 13509.8, 14852.7, 16866.8, 14854
   },
  {
  7474.07, 9492.28, 9493.62, 10167, 10168, 10841.1, 12858.6, 13531.6, 14876.7, 16894, 14877.9
   },
  {
  7486.04, 9507.52, 9508.86, 10183.3, 10184.3, 10858.6, 12879.3, 13553.3, 14900.6, 16921.1, 14901.9
   },
  {
  7498.02, 9522.76, 9524.09, 10199.6, 10200.7, 10876, 12899.9, 13575.1, 14924.5, 16948.3, 14925.8
   },
  {
  7509.99, 9537.99, 9539.33, 10216, 10217, 10893.4, 12920.6, 13596.8, 14948.5, 16975.5, 14949.7
   },
  {
  7521.97, 9553.23, 9554.57, 10232.3, 10233.3, 10910.8, 12941.3, 13618.6, 14972.4, 17002.7, 14973.7
   },
  {
  7533.94, 9568.47, 9569.81, 10248.6, 10249.6, 10928.2, 12962, 13640.4, 14996.4, 17029.9, 14997.6
   },
  {
  7545.92, 9583.7, 9585.04, 10264.9, 10266, 10945.6, 12982.6, 13662.1, 15020.3, 17057.1, 15021.5
   },
  {
  7557.89, 9598.94, 9600.28, 10281.3, 10282.3, 10963, 13003.3, 13683.9, 15044.2, 17084.3, 15045.5
   },
  {
  7569.87, 9614.18, 9615.52, 10297.6, 10298.6, 10980.4, 13024, 13705.7, 15068.2, 17111.5, 15069.4
   },
  {
  7581.84, 9629.42, 9630.75, 10313.9, 10314.9, 10997.8, 13044.7, 13727.4, 15092.1, 17138.7, 15093.3
   },
  {
  7593.82, 9644.65, 9645.99, 10330.2, 10331.3, 11015.3, 13065.3, 13749.2, 15116, 17165.9, 15117.3
   },
  {
  7605.79, 9659.89, 9661.23, 10346.6, 10347.6, 11032.7, 13086, 13770.9, 15140, 17193.1, 15141.2
   },
  {
  7617.77, 9675.13, 9676.47, 10362.9, 10363.9, 11050.1, 13106.7, 13792.7, 15163.9, 17220.3, 15165.2
   },
  {
  7629.74, 9690.36, 9691.7, 10379.2, 10380.2, 11067.5, 13127.4, 13814.5, 15187.8, 17247.5, 15189.1
   },
  {
  7641.72, 9705.6, 9706.94, 10395.5, 10396.6, 11084.9, 13148, 13836.2, 15211.8, 17274.7, 15213
   },
  {
  7653.69, 9720.84, 9722.18, 10411.9, 10412.9, 11102.3, 13168.7, 13858, 15235.7, 17301.9, 15237
   },
  {
  7665.67, 9736.07, 9737.41, 10428.2, 10429.2, 11119.7, 13189.4, 13879.7, 15259.7, 17329.1, 15260.9
   },
  {
  7677.64, 9751.31, 9752.65, 10444.5, 10445.5, 11137.1, 13210.1, 13901.5, 15283.6, 17356.3, 15284.8
   },
  {
  7689.62, 9766.55, 9767.89, 10460.8, 10461.9, 11154.5, 13230.7, 13923.3, 15307.5, 17383.5, 15308.8
   },
  {
  7701.59, 9781.78, 9783.12, 10477.2, 10478.2, 11172, 13251.4, 13945, 15331.5, 17410.7, 15332.7
   },
  {
  7713.57, 9797.02, 9798.36, 10493.5, 10494.5, 11189.4, 13272.1, 13966.8, 15355.4, 17437.9, 15356.6
   },
  {
  7725.54, 9812.26, 9813.6, 10509.8, 10510.8, 11206.8, 13292.7, 13988.5, 15379.3, 17465.1, 15380.6
   },
  {
  7737.51, 9827.49, 9828.83, 10526.1, 10527.1, 11224.2, 13313.4, 14010.3, 15403.3, 17492.3, 15404.5
   },
  {
  7749.49, 9842.73, 9844.07, 10542.4, 10543.5, 11241.6, 13334.1, 14032.1, 15427.2, 17519.5, 15428.4
   },
  {
  7761.46, 9857.97, 9859.31, 10558.8, 10559.8, 11259, 13354.8, 14053.8, 15451.1, 17546.7, 15452.4
   },
  {
  7773.44, 9873.2, 9874.54, 10575.1, 10576.1, 11276.4, 13375.4, 14075.6, 15475.1, 17573.9, 15476.3
   },
  {
  7785.41, 9888.44, 9889.78, 10591.4, 10592.4, 11293.8, 13396.1, 14097.4, 15499, 17601.1, 15500.3
   },
  {
  7797.39, 9903.68, 9905.02, 10607.7, 10608.8, 11311.3, 13416.8, 14119.1, 15522.9, 17628.3, 15524.2
   },
  {
  7809.36, 9918.91, 9920.25, 10624.1, 10625.1, 11328.7, 13437.5, 14140.9, 15546.9, 17655.5, 15548.1
   },
  {
  7821.34, 9934.15, 9935.49, 10640.4, 10641.4, 11346.1, 13458.1, 14162.6, 15570.8, 17682.7, 15572.1
   },
  {
  7833.31, 9949.39, 9950.73, 10656.7, 10657.7, 11363.5, 13478.8, 14184.4, 15594.8, 17709.9, 15596
   },
  {
  7845.29, 9964.62, 9965.96, 10673, 10674.1, 11380.9, 13499.5, 14206.2, 15618.7, 17737.1, 15619.9
   },
  {
  7857.26, 9979.86, 9981.2, 10689.4, 10690.4, 11398.3, 13520.2, 14227.9, 15642.6, 17764.3, 15643.9
   },
  {
  7869.23, 9995.1, 9996.44, 10705.7, 10706.7, 11415.7, 13540.8, 14249.7, 15666.6, 17791.5, 15667.8
   },
  {
  7881.21, 10010.3, 10011.7, 10722, 10723, 11433.1, 13561.5, 14271.4, 15690.5, 17818.7, 15691.7
   },
  {
  7893.18, 10025.6, 10026.9, 10738.3, 10739.4, 11450.5, 13582.2, 14293.2, 15714.4, 17845.9, 15715.7
   },
  {
  7905.16, 10040.8, 10042.1, 10754.7, 10755.7, 11468, 13602.8, 14315, 15738.4, 17873.1, 15739.6
   },
  {
  7917.13, 10056, 10057.4, 10771, 10772, 11485.4, 13623.5, 14336.7, 15762.3, 17900.3, 15763.5
   },
  {
  7929.11, 10071.3, 10072.6, 10787.3, 10788.3, 11502.8, 13644.2, 14358.5, 15786.2, 17927.5, 15787.5
   },
  {
  7941.08, 10086.5, 10087.9, 10803.6, 10804.7, 11520.2, 13664.9, 14380.2, 15810.2, 17954.7, 15811.4
   },
  {
  7953.05, 10101.8, 10103.1, 10820, 10821, 11537.6, 13685.5, 14402, 15834.1, 17981.9, 15835.3
   },
  {
  7965.03, 10117, 10118.3, 10836.3, 10837.3, 11555, 13706.2, 14423.8, 15858, 18009.1, 15859.3
   },
  {
  7977, 10132.2, 10133.6, 10852.6, 10853.6, 11572.4, 13726.9, 14445.5, 15882, 18036.3, 15883.2
   },
  {
  7988.98, 10147.5, 10148.8, 10868.9, 10869.9, 11589.8, 13747.6, 14467.3, 15905.9, 18063.5, 15907.2
   },
  {
  8000.95, 10162.7, 10164, 10885.2, 10886.3, 11607.2, 13768.2, 14489, 15929.8, 18090.7, 15931.1
   },
  {
  8012.92, 10177.9, 10179.3, 10901.6, 10902.6, 11624.7, 13788.9, 14510.8, 15953.8, 18117.8, 15955
   },
  {
  8024.9, 10193.2, 10194.5, 10917.9, 10918.9, 11642.1, 13809.6, 14532.6, 15977.7, 18145, 15979
   },
  {
  8036.87, 10208.4, 10209.7, 10934.2, 10935.2, 11659.5, 13830.3, 14554.3, 16001.7, 18172.2, 16002.9
   },
  {
  8048.85, 10223.6, 10225, 10950.5, 10951.6, 11676.9, 13850.9, 14576.1, 16025.6, 18199.4, 16026.8
   },
  {
  8060.82, 10238.9, 10240.2, 10966.9, 10967.9, 11694.3, 13871.6, 14597.8, 16049.5, 18226.6, 16050.8
   },
  {
  8072.8, 10254.1, 10255.5, 10983.2, 10984.2, 11711.7, 13892.3, 14619.6, 16073.5, 18253.8, 16074.7
   },
  {
  8084.77, 10269.4, 10270.7, 10999.5, 11000.5, 11729.1, 13912.9, 14641.4, 16097.4, 18281, 16098.6
   },
  {
  8096.74, 10284.6, 10285.9, 11015.8, 11016.9, 11746.5, 13933.6, 14663.1, 16121.3, 18308.2, 16122.6
   },
  {
  8108.72, 10299.8, 10301.2, 11032.2, 11033.2, 11763.9, 13954.3, 14684.9, 16145.3, 18335.4, 16146.5
   },
  {
  8120.69, 10315.1, 10316.4, 11048.5, 11049.5, 11781.4, 13975, 14706.6, 16169.2, 18362.6, 16170.4
   },
  {
  8132.67, 10330.3, 10331.6, 11064.8, 11065.8, 11798.8, 13995.6, 14728.4, 16193.1, 18389.8, 16194.4
   },
  {
  8144.64, 10345.5, 10346.9, 11081.1, 11082.2, 11816.2, 14016.3, 14750.2, 16217.1, 18417, 16218.3
   },
  {
  8156.61, 10360.8, 10362.1, 11097.5, 11098.5, 11833.6, 14037, 14771.9, 16241, 18444.2, 16242.2
   },
  {
  8168.59, 10376, 10377.3, 11113.8, 11114.8, 11851, 14057.7, 14793.7, 16264.9, 18471.4, 16266.2
   },
  {
  8180.56, 10391.2, 10392.6, 11130.1, 11131.1, 11868.4, 14078.3, 14815.4, 16288.9, 18498.6, 16290.1
   },
  {
  8192.53, 10406.5, 10407.8, 11146.4, 11147.4, 11885.8, 14099, 14837.2, 16312.8, 18525.8, 16314
   },
  {
  8204.51, 10421.7, 10423, 11162.7, 11163.8, 11903.2, 14119.7, 14859, 16336.7, 18553, 16338
   },
  {
  8216.48, 10436.9, 10438.3, 11179.1, 11180.1, 11920.6, 14140.3, 14880.7, 16360.7, 18580.2, 16361.9
   },
  {
  8228.46, 10452.2, 10453.5, 11195.4, 11196.4, 11938, 14161, 14902.5, 16384.6, 18607.4, 16385.9
   },
  {
  8240.43, 10467.4, 10468.8, 11211.7, 11212.7, 11955.5, 14181.7, 14924.2, 16408.5, 18634.6, 16409.8
   },
  {
  8252.4, 10482.7, 10484, 11228, 11229.1, 11972.9, 14202.4, 14946, 16432.5, 18661.8, 16433.7
   },
  {
  8264.38, 10497.9, 10499.2, 11244.4, 11245.4, 11990.3, 14223, 14967.8, 16456.4, 18689, 16457.7
   },
  {
  8276.35, 10513.1, 10514.5, 11260.7, 11261.7, 12007.7, 14243.7, 14989.5, 16480.3, 18716.2, 16481.6
   },
  {
  8288.32, 10528.4, 10529.7, 11277, 11278, 12025.1, 14264.4, 15011.3, 16504.3, 18743.4, 16505.5
   },
  {
  8300.3, 10543.6, 10544.9, 11293.3, 11294.4, 12042.5, 14285.1, 15033, 16528.2, 18770.6, 16529.5
   },
  {
  8312.27, 10558.8, 10560.2, 11309.7, 11310.7, 12059.9, 14305.7, 15054.8, 16552.2, 18797.8, 16553.4
   },
  {
  8324.25, 10574.1, 10575.4, 11326, 11327, 12077.3, 14326.4, 15076.6, 16576.1, 18825, 16577.3
   },
  {
  8336.22, 10589.3, 10590.6, 11342.3, 11343.3, 12094.7, 14347.1, 15098.3, 16600, 18852.2, 16601.3
   },
  {
  8348.19, 10604.5, 10605.9, 11358.6, 11359.6, 12112.2, 14367.7, 15120.1, 16624, 18879.4, 16625.2
   },
  {
  8360.17, 10619.8, 10621.1, 11374.9, 11376, 12129.6, 14388.4, 15141.8, 16647.9, 18906.6, 16649.1
   },
  {
  8372.14, 10635, 10636.4, 11391.3, 11392.3, 12147, 14409.1, 15163.6, 16671.8, 18933.8, 16673.1
   },
  {
  8384.11, 10650.2, 10651.6, 11407.6, 11408.6, 12164.4, 14429.8, 15185.4, 16695.8, 18961, 16697
   },
  {
  8396.09, 10665.5, 10666.8, 11423.9, 11424.9, 12181.8, 14450.4, 15207.1, 16719.7, 18988.2, 16720.9
   },
  {
  8408.06, 10680.7, 10682.1, 11440.2, 11441.3, 12199.2, 14471.1, 15228.9, 16743.6, 19015.3, 16744.9
   },
  {
  8420.03, 10696, 10697.3, 11456.6, 11457.6, 12216.6, 14491.8, 15250.6, 16767.6, 19042.5, 16768.8
   },
  {
  8432.01, 10711.2, 10712.5, 11472.9, 11473.9, 12234, 14512.5, 15272.4, 16791.5, 19069.7, 16792.7
   },
  {
  8443.98, 10726.4, 10727.8, 11489.2, 11490.2, 12251.4, 14533.1, 15294.2, 16815.4, 19096.9, 16816.7
   },
  {
  8455.95, 10741.7, 10743, 11505.5, 11506.6, 12268.8, 14553.8, 15315.9, 16839.4, 19124.1, 16840.6
   },
  {
  8467.93, 10756.9, 10758.2, 11521.9, 11522.9, 12286.3, 14574.5, 15337.7, 16863.3, 19151.3, 16864.5
   },
  {
  8479.9, 10772.1, 10773.5, 11538.2, 11539.2, 12303.7, 14595.1, 15359.4, 16887.2, 19178.5, 16888.5
   },
  {
  8491.87, 10787.4, 10788.7, 11554.5, 11555.5, 12321.1, 14615.8, 15381.2, 16911.2, 19205.7, 16912.4
   },
  {
  8503.85, 10802.6, 10803.9, 11570.8, 11571.8, 12338.5, 14636.5, 15403, 16935.1, 19232.9, 16936.3
   },
  {
  8515.82, 10817.8, 10819.2, 11587.1, 11588.2, 12355.9, 14657.2, 15424.7, 16959, 19260.1, 16960.3
   },
  {
  8527.8, 10833.1, 10834.4, 11603.5, 11604.5, 12373.3, 14677.8, 15446.5, 16983, 19287.3, 16984.2
   },
  {
  8539.77, 10848.3, 10849.6, 11619.8, 11620.8, 12390.7, 14698.5, 15468.2, 17006.9, 19314.5, 17008.2
   },
  {
  8551.74, 10863.5, 10864.9, 11636.1, 11637.1, 12408.1, 14719.2, 15490, 17030.8, 19341.7, 17032.1
   },
  {
  8563.71, 10878.8, 10880.1, 11652.4, 11653.5, 12425.5, 14739.8, 15511.8, 17054.8, 19368.9, 17056
   },
  {
  8575.69, 10894, 10895.4, 11668.8, 11669.8, 12442.9, 14760.5, 15533.5, 17078.7, 19396.1, 17080
   },
  {
  8587.66, 10909.3, 10910.6, 11685.1, 11686.1, 12460.4, 14781.2, 15555.3, 17102.6, 19423.3, 17103.9
   },
  {
  8599.63, 10924.5, 10925.8, 11701.4, 11702.4, 12477.8, 14801.9, 15577, 17126.6, 19450.5, 17127.8
   },
  {
  8611.61, 10939.7, 10941.1, 11717.7, 11718.8, 12495.2, 14822.5, 15598.8, 17150.5, 19477.7, 17151.8
   },
  {
  8623.58, 10955, 10956.3, 11734.1, 11735.1, 12512.6, 14843.2, 15620.6, 17174.4, 19504.9, 17175.7
   },
  {
  8635.55, 10970.2, 10971.5, 11750.4, 11751.4, 12530, 14863.9, 15642.3, 17198.4, 19532.1, 17199.6
   },
  {
  8647.53, 10985.4, 10986.8, 11766.7, 11767.7, 12547.4, 14884.6, 15664.1, 17222.3, 19559.3, 17223.6
   },
  {
  8659.5, 11000.7, 11002, 11783, 11784, 12564.8, 14905.2, 15685.8, 17246.3, 19586.5, 17247.5
   },
  {
  8671.47, 11015.9, 11017.2, 11799.3, 11800.4, 12582.2, 14925.9, 15707.6, 17270.2, 19613.7, 17271.4
   },
  {
  8683.45, 11031.1, 11032.5, 11815.7, 11816.7, 12599.6, 14946.6, 15729.4, 17294.1, 19640.9, 17295.4
   },
  {
  8695.42, 11046.4, 11047.7, 11832, 11833, 12617, 14967.2, 15751.1, 17318.1, 19668.1, 17319.3
   },
  {
  8707.39, 11061.6, 11062.9, 11848.3, 11849.3, 12634.5, 14987.9, 15772.9, 17342, 19695.3, 17343.2
   },
  {
  8719.37, 11076.8, 11078.2, 11864.6, 11865.7, 12651.9, 15008.6, 15794.6, 17365.9, 19722.5, 17367.2
   },
  {
  8731.34, 11092.1, 11093.4, 11881, 11882, 12669.3, 15029.3, 15816.4, 17389.9, 19749.7, 17391.1
   },
  {
  8743.31, 11107.3, 11108.6, 11897.3, 11898.3, 12686.7, 15049.9, 15838.2, 17413.8, 19776.8, 17415
   },
  {
  8755.29, 11122.5, 11123.9, 11913.6, 11914.6, 12704.1, 15070.6, 15859.9, 17437.7, 19804, 17439
   },
  {
  8767.26, 11137.8, 11139.1, 11929.9, 11930.9, 12721.5, 15091.3, 15881.7, 17461.7, 19831.2, 17462.9
   },
  {
  8779.23, 11153, 11154.4, 11946.2, 11947.3, 12738.9, 15111.9, 15903.4, 17485.6, 19858.4, 17486.8
   },
  {
  8791.2, 11168.2, 11169.6, 11962.6, 11963.6, 12756.3, 15132.6, 15925.2, 17509.5, 19885.6, 17510.8
   },
  {
  8803.18, 11183.5, 11184.8, 11978.9, 11979.9, 12773.7, 15153.3, 15947, 17533.5, 19912.8, 17534.7
   },
  {
  8815.15, 11198.7, 11200.1, 11995.2, 11996.2, 12791.1, 15174, 15968.7, 17557.4, 19940, 17558.6
   },
  {
  8827.12, 11214, 11215.3, 12011.5, 12012.6, 12808.6, 15194.6, 15990.5, 17581.3, 19967.2, 17582.6
   },
  {
  8839.1, 11229.2, 11230.5, 12027.9, 12028.9, 12826, 15215.3, 16012.2, 17605.3, 19994.4, 17606.5
   },
  {
  8851.07, 11244.4, 11245.8, 12044.2, 12045.2, 12843.4, 15236, 16034, 17629.2, 20021.6, 17630.4
   },
  {
  8863.04, 11259.7, 11261, 12060.5, 12061.5, 12860.8, 15256.6, 16055.7, 17653.1, 20048.8, 17654.4
   },
  {
  8875.01, 11274.9, 11276.2, 12076.8, 12077.8, 12878.2, 15277.3, 16077.5, 17677.1, 20076, 17678.3
   },
  {
  8886.99, 11290.1, 11291.5, 12093.1, 12094.2, 12895.6, 15298, 16099.3, 17701, 20103.2, 17702.2
   },
  {
  8898.96, 11305.4, 11306.7, 12109.5, 12110.5, 12913, 15318.7, 16121, 17724.9, 20130.4, 17726.2
   },
  {
  8910.93, 11320.6, 11321.9, 12125.8, 12126.8, 12930.4, 15339.3, 16142.8, 17748.9, 20157.6, 17750.1
   },
  {
  8922.91, 11335.8, 11337.2, 12142.1, 12143.1, 12947.8, 15360, 16164.5, 17772.8, 20184.8, 17774
   },
  {
  8934.88, 11351.1, 11352.4, 12158.4, 12159.5, 12965.2, 15380.7, 16186.3, 17796.7, 20212, 17798
   },
  {
  8946.85, 11366.3, 11367.6, 12174.8, 12175.8, 12982.7, 15401.3, 16208.1, 17820.7, 20239.2, 17821.9
   },
  {
  8958.82, 11381.5, 11382.9, 12191.1, 12192.1, 13000.1, 15422, 16229.8, 17844.6, 20266.4, 17845.8
   },
  {
  8970.8, 11396.8, 11398.1, 12207.4, 12208.4, 13017.5, 15442.7, 16251.6, 17868.5, 20293.6, 17869.8
   },
  {
  8982.77, 11412, 11413.3, 12223.7, 12224.7, 13034.9, 15463.4, 16273.3, 17892.5, 20320.8, 17893.7
   },
  {
  8994.74, 11427.2, 11428.6, 12240, 12241.1, 13052.3, 15484, 16295.1, 17916.4, 20348, 17917.6
   },
  {
  9006.71, 11442.5, 11443.8, 12256.4, 12257.4, 13069.7, 15504.7, 16316.9, 17940.3, 20375.2, 17941.6
   },
  {
  9018.69, 11457.7, 11459.1, 12272.7, 12273.7, 13087.1, 15525.4, 16338.6, 17964.3, 20402.4, 17965.5
   },
  {
  9030.66, 11472.9, 11474.3, 12289, 12290, 13104.5, 15546.1, 16360.4, 17988.2, 20429.6, 17989.4
   },
  {
  9042.63, 11488.2, 11489.5, 12305.3, 12306.4, 13121.9, 15566.7, 16382.1, 18012.1, 20456.7, 18013.4
   },
  {
  9054.61, 11503.4, 11504.8, 12321.7, 12322.7, 13139.3, 15587.4, 16403.9, 18036.1, 20483.9, 18037.3
   },
  {
  9066.58, 11518.7, 11520, 12338, 12339, 13156.7, 15608.1, 16425.7, 18060, 20511.1, 18061.2
   },
  {
  9078.55, 11533.9, 11535.2, 12354.3, 12355.3, 13174.2, 15628.7, 16447.4, 18083.9, 20538.3, 18085.2
   },
  {
  9090.52, 11549.1, 11550.5, 12370.6, 12371.6, 13191.6, 15649.4, 16469.2, 18107.9, 20565.5, 18109.1
   },
  {
  9102.5, 11564.4, 11565.7, 12386.9, 12388, 13209, 15670.1, 16490.9, 18131.8, 20592.7, 18133
   },
  {
  9114.47, 11579.6, 11580.9, 12403.3, 12404.3, 13226.4, 15690.8, 16512.7, 18155.7, 20619.9, 18157
   },
  {
  9126.44, 11594.8, 11596.2, 12419.6, 12420.6, 13243.8, 15711.4, 16534.4, 18179.7, 20647.1, 18180.9
   },
  {
  9138.41, 11610.1, 11611.4, 12435.9, 12436.9, 13261.2, 15732.1, 16556.2, 18203.6, 20674.3, 18204.8
   },
  {
  9150.39, 11625.3, 11626.6, 12452.2, 12453.3, 13278.6, 15752.8, 16578, 18227.5, 20701.5, 18228.8
   },
  {
  9162.36, 11640.5, 11641.9, 12468.6, 12469.6, 13296, 15773.4, 16599.7, 18251.5, 20728.7, 18252.7
   },
  {
  9174.33, 11655.8, 11657.1, 12484.9, 12485.9, 13313.4, 15794.1, 16621.5, 18275.4, 20755.9, 18276.7
   },
  {
  9186.3, 11671, 11672.3, 12501.2, 12502.2, 13330.8, 15814.8, 16643.2, 18299.3, 20783.1, 18300.6
   },
  {
  9198.28, 11686.2, 11687.6, 12517.5, 12518.5, 13348.3, 15835.5, 16665, 18323.3, 20810.3, 18324.5
   },
  {
  9210.25, 11701.5, 11702.8, 12533.8, 12534.9, 13365.7, 15856.1, 16686.8, 18347.2, 20837.5, 18348.5
   },
  {
  9222.22, 11716.7, 11718, 12550.2, 12551.2, 13383.1, 15876.8, 16708.5, 18371.1, 20864.7, 18372.4
   },
  {
  9234.19, 11731.9, 11733.3, 12566.5, 12567.5, 13400.5, 15897.5, 16730.3, 18395.1, 20891.9, 18396.3
   },
  {
  9246.17, 11747.2, 11748.5, 12582.8, 12583.8, 13417.9, 15918.1, 16752, 18419, 20919.1, 18420.3
   },
  {
  9258.14, 11762.4, 11763.7, 12599.1, 12600.2, 13435.3, 15938.8, 16773.8, 18442.9, 20946.3, 18444.2
   },
  {
  9270.11, 11777.6, 11779, 12615.5, 12616.5, 13452.7, 15959.5, 16795.5, 18466.9, 20973.5, 18468.1
   },
  {
  9282.08, 11792.9, 11794.2, 12631.8, 12632.8, 13470.1, 15980.1, 16817.3, 18490.8, 21000.7, 18492.1
   },
  {
  9294.05, 11808.1, 11809.4, 12648.1, 12649.1, 13487.5, 16000.8, 16839.1, 18514.7, 21027.9, 18516
   },
  {
  9306.03, 11823.3, 11824.7, 12664.4, 12665.4, 13504.9, 16021.5, 16860.8, 18538.7, 21055, 18539.9
   },
  {
  9318, 11838.6, 11839.9, 12680.7, 12681.8, 13522.3, 16042.2, 16882.6, 18562.6, 21082.2, 18563.9
   },
  {
  9329.97, 11853.8, 11855.1, 12697.1, 12698.1, 13539.8, 16062.8, 16904.3, 18586.5, 21109.4, 18587.8
   },
  {
  9341.94, 11869, 11870.4, 12713.4, 12714.4, 13557.2, 16083.5, 16926.1, 18610.5, 21136.6, 18611.7
   },
  {
  9353.92, 11884.3, 11885.6, 12729.7, 12730.7, 13574.6, 16104.2, 16947.9, 18634.4, 21163.8, 18635.7
   },
  {
  9365.89, 11899.5, 11900.9, 12746, 12747, 13592, 16124.8, 16969.6, 18658.3, 21191, 18659.6
   },
  {
  9377.86, 11914.7, 11916.1, 12762.3, 12763.4, 13609.4, 16145.5, 16991.4, 18682.3, 21218.2, 18683.5
   },
  {
  9389.83, 11930, 11931.3, 12778.7, 12779.7, 13626.8, 16166.2, 17013.1, 18706.2, 21245.4, 18707.5
   },
  {
  9401.8, 11945.2, 11946.6, 12795, 12796, 13644.2, 16186.9, 17034.9, 18730.1, 21272.6, 18731.4
   },
  {
  9413.78, 11960.5, 11961.8, 12811.3, 12812.3, 13661.6, 16207.5, 17056.7, 18754.1, 21299.8, 18755.3
   },
  {
  9425.75, 11975.7, 11977, 12827.6, 12828.7, 13679, 16228.2, 17078.4, 18778, 21327, 18779.3
   },
  {
  9437.72, 11990.9, 11992.3, 12844, 12845, 13696.4, 16248.9, 17100.2, 18801.9, 21354.2, 18803.2
   },
  {
  9449.69, 12006.2, 12007.5, 12860.3, 12861.3, 13713.8, 16269.5, 17121.9, 18825.9, 21381.4, 18827.1
   },
  {
  9461.67, 12021.4, 12022.7, 12876.6, 12877.6, 13731.3, 16290.2, 17143.7, 18849.8, 21408.6, 18851
   },
  {
  9473.64, 12036.6, 12038, 12892.9, 12893.9, 13748.7, 16310.9, 17165.4, 18873.7, 21435.8, 18875
   },
  {
  9485.61, 12051.9, 12053.2, 12909.2, 12910.3, 13766.1, 16331.6, 17187.2, 18897.7, 21463, 18898.9
   },
  {
  9497.58, 12067.1, 12068.4, 12925.6, 12926.6, 13783.5, 16352.2, 17209, 18921.6, 21490.2, 18922.8
   },
  {
  9509.55, 12082.3, 12083.7, 12941.9, 12942.9, 13800.9, 16372.9, 17230.7, 18945.5, 21517.4, 18946.8
   },
  {
  9521.53, 12097.6, 12098.9, 12958.2, 12959.2, 13818.3, 16393.6, 17252.5, 18969.5, 21544.6, 18970.7
   },
  {
  9533.5, 12112.8, 12114.1, 12974.5, 12975.6, 13835.7, 16414.2, 17274.2, 18993.4, 21571.8, 18994.6
   },
  {
  9545.47, 12128, 12129.4, 12990.9, 12991.9, 13853.1, 16434.9, 17296, 19017.3, 21599, 19018.6
   },
  {
  9557.44, 12143.3, 12144.6, 13007.2, 13008.2, 13870.5, 16455.6, 17317.8, 19041.3, 21626.2, 19042.5
   },
  {
  9569.41, 12158.5, 12159.8, 13023.5, 13024.5, 13887.9, 16476.3, 17339.5, 19065.2, 21653.3, 19066.4
   },
  {
  9581.39, 12173.7, 12175.1, 13039.8, 13040.8, 13905.3, 16496.9, 17361.3, 19089.1, 21680.5, 19090.4
   },
  {
  9593.36, 12189, 12190.3, 13056.1, 13057.2, 13922.7, 16517.6, 17383, 19113.1, 21707.7, 19114.3
   },
  {
  9605.33, 12204.2, 12205.5, 13072.5, 13073.5, 13940.2, 16538.3, 17404.8, 19137, 21734.9, 19138.2
   },
  {
  9617.3, 12219.4, 12220.8, 13088.8, 13089.8, 13957.6, 16558.9, 17426.5, 19160.9, 21762.1, 19162.2
   },
  {
  9629.27, 12234.7, 12236, 13105.1, 13106.1, 13975, 16579.6, 17448.3, 19184.9, 21789.3, 19186.1
   },
  {
  9641.25, 12249.9, 12251.2, 13121.4, 13122.4, 13992.4, 16600.3, 17470.1, 19208.8, 21816.5, 19210
   },
  {
  9653.22, 12265.1, 12266.5, 13137.7, 13138.8, 14009.8, 16621, 17491.8, 19232.7, 21843.7, 19234
   },
  {
  9665.19, 12280.4, 12281.7, 13154.1, 13155.1, 14027.2, 16641.6, 17513.6, 19256.7, 21870.9, 19257.9
   },
  {
  9677.16, 12295.6, 12296.9, 13170.4, 13171.4, 14044.6, 16662.3, 17535.3, 19280.6, 21898.1, 19281.8
   },
  {
  9689.13, 12310.8, 12312.2, 13186.7, 13187.7, 14062, 16683, 17557.1, 19304.5, 21925.3, 19305.8
   },
  {
  9701.11, 12326.1, 12327.4, 13203, 13204.1, 14079.4, 16703.6, 17578.9, 19328.5, 21952.5, 19329.7
   },
  {
  9713.08, 12341.3, 12342.6, 13219.3, 13220.4, 14096.8, 16724.3, 17600.6, 19352.4, 21979.7, 19353.6
   },
  {
  9725.05, 12356.5, 12357.9, 13235.7, 13236.7, 14114.2, 16745, 17622.4, 19376.3, 22006.9, 19377.6
   },
  {
  9737.02, 12371.8, 12373.1, 13252, 13253, 14131.7, 16765.6, 17644.1, 19400.3, 22034.1, 19401.5
   },
  {
  9748.99, 12387, 12388.3, 13268.3, 13269.3, 14149.1, 16786.3, 17665.9, 19424.2, 22061.3, 19425.4
   },
  {
  9760.96, 12402.2, 12403.6, 13284.6, 13285.7, 14166.5, 16807, 17687.6, 19448.1, 22088.5, 19449.4
   },
  {
  9772.94, 12417.5, 12418.8, 13301, 13302, 14183.9, 16827.7, 17709.4, 19472.1, 22115.7, 19473.3
   },
  {
  9784.91, 12432.7, 12434, 13317.3, 13318.3, 14201.3, 16848.3, 17731.2, 19496, 22142.9, 19497.2
   },
  {
  9796.88, 12447.9, 12449.3, 13333.6, 13334.6, 14218.7, 16869, 17752.9, 19519.9, 22170.1, 19521.2
   },
  {
  9808.85, 12463.2, 12464.5, 13349.9, 13350.9, 14236.1, 16889.7, 17774.7, 19543.9, 22197.2, 19545.1
   },
  {
  9820.82, 12478.4, 12479.7, 13366.2, 13367.3, 14253.5, 16910.3, 17796.4, 19567.8, 22224.4, 19569
   },
  {
  9832.79, 12493.6, 12495, 13382.6, 13383.6, 14270.9, 16931, 17818.2, 19591.7, 22251.6, 19593
   },
  {
  9844.77, 12508.9, 12510.2, 13398.9, 13399.9, 14288.3, 16951.7, 17839.9, 19615.7, 22278.8, 19616.9
   },
  {
  9856.74, 12524.1, 12525.4, 13415.2, 13416.2, 14305.7, 16972.4, 17861.7, 19639.6, 22306, 19640.8
   },
  {
  9868.71, 12539.3, 12540.7, 13431.5, 13432.5, 14323.1, 16993, 17883.5, 19663.5, 22333.2, 19664.8
   },
  {
  9880.68, 12554.6, 12555.9, 13447.8, 13448.9, 14340.6, 17013.7, 17905.2, 19687.5, 22360.4, 19688.7
   },
  {
  9892.65, 12569.8, 12571.1, 13464.2, 13465.2, 14358, 17034.4, 17927, 19711.4, 22387.6, 19712.6
   },
  {
  9904.62, 12585, 12586.4, 13480.5, 13481.5, 14375.4, 17055, 17948.7, 19735.3, 22414.8, 19736.6
   },
  {
  9916.6, 12600.3, 12601.6, 13496.8, 13497.8, 14392.8, 17075.7, 17970.5, 19759.3, 22442, 19760.5
   },
  {
  9928.57, 12615.5, 12616.9, 13513.1, 13514.2, 14410.2, 17096.4, 17992.3, 19783.2, 22469.2, 19784.4
   },
  {
  9940.54, 12630.7, 12632.1, 13529.5, 13530.5, 14427.6, 17117, 18014, 19807.1, 22496.4, 19808.4
   },
  {
  9952.51, 12646, 12647.3, 13545.8, 13546.8, 14445, 17137.7, 18035.8, 19831.1, 22523.6, 19832.3
   },
  {
  9964.48, 12661.2, 12662.6, 13562.1, 13563.1, 14462.4, 17158.4, 18057.5, 19855, 22550.8, 19856.2
   },
  {
  9976.45, 12676.4, 12677.8, 13578.4, 13579.4, 14479.8, 17179.1, 18079.3, 19878.9, 22578, 19880.2
   },
  {
  9988.43, 12691.7, 12693, 13594.7, 13595.8, 14497.2, 17199.7, 18101, 19902.9, 22605.2, 19904.1
   },
  {
  10000.4, 12706.9, 12708.3, 13611.1, 13612.1, 14514.6, 17220.4, 18122.8, 19926.8, 22632.4, 19928
   },
  {
  10012.4, 12722.1, 12723.5, 13627.4, 13628.4, 14532, 17241.1, 18144.6, 19950.7, 22659.6, 19952
   },
  {
  10024.3, 12737.4, 12738.7, 13643.7, 13644.7, 14549.5, 17261.7, 18166.3, 19974.6, 22686.8, 19975.9
   },
  {
  10036.3, 12752.6, 12754, 13660, 13661, 14566.9, 17282.4, 18188.1, 19998.6, 22713.9, 19999.8
   },
  {
  10048.3, 12767.8, 12769.2, 13676.3, 13677.4, 14584.3, 17303.1, 18209.8, 20022.5, 22741.1, 20023.8
   },
  {
  10060.3, 12783.1, 12784.4, 13692.7, 13693.7, 14601.7, 17323.8, 18231.6, 20046.4, 22768.3, 20047.7
   },
  {
  10072.2, 12798.3, 12799.7, 13709, 13710, 14619.1, 17344.4, 18253.3, 20070.4, 22795.5, 20071.6
   },
  {
  10084.2, 12813.5, 12814.9, 13725.3, 13726.3, 14636.5, 17365.1, 18275.1, 20094.3, 22822.7, 20095.6
   },
  {
  10096.2, 12828.8, 12830.1, 13741.6, 13742.6, 14653.9, 17385.8, 18296.9, 20118.2, 22849.9, 20119.5
   },
  {
  10108.1, 12844, 12845.4, 13757.9, 13759, 14671.3, 17406.4, 18318.6, 20142.2, 22877.1, 20143.4
   },
  {
  10120.1, 12859.2, 12860.6, 13774.3, 13775.3, 14688.7, 17427.1, 18340.4, 20166.1, 22904.3, 20167.4
   },
  {
  10132.1, 12874.5, 12875.8, 13790.6, 13791.6, 14706.1, 17447.8, 18362.1, 20190, 22931.5, 20191.3
   },
  {
  10144.1, 12889.7, 12891.1, 13806.9, 13807.9, 14723.5, 17468.4, 18383.9, 20214, 22958.7, 20215.2
   },
  {
  10156, 12904.9, 12906.3, 13823.2, 13824.3, 14740.9, 17489.1, 18405.7, 20237.9, 22985.9, 20239.1
   },
  {
  10168, 12920.2, 12921.5, 13839.5, 13840.6, 14758.4, 17509.8, 18427.4, 20261.8, 23013.1, 20263.1
   },
  {
  10180, 12935.4, 12936.8, 13855.9, 13856.9, 14775.8, 17530.5, 18449.2, 20285.8, 23040.3, 20287
   },
  {
  10191.9, 12950.6, 12952, 13872.2, 13873.2, 14793.2, 17551.1, 18470.9, 20309.7, 23067.5, 20310.9
   },
  {
  10203.9, 12965.9, 12967.2, 13888.5, 13889.5, 14810.6, 17571.8, 18492.7, 20333.6, 23094.7, 20334.9
   },
  {
  10215.9, 12981.1, 12982.5, 13904.8, 13905.9, 14828, 17592.5, 18514.4, 20357.6, 23121.9, 20358.8
   },
  {
  10227.9, 12996.4, 12997.7, 13921.2, 13922.2, 14845.4, 17613.1, 18536.2, 20381.5, 23149.1, 20382.7
   },
  {
  10239.8, 13011.6, 13012.9, 13937.5, 13938.5, 14862.8, 17633.8, 18558, 20405.4, 23176.3, 20406.7
   },
  {
  10251.8, 13026.8, 13028.2, 13953.8, 13954.8, 14880.2, 17654.5, 18579.7, 20429.4, 23203.4, 20430.6
   },
  {
  10263.8, 13042.1, 13043.4, 13970.1, 13971.1, 14897.6, 17675.1, 18601.5, 20453.3, 23230.6, 20454.5
   },
  {
  10275.7, 13057.3, 13058.6, 13986.4, 13987.5, 14915, 17695.8, 18623.2, 20477.2, 23257.8, 20478.5
   },
  {
  10287.7, 13072.5, 13073.9, 14002.8, 14003.8, 14932.4, 17716.5, 18645, 20501.2, 23285, 20502.4
   },
  {
  10299.7, 13087.8, 13089.1, 14019.1, 14020.1, 14949.8, 17737.2, 18666.7, 20525.1, 23312.2, 20526.3
   },
  {
  10311.7, 13103, 13104.3, 14035.4, 14036.4, 14967.3, 17757.8, 18688.5, 20549, 23339.4, 20550.3
   },
  {
  10323.6, 13118.2, 13119.6, 14051.7, 14052.7, 14984.7, 17778.5, 18710.3, 20573, 23366.6, 20574.2
   },
  {
  10335.6, 13133.4, 13134.8, 14068, 14069.1, 15002.1, 17799.2, 18732, 20596.9, 23393.8, 20598.1
   },
  {
  10347.6, 13148.7, 13150, 14084.4, 14085.4, 15019.5, 17819.8, 18753.8, 20620.8, 23421, 20622.1
   },
  {
  10359.5, 13163.9, 13165.3, 14100.7, 14101.7, 15036.9, 17840.5, 18775.5, 20644.8, 23448.2, 20646
   },
  {
  10371.5, 13179.1, 13180.5, 14117, 14118, 15054.3, 17861.2, 18797.3, 20668.7, 23475.4, 20669.9
   },
  {
  10383.5, 13194.4, 13195.7, 14133.3, 14134.3, 15071.7, 17881.8, 18819, 20692.6, 23502.6, 20693.9
   },
  {
  10395.5, 13209.6, 13211, 14149.6, 14150.7, 15089.1, 17902.5, 18840.8, 20716.6, 23529.8, 20717.8
   },
  {
  10407.4, 13224.8, 13226.2, 14166, 14167, 15106.5, 17923.2, 18862.6, 20740.5, 23557, 20741.7
   },
  {
  10419.4, 13240.1, 13241.4, 14182.3, 14183.3, 15123.9, 17943.9, 18884.3, 20764.4, 23584.2, 20765.7
   },
  {
  10431.4, 13255.3, 13256.7, 14198.6, 14199.6, 15141.3, 17964.5, 18906.1, 20788.3, 23611.4, 20789.6
   },
  {
  10443.3, 13270.5, 13271.9, 14214.9, 14215.9, 15158.7, 17985.2, 18927.8, 20812.3, 23638.6, 20813.5
   },
  {
  10455.3, 13285.8, 13287.1, 14231.2, 14232.3, 15176.1, 18005.9, 18949.6, 20836.2, 23665.7, 20837.5
   },
  {
  10467.3, 13301, 13302.4, 14247.6, 14248.6, 15193.6, 18026.5, 18971.3, 20860.1, 23692.9, 20861.4
   },
  {
  10479.2, 13316.2, 13317.6, 14263.9, 14264.9, 15211, 18047.2, 18993.1, 20884.1, 23720.1, 20885.3
   },
  {
  10491.2, 13331.5, 13332.8, 14280.2, 14281.2, 15228.4, 18067.9, 19014.9, 20908, 23747.3, 20909.3
   },
  {
  10503.2, 13346.7, 13348.1, 14296.5, 14297.6, 15245.8, 18088.5, 19036.6, 20931.9, 23774.5, 20933.2
   },
  {
  10515.2, 13361.9, 13363.3, 14312.8, 14313.9, 15263.2, 18109.2, 19058.4, 20955.9, 23801.7, 20957.1
   },
  {
  10527.1, 13377.2, 13378.5, 14329.2, 14330.2, 15280.6, 18129.9, 19080.1, 20979.8, 23828.9, 20981
   },
  {
  10539.1, 13392.4, 13393.8, 14345.5, 14346.5, 15298, 18150.6, 19101.9, 21003.7, 23856.1, 21005
   },
  {
  10551.1, 13407.6, 13409, 14361.8, 14362.8, 15315.4, 18171.2, 19123.6, 21027.7, 23883.3, 21028.9
   },
  {
  10563, 13422.9, 13424.2, 14378.1, 14379.2, 15332.8, 18191.9, 19145.4, 21051.6, 23910.5, 21052.8
   },
  {
  10575, 13438.1, 13439.5, 14394.4, 14395.5, 15350.2, 18212.6, 19167.2, 21075.5, 23937.7, 21076.8
   },
  {
  10587, 13453.3, 13454.7, 14410.8, 14411.8, 15367.6, 18233.2, 19188.9, 21099.5, 23964.9, 21100.7
   },
  {
  10599, 13468.6, 13469.9, 14427.1, 14428.1, 15385, 18253.9, 19210.7, 21123.4, 23992.1, 21124.6
   },
  {
  10610.9, 13483.8, 13485.2, 14443.4, 14444.4, 15402.4, 18274.6, 19232.4, 21147.3, 24019.3, 21148.6
   },
  {
  10622.9, 13499, 13500.4, 14459.7, 14460.8, 15419.9, 18295.2, 19254.2, 21171.3, 24046.5, 21172.5
   },
  {
  10634.9, 13514.3, 13515.6, 14476.1, 14477.1, 15437.3, 18315.9, 19275.9, 21195.2, 24073.7, 21196.4
   },
  {
  10646.8, 13529.5, 13530.8, 14492.4, 14493.4, 15454.7, 18336.6, 19297.7, 21219.1, 24100.9, 21220.4
   },
  {
  10658.8, 13544.7, 13546.1, 14508.7, 14509.7, 15472.1, 18357.3, 19319.5, 21243.1, 24128, 21244.3
   },
  {
  10670.8, 13560, 13561.3, 14525, 14526, 15489.5, 18377.9, 19341.2, 21267, 24155.2, 21268.2
   },
  {
  10682.8, 13575.2, 13576.5, 14541.3, 14542.4, 15506.9, 18398.6, 19363, 21290.9, 24182.4, 21292.2
   },
  {
  10694.7, 13590.4, 13591.8, 14557.7, 14558.7, 15524.3, 18419.3, 19384.7, 21314.9, 24209.6, 21316.1
   },
  {
  10706.7, 13605.7, 13607, 14574, 14575, 15541.7, 18439.9, 19406.5, 21338.8, 24236.8, 21340
   },
  {
  10718.7, 13620.9, 13622.2, 14590.3, 14591.3, 15559.1, 18460.6, 19428.2, 21362.7, 24264, 21364
   },
  {
  10730.6, 13636.1, 13637.5, 14606.6, 14607.6, 15576.5, 18481.3, 19450, 21386.6, 24291.2, 21387.9
   },
  {
  10742.6, 13651.4, 13652.7, 14622.9, 14624, 15593.9, 18501.9, 19471.8, 21410.6, 24318.4, 21411.8
   },
  {
  10754.6, 13666.6, 13667.9, 14639.3, 14640.3, 15611.3, 18522.6, 19493.5, 21434.5, 24345.6, 21435.8
   },
  {
  10766.5, 13681.8, 13683.2, 14655.6, 14656.6, 15628.7, 18543.3, 19515.3, 21458.4, 24372.8, 21459.7
   },
  {
  10778.5, 13697.1, 13698.4, 14671.9, 14672.9, 15646.2, 18564, 19537, 21482.4, 24400, 21483.6
   },
  {
  10790.5, 13712.3, 13713.6, 14688.2, 14689.2, 15663.6, 18584.6, 19558.8, 21506.3, 24427.2, 21507.5
   },
  {
  10802.5, 13727.5, 13728.9, 14704.5, 14705.6, 15681, 18605.3, 19580.5, 21530.2, 24454.4, 21531.5
   },
  {
  10814.4, 13742.8, 13744.1, 14720.9, 14721.9, 15698.4, 18626, 19602.3, 21554.2, 24481.6, 21555.4
   },
  {
  10826.4, 13758, 13759.3, 14737.2, 14738.2, 15715.8, 18646.6, 19624.1, 21578.1, 24508.8, 21579.3
   },
  {
  10838.4, 13773.2, 13774.6, 14753.5, 14754.5, 15733.2, 18667.3, 19645.8, 21602, 24536, 21603.3
   },
  {
  10850.3, 13788.5, 13789.8, 14769.8, 14770.8, 15750.6, 18688, 19667.6, 21626, 24563.2, 21627.2
   },
  {
  10862.3, 13803.7, 13805, 14786.1, 14787.2, 15768, 18708.6, 19689.3, 21649.9, 24590.3, 21651.1
   },
  {
  10874.3, 13818.9, 13820.3, 14802.5, 14803.5, 15785.4, 18729.3, 19711.1, 21673.8, 24617.5, 21675.1
   },
  {
  10886.3, 13834.2, 13835.5, 14818.8, 14819.8, 15802.8, 18750, 19732.8, 21697.8, 24644.7, 21699
   },
  {
  10898.2, 13849.4, 13850.7, 14835.1, 14836.1, 15820.2, 18770.7, 19754.6, 21721.7, 24671.9, 21722.9
   },
  {
  10910.2, 13864.6, 13866, 14851.4, 14852.4, 15837.6, 18791.3, 19776.4, 21745.6, 24699.1, 21746.9
   },
  {
  10922.2, 13879.9, 13881.2, 14867.7, 14868.8, 15855, 18812, 19798.1, 21769.6, 24726.3, 21770.8
   },
  {
  10934.1, 13895.1, 13896.4, 14884.1, 14885.1, 15872.5, 18832.7, 19819.9, 21793.5, 24753.5, 21794.7
   },
  {
  10946.1, 13910.3, 13911.7, 14900.4, 14901.4, 15889.9, 18853.3, 19841.6, 21817.4, 24780.7, 21818.7
   },
  {
  10958.1, 13925.6, 13926.9, 14916.7, 14917.7, 15907.3, 18874, 19863.4, 21841.4, 24807.9, 21842.6
   },
  {
  10970.1, 13940.8, 13942.1, 14933, 14934, 15924.7, 18894.7, 19885.1, 21865.3, 24835.1, 21866.5
   },
  {
  10982, 13956, 13957.4, 14949.3, 14950.4, 15942.1, 18915.3, 19906.9, 21889.2, 24862.3, 21890.5
   },
  {
  10994, 13971.3, 13972.6, 14965.7, 14966.7, 15959.5, 18936, 19928.7, 21913.1, 24889.5, 21914.4
   },
  {
  11006, 13986.5, 13987.8, 14982, 14983, 15976.9, 18956.7, 19950.4, 21937.1, 24916.7, 21938.3
   },
  {
  11017.9, 14001.7, 14003.1, 14998.3, 14999.3, 15994.3, 18977.3, 19972.2, 21961, 24943.9, 21962.3
   },
  {
  11029.9, 14017, 14018.3, 15014.6, 15015.6, 16011.7, 18998, 19993.9, 21984.9, 24971.1, 21986.2
   },
  {
  11041.9, 14032.2, 14033.5, 15030.9, 15032, 16029.1, 19018.7, 20015.7, 22008.9, 24998.3, 22010.1
   },
  {
  11053.8, 14047.4, 14048.8, 15047.3, 15048.3, 16046.5, 19039.4, 20037.4, 22032.8, 25025.4, 22034
   },
  {
  11065.8, 14062.7, 14064, 15063.6, 15064.6, 16063.9, 19060, 20059.2, 22056.7, 25052.6, 22058
   },
  {
  11077.8, 14077.9, 14079.2, 15079.9, 15080.9, 16081.3, 19080.7, 20081, 22080.7, 25079.8, 22081.9
   },
  {
  11089.8, 14093.1, 14094.5, 15096.2, 15097.2, 16098.8, 19101.4, 20102.7, 22104.6, 25107, 22105.8
   },
  {
  11101.7, 14108.4, 14109.7, 15112.5, 15113.6, 16116.2, 19122, 20124.5, 22128.5, 25134.2, 22129.8
   },
  {
  11113.7, 14123.6, 14124.9, 15128.9, 15129.9, 16133.6, 19142.7, 20146.2, 22152.5, 25161.4, 22153.7
   },
  {
  11125.7, 14138.8, 14140.2, 15145.2, 15146.2, 16151, 19163.4, 20168, 22176.4, 25188.6, 22177.6
   },
  {
  11137.6, 14154.1, 14155.4, 15161.5, 15162.5, 16168.4, 19184, 20189.7, 22200.3, 25215.8, 22201.6
   },
  {
  11149.6, 14169.3, 14170.6, 15177.8, 15178.8, 16185.8, 19204.7, 20211.5, 22224.3, 25243, 22225.5
   },
  {
  11161.6, 14184.5, 14185.9, 15194.1, 15195.2, 16203.2, 19225.4, 20233.3, 22248.2, 25270.2, 22249.4
   },
  {
  11173.5, 14199.8, 14201.1, 15210.5, 15211.5, 16220.6, 19246, 20255, 22272.1, 25297.4, 22273.4
   },
  {
  11185.5, 14215, 14216.3, 15226.8, 15227.8, 16238, 19266.7, 20276.8, 22296, 25324.6, 22297.3
   },
  {
  11197.5, 14230.2, 14231.6, 15243.1, 15244.1, 16255.4, 19287.4, 20298.5, 22320, 25351.8, 22321.2
   },
  {
  11209.5, 14245.4, 14246.8, 15259.4, 15260.4, 16272.8, 19308.1, 20320.3, 22343.9, 25379, 22345.2
   },
  {
  11221.4, 14260.7, 14262, 15275.7, 15276.8, 16290.2, 19328.7, 20342, 22367.8, 25406.2, 22369.1
   },
  {
  11233.4, 14275.9, 14277.3, 15292.1, 15293.1, 16307.6, 19349.4, 20363.8, 22391.8, 25433.3, 22393
   },
  {
  11245.4, 14291.1, 14292.5, 15308.4, 15309.4, 16325, 19370.1, 20385.5, 22415.7, 25460.5, 22416.9
   },
  {
  11257.3, 14306.4, 14307.7, 15324.7, 15325.7, 16342.5, 19390.7, 20407.3, 22439.6, 25487.7, 22440.9
   },
  {
  11269.3, 14321.6, 14322.9, 15341, 15342, 16359.9, 19411.4, 20429.1, 22463.6, 25514.9, 22464.8
   },
  {
  11281.3, 14336.8, 14338.2, 15357.3, 15358.4, 16377.3, 19432.1, 20450.8, 22487.5, 25542.1, 22488.7
   },
  {
  11293.3, 14352.1, 14353.4, 15373.7, 15374.7, 16394.7, 19452.7, 20472.6, 22511.4, 25569.3, 22512.7
   },
  {
  11305.2, 14367.3, 14368.6, 15390, 15391, 16412.1, 19473.4, 20494.3, 22535.4, 25596.5, 22536.6
   },
  {
  11317.2, 14382.5, 14383.9, 15406.3, 15407.3, 16429.5, 19494.1, 20516.1, 22559.3, 25623.7, 22560.5
   },
  {
  11329.2, 14397.8, 14399.1, 15422.6, 15423.6, 16446.9, 19514.7, 20537.8, 22583.2, 25650.9, 22584.5
   },
  {
  11341.1, 14413, 14414.3, 15438.9, 15440, 16464.3, 19535.4, 20559.6, 22607.2, 25678.1, 22608.4
   },
  {
  11353.1, 14428.2, 14429.6, 15455.3, 15456.3, 16481.7, 19556.1, 20581.4, 22631.1, 25705.3, 22632.3
   },
  {
  11365.1, 14443.5, 14444.8, 15471.6, 15472.6, 16499.1, 19576.8, 20603.1, 22655, 25732.5, 22656.3
   },
  {
  11377, 14458.7, 14460, 15487.9, 15488.9, 16516.5, 19597.4, 20624.9, 22679, 25759.7, 22680.2
   },
  {
  11389, 14473.9, 14475.3, 15504.2, 15505.2, 16533.9, 19618.1, 20646.6, 22702.9, 25786.9, 22704.1
   },
  {
  11401, 14489.2, 14490.5, 15520.5, 15521.6, 16551.3, 19638.8, 20668.4, 22726.8, 25814.1, 22728.1
   },
  {
  11413, 14504.4, 14505.7, 15536.9, 15537.9, 16568.7, 19659.4, 20690.1, 22750.7, 25841.2, 22752
   },
  {
  11424.9, 14519.6, 14521, 15553.2, 15554.2, 16586.2, 19680.1, 20711.9, 22774.7, 25868.4, 22775.9
   },
  {
  11436.9, 14534.9, 14536.2, 15569.5, 15570.5, 16603.6, 19700.8, 20733.7, 22798.6, 25895.6, 22799.9
   },
  {
  11448.9, 14550.1, 14551.4, 15585.8, 15586.8, 16621, 19721.4, 20755.4, 22822.5, 25922.8, 22823.8
   },
  {
  11460.8, 14565.3, 14566.7, 15602.1, 15603.2, 16638.4, 19742.1, 20777.2, 22846.5, 25950, 22847.7
   },
  {
  11472.8, 14580.6, 14581.9, 15618.5, 15619.5, 16655.8, 19762.8, 20798.9, 22870.4, 25977.2, 22871.6
   },
  {
  11484.8, 14595.8, 14597.1, 15634.8, 15635.8, 16673.2, 19783.4, 20820.7, 22894.3, 26004.4, 22895.6
   },
  {
  11496.7, 14611, 14612.4, 15651.1, 15652.1, 16690.6, 19804.1, 20842.4, 22918.3, 26031.6, 22919.5
   },
  {
  11508.7, 14626.3, 14627.6, 15667.4, 15668.4, 16708, 19824.8, 20864.2, 22942.2, 26058.8, 22943.4
   },
  {
  11520.7, 14641.5, 14642.8, 15683.7, 15684.8, 16725.4, 19845.5, 20886, 22966.1, 26086, 22967.4
   },
  {
  11532.7, 14656.7, 14658.1, 15700, 15701.1, 16742.8, 19866.1, 20907.7, 22990.1, 26113.2, 22991.3
   },
  {
  11544.6, 14672, 14673.3, 15716.4, 15717.4, 16760.2, 19886.8, 20929.5, 23014, 26140.4, 23015.2
   },
  {
  11556.6, 14687.2, 14688.5, 15732.7, 15733.7, 16777.6, 19907.5, 20951.2, 23037.9, 26167.6, 23039.2
   },
  {
  11568.6, 14702.4, 14703.8, 15749, 15750, 16795, 19928.1, 20973, 23061.8, 26194.8, 23063.1
   },
  {
  11580.5, 14717.7, 14719, 15765.3, 15766.3, 16812.4, 19948.8, 20994.7, 23085.8, 26222, 23087
   },
  {
  11592.5, 14732.9, 14734.2, 15781.6, 15782.7, 16829.8, 19969.5, 21016.5, 23109.7, 26249.1, 23111
   },
  {
  11604.5, 14748.1, 14749.5, 15798, 15799, 16847.3, 19990.1, 21038.2, 23133.6, 26276.3, 23134.9
   },
  {
  11616.4, 14763.3, 14764.7, 15814.3, 15815.3, 16864.7, 20010.8, 21060, 23157.6, 26303.5, 23158.8
   },
  {
  11628.4, 14778.6, 14779.9, 15830.6, 15831.6, 16882.1, 20031.5, 21081.8, 23181.5, 26330.7, 23182.7
   },
  {
  11640.4, 14793.8, 14795.2, 15846.9, 15847.9, 16899.5, 20052.1, 21103.5, 23205.4, 26357.9, 23206.7
   },
  {
  11652.4, 14809, 14810.4, 15863.2, 15864.3, 16916.9, 20072.8, 21125.3, 23229.4, 26385.1, 23230.6
   },
  {
  11664.3, 14824.3, 14825.6, 15879.6, 15880.6, 16934.3, 20093.5, 21147, 23253.3, 26412.3, 23254.5
   },
  {
  11676.3, 14839.5, 14840.8, 15895.9, 15896.9, 16951.7, 20114.2, 21168.8, 23277.2, 26439.5, 23278.5
   },
  {
  11688.3, 14854.7, 14856.1, 15912.2, 15913.2, 16969.1, 20134.8, 21190.5, 23301.2, 26466.7, 23302.4
   },
  {
  11700.2, 14870, 14871.3, 15928.5, 15929.5, 16986.5, 20155.5, 21212.3, 23325.1, 26493.9, 23326.3
   },
  {
  11712.2, 14885.2, 14886.5, 15944.8, 15945.9, 17003.9, 20176.2, 21234.1, 23349, 26521.1, 23350.3
   },
  {
  11724.2, 14900.4, 14901.8, 15961.2, 15962.2, 17021.3, 20196.8, 21255.8, 23373, 26548.3, 23374.2
   },
  {
  11736.1, 14915.7, 14917, 15977.5, 15978.5, 17038.7, 20217.5, 21277.6, 23396.9, 26575.5, 23398.1
   },
  {
  11748.1, 14930.9, 14932.2, 15993.8, 15994.8, 17056.1, 20238.2, 21299.3, 23420.8, 26602.7, 23422.1
   },
  {
  11760.1, 14946.1, 14947.5, 16010.1, 16011.1, 17073.5, 20258.8, 21321.1, 23444.7, 26629.9, 23446
   },
  {
  11772.1, 14961.4, 14962.7, 16026.4, 16027.5, 17091, 20279.5, 21342.8, 23468.7, 26657, 23469.9
   },
  {
  11784, 14976.6, 14977.9, 16042.8, 16043.8, 17108.4, 20300.2, 21364.6, 23492.6, 26684.2, 23493.9
   },
  {
  11796, 14991.8, 14993.2, 16059.1, 16060.1, 17125.8, 20320.8, 21386.3, 23516.5, 26711.4, 23517.8
   },
  {
  11808, 15007.1, 15008.4, 16075.4, 16076.4, 17143.2, 20341.5, 21408.1, 23540.5, 26738.6, 23541.7
   },
  {
  11819.9, 15022.3, 15023.6, 16091.7, 16092.7, 17160.6, 20362.2, 21429.9, 23564.4, 26765.8, 23565.6
   },
  {
  11831.9, 15037.5, 15038.9, 16108, 16109.1, 17178, 20382.8, 21451.6, 23588.3, 26793, 23589.6
   },
  {
  11843.9, 15052.8, 15054.1, 16124.4, 16125.4, 17195.4, 20403.5, 21473.4, 23612.3, 26820.2, 23613.5
   },
  {
  11855.8, 15068, 15069.3, 16140.7, 16141.7, 17212.8, 20424.2, 21495.1, 23636.2, 26847.4, 23637.4
   },
  {
  11867.8, 15083.2, 15084.6, 16157, 16158, 17230.2, 20444.9, 21516.9, 23660.1, 26874.6, 23661.4
   },
  {
  11879.8, 15098.5, 15099.8, 16173.3, 16174.3, 17247.6, 20465.5, 21538.6, 23684.1, 26901.8, 23685.3
   },
  {
  11891.8, 15113.7, 15115, 16189.6, 16190.7, 17265, 20486.2, 21560.4, 23708, 26929, 23709.2
   },
  {
  11903.7, 15128.9, 15130.3, 16206, 16207, 17282.4, 20506.9, 21582.2, 23731.9, 26956.2, 23733.2
   },
  {
  11915.7, 15144.1, 15145.5, 16222.3, 16223.3, 17299.8, 20527.5, 21603.9, 23755.8, 26983.4, 23757.1
   },
  {
  11927.7, 15159.4, 15160.7, 16238.6, 16239.6, 17317.2, 20548.2, 21625.7, 23779.8, 27010.6, 23781
   },
  {
  11939.6, 15174.6, 15176, 16254.9, 16255.9, 17334.6, 20568.9, 21647.4, 23803.7, 27037.7, 23805
   },
  {
  11951.6, 15189.8, 15191.2, 16271.2, 16272.3, 17352.1, 20589.5, 21669.2, 23827.6, 27064.9, 23828.9
   },
  {
  11963.6, 15205.1, 15206.4, 16287.5, 16288.6, 17369.5, 20610.2, 21690.9, 23851.6, 27092.1, 23852.8
   },
  {
  11975.5, 15220.3, 15221.6, 16303.9, 16304.9, 17386.9, 20630.9, 21712.7, 23875.5, 27119.3, 23876.7
   },
  {
  11987.5, 15235.5, 15236.9, 16320.2, 16321.2, 17404.3, 20651.5, 21734.4, 23899.4, 27146.5, 23900.7
   },
  {
  11999.5, 15250.8, 15252.1, 16336.5, 16337.5, 17421.7, 20672.2, 21756.2, 23923.4, 27173.7, 23924.6
   },
  {
  12011.5, 15266, 15267.3, 16352.8, 16353.8, 17439.1, 20692.9, 21778, 23947.3, 27200.9, 23948.5
   },
  {
  12023.4, 15281.2, 15282.6, 16369.1, 16370.2, 17456.5, 20713.5, 21799.7, 23971.2, 27228.1, 23972.5
   },
  {
  12035.4, 15296.5, 15297.8, 16385.5, 16386.5, 17473.9, 20734.2, 21821.5, 23995.2, 27255.3, 23996.4
   },

   };


  Ptr<MobilityModel> srcMobility = src->GetP1906NetDevice ()->GetNode ()->GetObject<MobilityModel> ();
  Ptr<MobilityModel> dstMobility = dst->GetP1906NetDevice ()->GetNode ()->GetObject<MobilityModel> ();
  double distance = dstMobility->GetDistanceFrom (srcMobility);

  NS_LOG_FUNCTION (this << "[distance]" << distance);

  int index_d;
  if (distance <= distances [0])
	index_d = 0;
  else if (distance >= distances [999])
	  index_d = 999;
  else
    {
	  index_d = 0;
	  for (int i = 1; i < 1000; i++)
	    {
		  if (distances [i] > distance)
		    {
			  index_d = i-1;
			  i=1000;
		    }
	    }
    }

  Ptr<P1906EMMessageCarrier> m = message->GetObject <P1906EMMessageCarrier> ();
  Ptr<SpectrumValue> sv = m->GetSpectrumValue ();

  NS_LOG_FUNCTION (this << "[txPsd]" << *sv);
  for (int i=0; i<11; i++)
    {
      (*sv)[i] = pow(10., ((10*log10 ((*sv)[i])) - pathloss[index_d][i])/10.);
    }
  NS_LOG_FUNCTION (this << "[rxPsd]" << *sv);

  return message;
}



void
P1906EMMotion::SetWaveSpeed (double s)
{
  NS_LOG_FUNCTION (this << s);
  m_waveSpeed = s;
}

double
P1906EMMotion::GetWaveSpeed (void)
{
  NS_LOG_FUNCTION (this);
  return m_waveSpeed;
}


} // namespace ns3
